#
#	Command - manage a command window
#
# Some of this file is patterned after the excellent program "tkCon", written
# by Jeffery Hobbs.
#

array set code::Command {
    initialized 0
    subhistory 1
    event 1
    cmdbuf {}
    pwd {}
}

#
# bindings used in the Command window
#
set code::COMMANDbindings {
    <Return> {
        code::CommandEval %W; break
    }
    <KP_Enter> {
        code::CommandEval %W; break
    }
    <Up> {
       if [%W compare {insert linestart} != {limit linestart}] {
            tkTextSetCursor %W [tkTextUpDownLine %W -1]
        } else {
            if {$code::Command(event) \
             == [history nextid]} {
                set code::Command(cmdbuf) \
	            [code::CommandCmdGet %W]
            }
            if {[catch {history event \
	        [incr code::Command(event) -1]} \
	          code::Command(tmp)]} {
	        incr code::Command(event)
            } else {
	        %W delete limit end
	        %W insert limit $code::Command(tmp)
	        %W see end
            }
        }
    	break
    }
    <Down> {
        if {[%W compare {insert linestart} != {end-1c linestart}]} {
            tkTextSetCursor %W [tkTextUpDownLine %W 1]
        } else {
            if {$code::Command(event) \
              < [history nextid]} {
	        %W delete limit end
	        if {[incr code::Command(event)] \
	          == [history nextid]} {
	            %W insert limit \
		        $code::Command(cmdbuf)
	        } else {
	            %W insert limit \
		        [history event $code::Command(event)]
	        }
	        %W see end
            }
        }
	break
    }
    <Control-a> {
        if [%W compare {limit linestart} == {insert linestart}] {
            tkTextSetCursor %W limit
        } else {
            tkTextSetCursor %W {insert linestart}
        }
        break
    }
    <Control-l> {
        # Clear console buffer, without losing current command line
        set Command(tmp) [code::CommandCmdGet %W]
        code::CommandClear %W
        code::CommandPrompt %W {} $Command(tmp)
	break
    }
    <Control-u> {
        ## Clear command line (Unix shell staple)
        %W delete limit end
        break
    }
}

#
# COMMANDstartup - return the command to open the command prompt
#
proc code::COMMANDstartup {doc} {
    return "COMMAND"
}

#
# COMMAND - open the command prompt window
#
proc code::COMMAND {args} {
    variable FileIcons

    set type "Command"
    set name "Command Prompt"
    set w [CODEfind $type $name]
    if {$w != {}} {
	# have a command prompt, raise and return
	if {   [$w cget -state] == "minimized" 
	    || [$w cget -state] == "withdrawn"} {
	    $w configure -state normal
	}
	$w raise
	return $w
    }
    # a new command prompt
    set w [eval TEXT $args]
    $w menu entryconfigure Close -command "$w configure -state withdrawn"
    set t $w.contents
    CommandNew $w $t
    $w configure -startupproc "code::COMMANDstartup"
    set image $FileIcons(tclfile.icon)
    $w configure -image $image -icontext $type -title $name
    return $w
}

proc code::CommandNew {w t} {
    variable Command
    variable COMMANDbindings
    global tcl_interactive

    set type "Command"
    set name "Command Prompt"
    set Command(text) $t
    $w configure -type $type
    $t configure -contents $name

    $t delete 1.0 end
    $t configure -highlightthickness 0
    $t tag configure limit
    foreach "bind command" $COMMANDbindings {
	regsub -all %W $command $t command
	bind $t $bind $command
    }
          
    PreferenceWhenChanged Command $w "CommandPreferences $t"
    CommandPreferences $t
        
    set tcl_interactive 1
    # get the initial working directory
    set Command(pwd) [pwd]
}

#
# CommandAdd - add a command (and result) to the command window
#
proc code::CommandAdd {line {stdout {}} {stderr {}}} {
    variable Command
    variable CODEdocument
    variable CODEstarted

    if {![info exists CODEstarted]} {
	# not initialized enough
	return
    }
    # add a command to the command history
    if {[catch {set Command(text)} t]} {
	# no command window yet, make one
	set olddoc $CODEdocument
	set doc [COMMAND]
	update idletasks; # arg! This seems to be needed for bindings to work
	$doc configure -state withdrawn
	if {$olddoc != {}} {
	    $olddoc raise
	}
        set t $Command(text)
    }
    CommandInsert $t insert $line\n
    history add $line
    $t mark set output end
    if {$stdout != ""} {
        $t insert output $stdout\n stdout
    }
    if {$stderr != ""} {
        $t insert output $stderr\n stderr
    }
    CommandPrompt $t
    set Command(event) [history nextid]
}

#
# CommandPreferences - set preferences for the command window
#
proc code::CommandPreferences {t} {
    $t tag configure prompt -foreground [Preference Command colorprompt]
    $t tag configure stdout -foreground [Preference Command colorstdout]
    $t tag configure stderr -foreground [Preference Command colorstderr]
    $t tag configure sel -background [Preference General colorselection]
    $t configure -font [Preference Command fontcommand]
    $t config -bg [Preference General colorbackground]
    CommandPrompt $t
}

#
# CommandInsert - insert a string in the command window
#
proc code::CommandInsert {w index s} {
    if {[string match {} $s] || [string match disabled [$w cget -state]]} {
        return 0
    }
    if {[$w comp insert != $index]} {
        catch {
            if {[$w comp sel.first <= $index] && [$w comp sel.last >= $index]} {
                $w delete sel.first sel.last
            }
	}
        $w insert $index $s
	return 0
    }
    if [$w comp insert < limit] {
        $w mark set insert end
    }
    catch {
        if {[$w comp sel.first <= insert] && [$w comp sel.last >= insert]} {
            $w delete sel.first sel.last
        }
    }
    $w insert insert $s
    $w see insert
    return 0
}

#
# CommandDelete - delete from the command window
#
proc code::CommandDelete {w start end} {
    if {[string match disabled [$w cget -state]]} {
        return 0
    }
    if {[$w compare insert < $start] || [$w compare insert >= $end]} {
        catch {
            if {[$w comp sel.first <= $start] && [$w comp sel.last >= $start]} {
                $w delete sel.first sel.last
            }
	}
        $w delete $start $end
	return 0
    }
    if {   [string compare {} [$w tag nextrange sel 1.0 end]] \
        && [$w compare sel.first >= limit]} {
        $w delete sel.first sel.last
    } 
    if {[$w compare $start < limit]} {
	set start limit
        if {[$w compare $end >= limit]} {
            $w delete $start $end
	    $w index insert $start
            $w see insert
	}
    }
    return 0
}

#
# CommandEval - evaluates commands input into console window
# This is the first stage of the evaluating commands in the console.
# They need to be broken up into consituent commands (by CommandCmdSep) in
# case a multiple commands were pasted in, then each is eval'ed (by
# CommandEvalCmd) in turn.  Any uncompleted command will not be eval'ed.
# ARGS:	w	- console text widget
# 
proc code::CommandEval {t} {
    global code::Command

    CommandCmdSep [CommandCmdGet $t] cmds Command(cmd)
    $t mark set insert end-1c
    $t insert end \n
    if [llength $cmds] {
        foreach cmd $cmds {CommandEvalCmd $t $cmd}
        $t insert insert $Command(cmd) {}
    } elseif {[info complete $Command(cmd)] && \
	![regexp {[^\\]\\$} $Command(cmd)]} {
        CommandEvalCmd $t $Command(cmd)
    }
    $t see insert
}

#
# CommandEvalCmd - evaluates a single command, adding it to history
# ARGS:	w	- console text widget
# 	cmd	- the command to evaluate
# 
proc code::CommandEvalCmd {t cmd} {
    global code::Command

    $t mark set output end
    if {[string compare {} $cmd]} {
        set err 0
        if {$Command(subhistory)} {
            set ev [history nextid]
            incr ev -1
            if {[string match !! $cmd]} {
                set err [catch {history event $ev} cmd]
                if {!$err} {
	            $t insert output $cmd\n stdin
	        }
            } elseif {[regexp {^!(.+)$} $cmd dummy event]} {
                set err [catch {history event $event} cmd]
                if {!$err} {
		    $t insert output $cmd\n stdin
	        }
            } elseif {[regexp {^\^([^^]*)\^([^^]*)\^?$} $cmd dummy old new]} {
                if ![set err [catch {history event $ev} cmd]] {
                    regsub -all -- $old $cmd $new cmd
                    $t insert output $cmd\n stdin
                }
            }
        }
        if $err {
            $t insert output $cmd\n stderr
        } else {
	    # change to the command directory
	    set pwd [pwd]
	    cd $Command(pwd)
	    # execute the command
            if [catch {uplevel #0 $cmd} res] {
                set Commmand(errorInfo) $res
                set err 1
            }
	    # remember current directory
	    set Command(pwd) [pwd]
	    # change back to the old directory
	    cd $pwd
            history add $cmd
            if $err {
                $t insert output $res\n stderr
            } elseif {[string compare {} $res]} {
                $t insert output $res\n stdout
            }
        }
    }
    CommandPrompt $t
    set Command(event) [history nextid]
}

#
# CommandPrompt - displays the prompt in the console widget
# Outputs:	prompt to console
# 
proc code::CommandPrompt {t {pre {}} {post {}} {prompt {}}} {

    if {[string compare {} $pre]} {
        $t insert end $pre stdout
        }
    set i [$t index end-1c]
    if {[string compare {} $prompt]} {
        $t insert end $prompt prompt
    } else {
        $t insert end [subst [Preference Command prompt]] prompt
    }
    $t mark set output $i
    $t mark set insert end
    $t mark set limit insert
    $t mark gravity limit left
    if {[string compare {} $post]} {
	$t insert end $post stdin
    }
    $t see end
}

#
# CommandCmdGet - gets the current command from the console widget
# ARGS:	w	- console text widget
# Returns:	text which compromises current command line
# 
proc code::CommandCmdGet {w} {
    if {[string match {} [set ix [$w tag nextrange prompt limit end]]]} {
      $w tag add stdin limit end-1c
      return [$w get limit end-1c]
    }
}

#
# CommandCmdSep - separates multiple commands into a list and remainder
# ARGS:	cmd	- (possible) multiple command to separate
# 	list	- varname for the list of commands that were separated.
#	rmd	- varname of any remainder (like an incomplete final command).
#		If there is only one command, it's placed in this var.
# Returns:	constituent command info in varnames specified by list & rmd.
# 
proc code::CommandCmdSep {cmd ls rmd} {
    upvar $ls cmds $rmd tmp
    set tmp {}
    set cmds {}
    foreach cmd [split [set cmd] \n] {
        if [string comp {} $tmp] {
          append tmp \n$cmd
        } else {
            append tmp $cmd
        }
        if {[info complete $tmp] && ![regexp {[^\\]\\$} $tmp]} {
            lappend cmds $tmp
            set tmp {}
        }
    }
    if {[string comp {} [lindex $cmds end]] && [string match {} $tmp]} {
        set tmp [lindex $cmds end]
        set cmds [lreplace $cmds end end]
    }
}

# Unknown changed to get output into Command window
# unknown:
# Invoked automatically whenever an unknown command is encountered.
# Works through a list of "unknown handlers" that have been registered
# to deal with unknown commands.  Extensions can integrate their own
# handlers into the "unknown" facility via "unknown_handle".
#
# If a handler exists that recognizes the command, then it will
# take care of the command action and return a valid result or a
# Tcl error.  Otherwise, it should return "-code continue" (=2)
# and responsibility for the command is passed to the next handler.
#
# Arguments:
# args -	A list whose elements are the words of the original
#		command, including the command name.

proc unknown args {
    global unknown_handler_order unknown_handlers errorInfo errorCode

    #
    # Be careful to save error info now, and restore it later
    # for each handler.  Some handlers generate their own errors
    # and disrupt handling.
    #
    set savedErrorCode $errorCode
    set savedErrorInfo $errorInfo

    if {![info exists unknown_handler_order] || ![info exists unknown_handlers]} {
	set unknown_handlers(tcl) tcl_unknown
	set unknown_handler_order tcl
    }

    foreach handler $unknown_handler_order {
        set status [catch {uplevel $unknown_handlers($handler) $args} result]

        if {$status == 1} {
            #
            # Strip the last five lines off the error stack (they're
            # from the "uplevel" command).
            #
            set new [split $errorInfo \n]
            set new [join [lrange $new 0 [expr [llength $new] - 6]] \n]
            return -code $status -errorcode $errorCode \
                -errorinfo $new $result

        } elseif {$status != 4} {
            return -code $status $result
        }

        set errorCode $savedErrorCode
        set errorInfo $savedErrorInfo
    }

    set name [lindex $args 0]
    return -code error "invalid command name \"$name\""
}

# tcl_unknown:
# Invoked when a Tcl command is invoked that doesn't exist in the
# interpreter:
#
#	1. See if the autoload facility can locate the command in a
#	   Tcl script file.  If so, load it and execute it.
#	2. If the command was invoked interactively at top-level:
#	    (a) see if the command exists as an executable UNIX program.
#		If so, "exec" the command.
#	    (b) see if the command requests csh-like history substitution
#		in one of the common forms !!, !<number>, or ^old^new.  If
#		so, emulate csh's history substitution.
#	    (c) see if the command is a unique abbreviation for another
#		command.  If so, invoke the command.
#
# Arguments:
# args -	A list whose elements are the words of the original
#		command, including the command name.

proc tcl_unknown args {
  global auto_noexec auto_noload env unknown_pending tcl_interactive
  global errorCode errorInfo

  # Save the values of errorCode and errorInfo variables, since they
  # may get modified if caught errors occur below.  The variables will
  # be restored just before re-executing the missing command.

  set savedErrorCode $errorCode
  set savedErrorInfo $errorInfo
  set name [lindex $args 0]
  if ![info exists auto_noload] {
    #
    # Make sure we're not trying to load the same proc twice.
    #
    if [info exists unknown_pending($name)] {
      unset unknown_pending($name)
      if ![array size unknown_pending] { unset unknown_pending }
      return -code error \
	  "self-referential recursion in \"unknown\" for command \"$name\""
    }
    ## FIX delete line
    set unknown_pending(dummy) dummy
    set unknown_pending($name) pending
    set ret [catch {auto_load $name [uplevel 1 {namespace current}]} msg]
    ## FIX no catch
    catch { unset unknown_pending($name) }
    if $ret {
      return -code $ret -errorcode $errorCode \
	  "error while autoloading \"$name\": $msg"
    }
    if ![array size unknown_pending] { unset unknown_pending }
    if $msg {
      set errorCode $savedErrorCode
      set errorInfo $savedErrorInfo
      set code [catch {uplevel $args} msg]
      if {$code ==  1} {
	#
	# Strip the last five lines off the error stack (they're
	# from the "uplevel" command).
	#

	set new [split $errorInfo \n]
	set new [join [lrange $new 0 [expr [llength $new] - 6]] \n]
	return -code error -errorcode $errorCode \
	    -errorinfo $new $msg
      } else {
	return -code $code $msg
      }
    }
  }
  if {[info level] == 1 && [string match {} [info script]] \
	  && [info exists tcl_interactive] && $tcl_interactive} {
    if ![info exists auto_noexec] {
      set new [auto_execok $name]
      if {$new != ""} {
	set errorCode $savedErrorCode
	set errorInfo $savedErrorInfo
	return [uplevel exec $new [lrange $args 1 end]]
	#return [uplevel exec >&@stdout <@stdin $new [lrange $args 1 end]]
      }
    }
    set errorCode $savedErrorCode
    set errorInfo $savedErrorInfo
    ##
    ## History substitution moved into CommandEvalCmd
    ##
    set cmds [info commands $name*]
    if {[llength $cmds] == 1} {
      return [uplevel [lreplace $args 0 0 $cmds]]
    }
    if {[llength $cmds]} {
      if {$name == ""} {
	return -code error "empty command name \"\""
      } else {
	return -code error \
	    "ambiguous command name \"$name\": [lsort $cmds]"
      }
    }
  }
  return -code continue
}

#
# CommandClear - clears the buffer of the console (not the history though)
# 
proc code::CommandClear {t {pcnt 100}} {
  if {![regexp {^[0-9]*$} $pcnt] || $pcnt < 1 || $pcnt > 100} {
    return -code error \
	"invalid percentage to clear: must be 1-100 (100 default)"
  } elseif {$pcnt == 100} {
    $t delete 1.0 end
    eval $t mark unset [$t mark names]
  } else {
    set tmp [expr $pcnt/100.0*[$t index end]]
    $t delete 1.0 "$tmp linestart"
  }
}

