#
#	Configure - configure code preferences
#

set code::ConfigureFileOptionsCount 0

proc code::Configure {w command args} {
    variable Configure
    variable Build
    variable PreferenceFile

    switch -exact $command {
    build {
	# open specified file as a project and build it
	PreferenceOpen $w [lindex $args 0]
	if {[catch {Build $w build} msg]} {
            tk_messageBox \
		-parent $w \
	        -icon error \
	        -message "Error building $args: $msg" \
	        -type ok
	}
    }
    edit {
	# open specified file as a project and edit it
	PreferenceOpen $w [lindex $args 0]
	PreferenceProject
    }
    open {
	# open the specified file as a project
	set file [lindex $args 0]
	set oldrecent [Preference General recentprojects]
	PreferenceOpen $w [lindex $args 0]
	Directory {} refresh
	# add to the recent files list
	set name [BuildRealFileName $file]
	set files [Preference General recentprojects]
	set count [Preference General recentprojectcount]
	if {[lsearch $files $name] == -1} {
	    # add this one
	    set files [concat [list $name] $files]
	    if {[llength $files] > $count} {
	        set files [lreplace $files $count end]
	    }
    	    Preference General recentprojects $files
	}
	# add recent projects from the old project
	set hasone 0
	foreach oldproject $oldrecent {
	    if {[lsearch $files $oldproject] == -1 && \
	      [llength $files] <= $count} {
		set hasone 1
		lappend files $oldproject
	    }
	}
        if {$hasone} {
    	    Preference General recentprojects $files
	}
    }
    }
}

#
# ConfigureRename - rename a file in a project
#
proc code::ConfigureRename {orig new {newdir {}}} {
    variable Build

    set orig [BuildFileName $orig]
    if {$newdir != {}} {
        # changing project directory
        set old $Build(PROJECT)
        set Build(PROJECT) $newdir
        set new [BuildFileName $new]
        set Build(PROJECT) $old
    } else {
        set new [BuildFileName $new]
    }

    set fileschanged 0
    set files [Preference Build files]
    set librarieschanged 0
    set libraries [Preference Build libraries]
    set index 0
    set match 0
    foreach afile $files {
        if {[lindex $afile 0] == $orig} {
   	    set match 1
	    break
	}
	incr index
    }
    if {$match} {
        set options [lindex $afile 1]
        set depends [lindex $afile 2]
  
	if {$options == {} && $depends == {}} {
	    # just the file name
	    set result [list $new]
	} else {
	    set result [concat [list $new] [list $options] [list $depends]]
        }
        set fileschanged 1
        set files [lreplace $files $index $index $result]
    } else {
        # check libraries
        set index 0
        set match 0
        foreach afile $libraries {
            if {[lindex $afile 0] == $orig} {
    	        set match 1
	        break
	    }
	    incr index
	}
	if {$match} {
	    set options [lindex $afile 1]
	    set depends [lindex $afile 2]
       
            if {$options == {} && $depends == {}} {
	        # just the file name
	        set result [list $new]
  	    } else {
	        set result [concat [list $new] [list $options] [list $depends]]
	    }
	    set librarieschanged 1
	    set libraries [lreplace $libraries $index $index $result]
        }
    }
    if {$fileschanged} {
        Preference Build files $files
    }
    if {$librarieschanged} {
        Preference Build libraries $libraries
    }
}

#
# ConfigureHasDepends - does the specified file have dependencies?
#
proc code::ConfigureHasDepends {files} {
    set inproject 0
    set projectfiles [Preference Build files]
    foreach file $files {
        set file [BuildFileName $file]
	foreach afile $projectfiles {
	    if {[lindex $afile 0] == $file} {
	        if {[lindex $afile 2] != {}} {
	   	    return 1
		}
		return 0
	    }
	}
    }
    set projectfiles [Preference Build libraries]
    foreach file $files {
        set file [BuildFileName $file]
	foreach afile $projectfiles {
	    if {[lindex $afile 0] == $file} {
	        if {[lindex $afile 2] != {}} {
	  	    return 1
		}
		return 0
	    }
	}
    }
    return 0
}

#
# ConfigureAddDependency - add a dependency to the specified file(s)
#
proc code::ConfigureAddDependency {files} {
    set initialdir [Preference Configure dirdependadd]
    if {$initialdir == {} || ![file isdirectory $initialdir]} {
        set initialdir [pwd]
    }
    set add [tk_getOpenFile -title "Select file to add to dependency list" \
        -initialdir $initialdir]
    if {$add == {}} {
        return
    }
    Preference Configure dirdependadd [file dirname $add]
    set add [BuildFileName $add]

    # add dependancies given files
    set fileschanged 0
    set projectfiles [Preference Build files]
    set librarieschanged 0
    set libraries [Preference Build libraries]
    foreach file $files {
        set file [BuildFileName $file]
	set index 0
	set match 0
	foreach afile $projectfiles {
	    if {[lindex $afile 0] == $file} {
	        set match 1
	        break
	    }
	    incr index
	}
	if {$match} {
	    set options [lindex $afile 1]
	    set depends [lindex $afile 2]
	    set newdepends $depends
	    if {[lsearch $depends $add] == -1} {
	        set newdepends $depends
	        lappend newdepends $add
	    }
	    if {$newdepends != $depends} {
	        # something has changed
   
    	        if {$options == {} && $newdepends == {}} {
		    # just the file name
		    set result $file
		} else {
		    set result [concat [list $file] [list $options] [list $newdepends]]
		}
		set fileschanged 1
		set projectfiles [lreplace $projectfiles $index $index $result]
	    }
	} else {
	    # check libraries
	    set index 0
	    set match 0
	    foreach afile $libraries {
	        if {[lindex $afile 0] == $file} {
	            set match 1
	            break
	        }
	        incr index
	    }
	    if {$match} {
	        set options [lindex $afile 1]
	        set depends [lindex $afile 2]
	        set newdepends $depends
	        if {[lsearch $depends $add] == -1} {
	            set newdepends $depends
	            lappend newdepends $add
	        }
	        if {$newdepends != $depends} {
	            # something has changed
        
	            if {$options == {} && $newdepends == {}} {
	                # just the file name
	                set result $file
	            } else {
	                set result [concat [list $file] [list $options] [list $newdepends]]
	            }
	            set librarieschanged 1
	            set libraries [lreplace $libraries $index $index $result]
	        }
	    }
	}
    }
    if {$fileschanged} {
        Preference Build files $projectfiles
    }
    if {$librarieschanged} {
        Preference Build libraries $libraries
    }
}

#
# ConfigureRemoveDependMenu - create the remove depend menu
#
proc code::ConfigureRemoveDependMenu {m} {
    if {![winfo exists $m]} {
        menu $m -tearoff 0 \
	    -postcommand "code::ConfigureRemoveDependMenuBuild $m"
    }
    return $m
}

#
# ConfigureRemoveDependMenuBuild - build a menu file depends
#
proc code::ConfigureRemoveDependMenuBuild {m} {
    variable Configure
    variable CODEcurrentfiles

    set file [lindex $CODEcurrentfiles 0]
    $m delete 0 end
    # build the menu entries
    foreach depend [ConfigureGetDepends $file] {
        set fullname [BuildRealFileName $depend]
        set depend [BuildSubstitute $depend]
        $m add command -label $depend \
            -command "code::ConfigureRemoveDepend $file \{$fullname\}"
    }
    return $m
}

#
# ConfigureDepends - find #include dependencies for files
#
proc code::ConfigureDepends {files} {
    foreach file $files {
        set cpp [cppopen [BuildRealFileName $file]]
	foreach dir [BuildCIncludes $file] {
	    cppinclude $cpp $dir
	}
	foreach define [BuildCDefines $file] {
	    eval cppdefine $cpp $define
	}
	ConfigureRemoveAllDepends $file
	set depends [cppdepends $cpp]
	foreach depend $depends {
	    ConfigureAddDepend $file $depend
	}
	cppclose $cpp
    }
}

#
# ConfigureDoDepends - find all #include dependencies for a project
#
proc code::ConfigureDoDepends {} {
    foreach file [Preference Build files] {
        set file [lindex $file 0]
	set extension [string tolower [file extension $file]]
        set type [Preference FileTypes file$extension]
        if {$type == {}} {
	    continue
        }
        set type [lindex $type 3]
        if {$type == "c"} {
	    ConfigureDepends $file
	}
    }
}

#
# ConfigureInProject - check to see whether file(s) are in the project
#
proc code::ConfigureInProject {files} {
    variable PreferenceFile

    set inproject 0
    # check project source files
    set projectfiles [Preference Build files]
    foreach file $files {
        set file [BuildFileName $file]
        foreach afile $projectfiles {
	    if {[lindex $afile 0] == $file} {
	        set inproject 1
	        break
	    }
	}
    }
    # check project libraries
    if {!$inproject} {
        set projectfiles [Preference Build libraries]
        foreach file $files {
            set file [BuildFileName $file]
            foreach afile $projectfiles {
   	        if {[BuildRealFileName [lindex $afile 0]] == 
	          [BuildRealFileName $file]} {
	            set inproject 1
	            break
	        }
	    }
        }
    }
    # check subprojects
    if {!$inproject} {
        set projectfiles [Preference Build subprojects]
        foreach file $files {
            set file [BuildFileName $file]
            foreach afile $projectfiles {
 	        if {[BuildRealFileName [lindex $afile 0]] == 
	          [BuildRealFileName $file]} {
	            set inproject 1
	            break
	        }
	    }
        }
    }
    # project itself is "in" the project
    if {!$inproject} {
        foreach file $files {
            set file [BuildFileName $file]
            if {[BuildRealFileName $PreferenceFile] == 
                 [BuildRealFileName $file]} {
                set inproject 1
            }
	}
    }
    return $inproject
}

#
# OPTIONS - set file(s) options
#
proc code::OPTIONS {files} {
    # set options for the given files
    foreach file $files {
        set extension [string tolower [file extension $file]]
        set type [Preference FileTypes file$extension]
        set type [lindex $type 3]
	if {$type == "c"} {
	    ConfigureCCOptions $file
	} elseif {$type == "asm"} {
	    ConfigureASOptions $file
	}
   }
}

#
# ConfigureASOptions - set assembler options for file(s)
#
proc code::ConfigureASOptions {files} {
    foreach file $files {
        set file [BuildFileName $file]
        ConfigureEditFileOptions $file [Preference Build processor]asoptions
    }
}

#
# ConfigureCCOptions - set C compiler options for file(s)
#
proc code::ConfigureCCOptions {files} {
    foreach file $files {
        set file [BuildFileName $file]
        ConfigureEditFileOptions $file [Preference Build processor]ccoptions
    }
}

#
# ConfigureCanAdd - check if a file can be added to a project
#
proc code::ConfigureCanAdd {file} {
    # find the class of the file, must be c, asm, project, obj, or lib
    if {[catch {file stat $file stat}]} {
        # file is inaccessable for some reason
	set stat(type) file
    }
    set extension [string tolower [file extension $file]]
    set Type [Preference FileTypes $stat(type)$extension]
    if {$Type == {}} {
        set Type [Preference FileTypes $stat(type)]
        if {$Type == {}} {
            error "File type botch."
        }
    }
    set class [lindex $Type 3]
    switch $class {
	c -
	asm -
	obj { set type normal }
	lib { set type lib }
	project { set type project }
	default { set type {} }
    }
    return $type
}

#
# ConfigureAddProject - add file(s) to the current project
#
proc code::ConfigureAddProject {files} {
    variable Build

    if {$Build(PROJECT) == {}} {
        # set up the project initially
        PreferencesSave
    }
    if {$files == {}} {
        # no file given, ask for a specific file
	set extension {.c .code .s .asm .ddf}
	set proc [Preference Build processor]
        if {$proc != {}} {
	    lappend extension .s[Preference Build [set proc]E]
	    lappend extension .a[Preference Build [set proc]E]
	}

	set initialdir [Preference Configure dirprojectadd]
	if {$initialdir == {} || ![file isdirectory $initialdir]} {
	    set initialdir [pwd]
	}
	set files [tk_getOpenFile -title "Select file to add to Project" \
	    -initialdir $initialdir \
            -filetypes "
                \"{Project Files} [list $extension]\"
            " ]
	if {$files == {}} {
	    return
	}
        Preference Configure dirprojectadd [file dirname $files]
	set files [list $files]
    }

    set fileschanged 0
    set projectfiles [Preference Build files]
    set librarieschanged 0
    set libraries [Preference Build libraries]
    set subprojectschanged 0
    set subprojects [Preference Build subprojects]
    foreach file $files {

	set type [ConfigureCanAdd $file]
	if {$type == {}} {
            tk_messageBox \
	        -parent . \
	        -icon error \
	        -message "\"[file tail $file]\" cannot be added to the current project.\nIt is not a C, Assembly, Object, Library, or Project file." \
	        -type ok
	    continue
	}
        set file [BuildFileName $file]

        if {$type == "library"} {
            set add 1
	    foreach afile $libraries {
	        if {[lindex $afile 0] == $file} {
	            set add 0
		    break
		}
	    }
	    if {$add} {
	        set librarieschanged 1
	        set libraries [concat [list $file] $libraries]
	    }
        } elseif {$type == "project"} {
	    set add 1
	    foreach afile $subprojects {
	        if {[lindex $afile 0] == $file} {
	            set add 0
	            break
	        }
	    }
	    if {$add} {
	        set subprojectschanged 1
	        set subprojects [concat [list $file] $subprojects]
	    }
	} else {
	    set add 1
	    foreach afile $projectfiles {
	        if {[lindex $afile 0] == $file} {
	            set add 0
	            break
	        }
	    }
	    if {$add} {
	        set fileschanged 1
	        lappend projectfiles [list $file]
	    }
	}
    }
    if {$fileschanged} {
        Preference Build files $projectfiles
    }
    if {$librarieschanged} {
        Preference Build libraries $libraries
    }
    if {$subprojectschanged} {
        Preference Build subprojects $subprojects
    }
}

#
# ConfigureRemoveProject - remove file(s) from the current project
#
proc code::ConfigureRemoveProject {files} {
    if {$files == {}} {
        tk_messageBox \
	    -parent . \
	    -icon warning \
	    -message "Unnamed file is not in the current project" \
	    -type ok
	return
    }
    set fileschanged 0
    set projectfiles [Preference Build files]
    set librarieschanged 0
    set libraries [Preference Build libraries]
    set subprojectschanged 0
    set subprojects [Preference Build subprojects]
    foreach file $files {
        set file [BuildFileName $file]
	set index 0
	set foundone 0
	set found 0
	foreach afile $projectfiles {
	    if {[lindex $afile 0] == $file} {
	        set found 1
	        set foundone 1
	        break
	    }
	    incr index
	}
	if {$found} {
	    set fileschanged 1
	    set projectfiles [concat [lrange $projectfiles 0 [expr $index - 1]] \
	        [lrange $projectfiles [expr $index + 1] end]]
	}
	set index 0
	set found 0
	foreach afile $libraries {
	    if {[lindex $afile 0] == $file} {
	        set found 1
	        set foundone 1
	        break
	    }
	    incr index
	}
	if {$found} {
	    set librarieschanged 1
	    set libraries [concat [lrange $libraries 0 [expr $index - 1]] \
	        [lrange $libraries [expr $index + 1] end]]
	}
	set index 0
	set found 0
	foreach afile $subprojects {
	    if {[lindex $afile 0] == $file} {
	        set found 1
	        set foundone 1
	        break
	    }
	    incr index
	}
	if {$found} {
	    set subprojectschanged 1
	    set subprojects [concat [lrange $subprojects 0 [expr $index - 1]] \
	        [lrange $subprojects [expr $index + 1] end]]
	}
	if {!$foundone} {
	    tk_messageBox \
	        -parent . \
		-icon warning \
	        -message "\"[file tail $file]\" is not in the current project" \
	        -type ok
	}
    }
    if {$fileschanged} {
        Preference Build files $projectfiles 1
    }
    if {$librarieschanged} {
        Preference Build libraries $libraries 1
    }
    if {$subprojectschanged} {
        Preference Build subprojects $subprojects 1
    }
}

#
# ConfigureGetDepends - get the dependencies for a file
#
proc code::ConfigureGetDepends {file} {
    set file [BuildFileName $file]
    set match 0
    foreach afile [Preference Build files] {
        if {[lindex $afile 0] == $file} {
	    set match 1
	    break
	}
    }
    if {$match} {
        return [lindex $afile 2]
    }
    foreach afile [Preference Build libraries] {
        if {[lindex $afile 0] == $file} {
            return [lindex $afile 2]
	}
    }
    return {}
}

#
# ConfigureAddDepend - add a dependency to a file
#
proc code::ConfigureAddDepend {file depend} {
    set file [BuildFileName $file]
    set depend [BuildFileName $depend]
    set match 0
    set index 0
    foreach afile [Preference Build files] {
        if {[lindex $afile 0] == $file} {
	    set match 1
	    break
	}
	incr index
    }
    if {$match} {
        set options [lindex $afile 1]
	set depends [lindex $afile 2]
        if {[lsearch $depends $depend] == -1} {
	    # adding a depend
	    lappend depends $depend
	    set result [concat [list $file] [list $options] [list $depends]]
            set files [lreplace [Preference Build files] $index $index $result]
	    Preference Build files $files 1
        }
	return
    }
    set index 0
    foreach afile [Preference Build libraries] {
        if {[lindex $afile 0] == $file} {
	    set match 1
	    break
	}
	incr index
    }
    if {$match} {
        set options [lindex $afile 1]
	set depends [lindex $afile 2]
        if {[lsearch $depends $depend] == -1} {
	    # adding a depend
	    lappend depends $depend
	    set result [concat [list $file] [list $options] [list $depends]]
            set libraries [lreplace [Preference Build libraries] $index $index $result]
	    Preference Build libraries $libraries 1
        }
	return
    }
}

#
# ConfigureRemoveDepend - remove a dependency from a file
#
proc code::ConfigureRemoveDepend {file depend} {
    set file [BuildFileName $file]
    set depend [BuildFileName $depend]
    set match 0
    set index 0
    foreach afile [Preference Build files] {
        if {[lindex $afile 0] == $file} {
	    set match 1
	    break
	}
	incr index
    }
    if {$match} {
        set options [lindex $afile 1]
	set depends [lremove [lindex $afile 2] $depend]
        if {$options == {} && $depends == {}} {
	    # just the file name
	    set result $file
	} else {
	    set result [concat [list $file] [list $options] [list $depends]]
        }
        set files [lreplace [Preference Build files] $index $index $result]
	Preference Build files $files 1
	return
    }
    set index 0
    foreach afile [Preference Build libraries] {
        if {[lindex $afile 0] == $file} {
	    set match 1
	    break
	}
	incr index
    }
    if {$match} {
        set options [lindex $afile 1]
	set depends [lremove [lindex $afile 2] $depend]
        if {$options == {} && $depends == {}} {
	    # just the file name
	    set result $file
	} else {
	    set result [concat [list $file] [list $options] [list $depends]]
        }
        set libraries [lreplace [Preference Build libraries] $index $index $result]
	Preference Build libraries $libraries 1
	return
    }
}

#
# ConfigureRemoveAllDepends - remove all dependencies from a file
#
proc code::ConfigureRemoveAllDepends {file} {
    set file [BuildFileName $file]
    set match 0
    set index 0
    foreach afile [Preference Build files] {
        if {[lindex $afile 0] == $file} {
	    set match 1
	    break
	}
	incr index
    }
    if {$match} {
        set options [lindex $afile 1]
        if {$options == {}} {
	    # just the file name
	    set result $file
	} else {
	    set result [concat [list $file] [list $options] {}]
        }
        set files [lreplace [Preference Build files] $index $index $result]
	Preference Build files $files 1
	return
    }
    set index 0
    foreach afile [Preference Build libraries] {
        if {[lindex $afile 0] == $file} {
	    set match 1
	    break
	}
	incr index
    }
    if {$match} {
        set options [lindex $afile 1]
        if {$options == {}} {
	    # just the file name
	    set result $file
	} else {
	    set result [concat [list $file] [list $options] {}]
        }
        set libraries [lreplace [Preference Build libraries] $index $index $result]
	Preference Build libraries $libraries 1
	return
    }
}

#
# ConfigureColor - set a color
#
proc code::ConfigureColor {w name cat elt b} {
    set color [tk_chooseColor -title "Choose the $cat $name color" -parent $w \
	-initialcolor [Preference $cat $elt]]
    if {[string compare $color ""]} {
	if {[Preference $cat $elt] != $color} {
	    Preference $cat $elt $color
	    $b configure -fg $color
	}
    }
}

#
# ConfigureFont - set a font
#
proc code::ConfigureFont {w name cat elt b} {
    set font [tk_chooseFont -title "Choose the $cat $name" -parent $w \
	-initialfont [Preference $cat $elt]]
    if {[string compare $font ""]} {
	if {[Preference $cat $elt] != $font} {
	    Preference $cat $elt $font
	}
    }
}

#
# ConfigureEdit - edit a complex option
#
proc code::ConfigureEdit {w name cat elt {proc {}}} {
    set value [Preference $cat $elt]
    set newvalue [Configure $w $name $value $proc]
    if {$value != $newvalue} {
        Preference $cat $elt $newvalue 1
    }
}

#
# ConfigureOptions - configure build options
#
proc code::ConfigureOptions {w name cat elt} {
    set options [Preference $cat $elt]
    set newoptions [Configure $w $elt $options]
    if {$options != $newoptions} {
        Preference $cat $elt $newoptions 1
    }
}

#
# ConfigureEditGlobalOptions - Edit global program options
#
proc code::ConfigureEditGlobalOptions {which} {
    set value [Preference Build $which]
    ConfigureEditOptions Global $which "code::ConfigureSetGlobalOptions $which" $value
}

proc code::ConfigureSetGlobalOptions {which value} {
    set oldvalue [Preference Build $which]
    set value [lsort -dictionary $value]
    if {$value != $oldvalue} {
	# value has changed
        Preference Build $which $value 1
	if {[lsearch $value "-gl"] != -1} {
	    # large model is set
	    if {![Preference Build largemodel]} {
		Preference Build largemodel 1
	    }
	} elseif {[Preference Build largemodel]} {
	    # large model is clear, clear global flag if set
  	    Preference Build largemodel 0
	}
    }
}

#
# ConfigureEditFileOptions - Edit file specific program options
#
proc code::ConfigureEditFileOptions {file which} {
    variable ConfigureFileOptions
    variable ConfigureFileOptionsCount

    set global [Preference Build $which]

    if {[info exists ConfigureFileOptions($file)]} {
	# already used this one
	set which $ConfigureFileOptions($file)
    } else {
	# a new one
	set which $which$ConfigureFileOptionsCount
	incr ConfigureFileOptionsCount
	set ConfigureFileOptions($file) $which
    }
    set files [Preference Build files]
    # find the file info
    foreach afile $files {
	if {[lindex $afile 0] == $file} {
	    break
	}
    }
    set value [lindex $afile 1]
    if {$value == {}} {
	# use global options
	set value $global
    }
    ConfigureEditOptions $file $which \
	"code::ConfigureSetFileOptions [BuildRealFileName $file] $which" $value
}

proc code::ConfigureSetFileOptions {file which value} {
    set value [lsort -dictionary $value]
    set files [Preference Build files]
    set index 0
    set file [BuildFileName $file]
    foreach afile $files {
        if {[lindex $afile 0] == $file} {
            break
        }
        incr index
    }

    set depends [lindex $afile 2]
    if {$value == {} && $depends == {}} {
        # just the file name
        set result $file
    } else {
        set result [concat [list $file] [list $value] [list $depends]]
    }
    set files [lreplace $files $index $index $result]
    Preference Build files $files
}

#
# ConfigureEditOptionsCheck - change option windows when processor changes
#
proc code::ConfigureEditOptionsCheck {} {
    variable ConfigureFileOptions
    foreach file [array names ConfigureFileOptions *] {
	 catch {destroy .$ConfigureFileOptions($file)}
    }
    set proc [Preference Build processor]
    foreach type {${proc}ccoptions ${proc}asoptions ${proc}ldoptions} {
        set value [Preference Build $type]
        ConfigureEditOptions Global $type \
	    "code::ConfigureSetGlobalOptions $type" $value 1
    }
}

#
# ConfigureEditOptions - set up program options
#
proc code::ConfigureEditOptions {whose which func value {check 0}} {
    set top .$which
    # check to see if window should be recreated for a new processor
    if {$check} {
        if {![catch {wm geometry $top} geometry]} {
            # save only the window's position, not its size
	    regexp {([0-9]*x[0-9]*)(.*)} $geometry foo foo geometry
            Preference Geometry $which $geometry
        }
        if {[winfo exists $top] && [winfo ismapped $top]} {
	    # destroy old one and recreate
	    destroy $top
        } else {
	    # otherwise destroy a potential and return
	    catch {destroy $top}
	    return
        }
    }

    set proc [Preference Build processor]
    if {$proc == {}} {
        tk_messageBox \
	    -icon error \
	    -message "You have to select a processor first." \
	    -type ok
	    return 
    }

    set geometry [Preference Geometry $which]
    if {[winfo exists $top]} {
        if {$geometry != {}} {
	    # the window has had its geometry change saved
	    setGeometry $top $geometry
	}
        wm deiconify $top
	raise $top
        # this code seems to get around a bug in Tk (wm geometry for
        # a hidden window)
        global tcl_platform
        if {$tcl_platform(platform) == "windows" && $geometry != {}} {
            setGeometry $top $geometry
        }
	return
    }

    switch -glob $which {
	*ccoptions* {
	    set dir "CCoptions"
	    set name "C compiler"
	}
	*asoptions* {
	    set dir "ASoptions"
	    set name "assembler"
	}
	*ldoptions* {
	    set dir "LDoptions"
	    set name "linker"
	}
    }
    # create the Options window
    if {[ConfigureNotebook $top $proc $dir $which $func $value] == {}} {
        tk_messageBox \
	    -icon info \
	    -message "Option configuration not available for the $proc." \
	    -type ok
	return
    }

    # set the Options Edit name
    wm title $top "$whose $name options" 
    if {[FindManual Applications $dir$proc.html] != {}} {
        bind $top <F1> "code::OpenHelp Applications Project $dir$proc.html"
    } else {
        bind $top <F1> "code::OpenHelp Applications Project $dir.html"
    }
}

#
# ConfigureGetValue - get the value of a variable, or the system default
#
proc code::ConfigureGetValue {cat elt {proc {}}} {
    variable INTROL

    if {$proc == {}} {
        set proc [Preference Build processor]
    }
    if {$proc == {}} {
        tk_messageBox \
	    -icon error \
	    -message "You have to select a processor first." \
	    -type ok
	    return {}
    }

    # look it up in the option database
    set value [Preference $cat $elt]
    if {$value != {}} {
	return $value
    }
    set name [Preference Build $elt]
    if {$name == {}} {
        set name [Preference Build [Preference Build variant]$elt]
        if {$name == {}} {
            set name $elt.cfg
	}
    } 
    # search locally and in the standard place
    if {[catch {open $name} f]} {
        set name [file join $INTROL Libraries Assembly \
	[Preference Build [set proc]gen] $name]
        if {[catch {open $name} f]} {
	    # no value file found
	    return {}
        }
    }
    set value [read $f]
    close $f
    return $value
}

#
# ConfigureSetVector - change the value of a vector
#
proc code::ConfigureSetVector {newname newvalue} {
    variable ConfigureVectors

    set list [ConfigureGetValue System vectors]
    set newlist {}
    set changed 0
    foreach "name value description" $list {
	if {$name == $newname} {
	    if {$value == $newvalue} {
		# unchanged
		return
	    }
	    set changed 1
	    set value $newvalue
	    # update display if called from elsewhere
	    set ConfigureVectors($name) $value
	}
	lappend newlist $name $value $description
    }
    if {$changed} {
	Preference System vectors $newlist
    }
}

#
# ConfigureEditVariables - Edit link time variables
#
proc code::ConfigureEditVariables {w} {
    variable Preferences

    if {[Preference Build ldfile]} {
	# the ld file overrides this
        tk_messageBox \
	    -icon error \
	    -message "Linker variable editing disabled. You have selected a linker command file in the Project Tools window." \
	    -type ok
	return
    }
    set value [ConfigureGetValue System variables]
    if {![info exists Preferences(System,variables)] || \
	    $Preferences(System,variables) == {}} {
        set Preferences(System,variables) $value
    }
   EDITvariable System variables {Linker variables and additional commands}
}

#
# ConfigureGetSections - the predefined section groups
#
proc code::ConfigureGetSections {{proc {}}} {
    variable Build
    set sections [ConfigureGetValue System sections $proc]
    foreach {name unique description value} $sections {
	# inform build of section definitions
	set Build($name) [ConfigureTrim $value]
    }
    return $sections
}

#
# ConfigureTrim - remove leading whitewpace form all lines in value
#
proc code::ConfigureTrim {v {leading {}}} {
    set result ""
    set v [string trim $v]
    foreach line [split $v \n] {
        append result "$leading[string trim $line]\n"
    }
    return $result
}

#
# ConfigureGetMap - get a specific map for a processor
#
proc code::ConfigureGetMap {which {proc {}}} {

    # set up the predefined section groups
    set sections [ConfigureGetSections $proc]
    if {$sections == {}} {
	# no processor set
	return {}
    }
    set map [Preference System memory]
    if {$map != {}} {
	return $map
    }
    foreach {name desc map} [ConfigureGetValue Configure memory $proc] {
	if {$name == $which} {
	    return $map
	}
    }
    return {}
}

#
# ConfigureEditEnvironment - set up the runtime environment preferences
#
proc code::ConfigureEditEnvironment {{check 0}} {
    variable INTROL
    variable Preferences

    set top .environment
    # check to see if window should be recreated for a new processor
    if {$check} {
        # do the chip selects window
        ConfigureEditChipSelects 1
        if {![catch {wm geometry $top} geometry]} {
            # save only the window's position, not its size
	    regexp {([0-9]*x[0-9]*)(.*)} $geometry foo foo geometry
            Preference Geometry environment $geometry
        }
        if {[winfo exists $top] && [winfo ismapped $top]} {
	    # destroy old one and recreate
	    destroy $top
        } else {
	    # otherwise destroy a potential and return
	    catch {destroy $top}
	    return
        }
    }

    set proc [Preference Build processor]
    if {$proc == {}} {
        tk_messageBox \
	    -icon error \
	    -message "You have to select a processor first." \
	    -type ok
	    return 
    }

    if {[Preference System startfile]} {
	# the start file overrides this
        tk_messageBox \
	    -icon error \
	    -message "Environment editing disabled. You have selected a startup code file in the Project Tools window." \
	    -type ok
	return
    }
	
    set geometry [Preference Geometry environment]
    if {[winfo exists $top]} {
        if {$geometry != {}} {
	    # the window has had its geometry change saved
	    setGeometry $top $geometry
	}
        wm deiconify $top
	raise $top
        # this code seems to get around a bug in Tk (wm geometry for
        # a hidden window)
        global tcl_platform
        if {$tcl_platform(platform) == "windows" && $geometry != {}} {
            setGeometry $top $geometry
        }
	return
    }

    # create the Environment window

    set variant [Preference Build variant]
    set tabfile [DdfGetTabs]
    set list {Startup IO}
    set hasvectors 0
    if {[ConfigureGetValue System vectors] != {}} {
	set hasvectors 1
        lappend list Vectors
    }
    foreach tab $tabfile {
	lappend list [lindex $tab 0]
    }

    set hasregs 0
    if {[DdfHasRegs]} {
	# add a register value tab
	lappend list Registers
	set hasregs 1
    }
    set top [toplevel $top]
    wm protocol $top WM_DELETE_WINDOW "code::PreferenceClose $top environment"
    Preference General showenvironment 1
    set n $top.nb

    set geometry [Preference Geometry environment]
    setGeometry $top $geometry place
    Notebook:create $n -pages $list -pad 5 -status code::status($top) \
	-width [Preference General projectwidth]

    grid $n -in $top -row 0 -column 0 -sticky nsew
    label $top.l -text "" -textvariable code::status($top) \
	-relief sunken -bd 2 \
        -anchor w -font [Preference General fontstatus] \
	-bg [Preference General colorstatusbackground]
    PreferenceWhenChanged General $top "$top.l config \
        -font \[Preference General fontstatus] \
	-bg \[Preference General colorstatusbackground]"
    grid $top.l -in $top -row 1 -column 0 -sticky ew
    grid columnconfigure $top.l 0 -weight 1
    grid columnconfigure $top 0 -weight 1
    grid rowconfigure $top 0 -weight 1

    # create the Startup tab
    set w [Notebook:frame $n Startup]
    set file [file join $INTROL tcltk code Startup.ui.tcl]
    if {[file exists $file]} {
        source $file
        set help [Startup_ui $w $top [DdfGetStartup]]
        Notebook:pageconfig $n Startup -status $help
        help $top $w $help
    }

    # get the device table
    set devtab [DdfGetDevTab]
    # get the list of legal devices
    set devlist [DdfGet DEVS]
    # add the default device
    lappend devlist [DdfDefaultDevice]
    # create the IO tab
    set w [Notebook:frame $n IO]
    set file [file join $INTROL tcltk code IO.ui.tcl]
    if {[file exists $file]} {
        source $file
        set help [IO_ui $w $top $devlist $devtab]
        Notebook:pageconfig $n IO -status $help
        help $top $w $help
    }

    if {$hasvectors} {
        # create the Vectors tab
        set w [Notebook:frame $n Vectors]
        set file [file join $INTROL tcltk code Vectors.ui.tcl]
        if {[file exists $file]} {
            source $file
            set help [Vectors_ui $w $top]
            Notebook:pageconfig $n Vectors -status $help
            help $top $w $help
        }
    }

    # initialize each tab
    foreach tab $tabfile {
	set tabfile [lindex $tab 1]
	set tabargs [lrange $tab 2 end]
	set tab [lindex $tab 0]
	if {$tabfile == {}} {
	    # no file 
	    set tabfile $tab
	}
        set w [Notebook:frame $n $tab]
  	set file [file join $INTROL tcltk code $tabfile.ui.tcl]
	if {[file exists $file]} {
	    source $file
	    set help [eval ${tabfile}_ui $w $top $tabargs]
            Notebook:pageconfig $n $tab -status $help
	    help $top $w $help
	}
    }

    if {$hasregs} {
        # create the Registers tab
        set w [Notebook:frame $n Registers]
        set file [file join $INTROL tcltk code Registers.ui.tcl]
        if {[file exists $file]} {
            source $file
            set help [Registers_ui $w $top]
            Notebook:pageconfig $n Registers -status $help
            help $top $w $help
        }
    }

    # this code seems to get around a bug in Tk (wm geometry for
    # a hidden window)
    global tcl_platform
    if {$tcl_platform(platform) == "windows" && $geometry != {}} {
	update idletasks
        setGeometry $top $geometry
    }

    bind $top <Configure> "code::PreferenceMove $top environment"

    # set the Environment Edit name
    if {$variant == {}} {
        wm title $top "Configure the $proc (generic) Environment" 
    } else {
        wm title $top "Configure the $variant Environment" 
    }
    if {[FindManual Applications Environment$proc.html] != {}} {
        bind $top <F1> "code::OpenHelp Applications Project Environment$proc.html"
    } else {
        bind $top <F1> "code::OpenHelp Applications Project Environment.html"
    }
}

#
# ConfigureEditChipSelects - set up the chip selects
#
proc code::ConfigureEditChipSelects {{check 0}} {
    variable INTROL

    set top .chipselects
    # check to see if window should be recreated for a new processor
    if {$check} {
        if {![catch {wm geometry $top} geometry]} {
            # save only the window's position, not its size
	    regexp {([0-9]*x[0-9]*)(.*)} $geometry foo foo geometry
            Preference Geometry chipslects $geometry
        }
	# destroy a potential and return
	catch {destroy $top}
	return
    }

    set proc [Preference Build processor]
    if {$proc == {}} {
        tk_messageBox \
	    -icon error \
	    -message "You have to select a processor first." \
	    -type ok
	    return 
    }

    set geometry [Preference Geometry chipselects]
    if {[winfo exists $top]} {
        if {$geometry != {}} {
	    # the window has had its geometry change saved
	    setGeometry $top $geometry
	}
        wm deiconify $top
	raise $top
        # this code seems to get around a bug in Tk (wm geometry for
        # a hidden window)
        global tcl_platform
        if {$tcl_platform(platform) == "windows" && $geometry != {}} {
            setGeometry $top $geometry
        }
	return
    }

    # create the chip selects window

    set variant [Preference Build variant]
    set tabfile [DdfGetCS]
    if {$tabfile == {}} {
        tk_messageBox \
	    -icon info \
	    -message "Chip select configuration not available for the $proc.
Perhaps you need to set a processor variant." \
	    -type ok
	return
    }

    set list {}
    foreach tab $tabfile {
	lappend list [lindex $tab 0]
    }
    set top [toplevel $top]
    wm protocol $top WM_DELETE_WINDOW "code::PreferenceClose $top chipselects"
    wm resizable $top 0 0
    Preference General showchipselects 1
    set n $top.nb

    set geometry [Preference Geometry chipselects]
    setGeometry $top $geometry place
    Notebook:create $n -pages $list -pad 5 -status code::status($top) \
	-width 550 -height 200

    grid $n -in $top -row 0 -column 0 
    label $top.l -text "" -textvariable code::status($top) \
	-relief sunken -bd 2 \
        -anchor w -font [Preference General fontstatus] \
	-bg [Preference General colorstatusbackground]
    PreferenceWhenChanged General $top "$top.l config \
        -font \[Preference General fontstatus] \
	-bg \[Preference General colorstatusbackground]"
    grid $top.l -in $top -row 1 -column 0 -sticky ew
    grid columnconfigure $top.l 0 -weight 1

    # initialize each tab
    foreach tab $tabfile {
	set tabfile [lindex $tab 1]
	set tabargs [lrange $tab 2 end]
	set tab [lindex $tab 0]
	if {$tabfile == {}} {
	    # no file 
	    set tabfile $tab
	}
        set w [Notebook:frame $n $tab]
  	set file [file join $INTROL tcltk code $tabfile.ui.tcl]
	if {[file exists $file]} {
	    source $file
	    set help [eval ${tabfile}_ui $w $top $tabargs]
            Notebook:pageconfig $n $tab -status $help
	    help $top $w $help
	}
    }

    # this code seems to get around a bug in Tk (wm geometry for
    # a hidden window)
    global tcl_platform
    if {$tcl_platform(platform) == "windows" && $geometry != {}} {
	update idletasks
        setGeometry $top $geometry
    }

    bind $top <Configure> "code::PreferenceMove $top chipselects"

    set n $top.nb

    # set the Chip select edit name
    wm title $top "Configure the $variant Chip Selects" 
    if {[FindManual Applications CS$proc.html] != {}} {
        bind $top <F1> "code::OpenHelp Applications Project CS$proc.html"
    } else {
        bind $top <F1> "code::OpenHelp Applications Project CS.html"
    }
}

#
# ConfigureNotebook - create a notebook containing ui tabs
#
proc code::ConfigureNotebook {top proc name pref args} {
    variable INTROL
    variable Preferences

    # get the ui directorys in order:
    # 1. $proc/$name/$variant (if it exists)
    # 2. $proc/$name (if it exists)
    # 3. Common/$name (if it exists)
    set dirs {}
    if {[Preference Build variant] != {}} {
	 # have a variant
	 set dir [file join $INTROL tcltk code $proc $name [Preference Build variant]]
	 if {[file exists $dir]} {
	     lappend dirs [list $dir]
	 }
    }
    set dir [file join $INTROL tcltk code $proc $name]
    if {[file exists $dir]} {
        lappend dirs [list $dir]
    }
    set dir [file join $INTROL tcltk code Common $name]
    if {[file exists $dir]} {
        lappend dirs [list $dir]
    }

    # get the tab order
    set found 0
    foreach dir $dirs {
        if {![catch {open [file join $dir Tabs]} f]} {
	    set found 1
	    break 
        }
    }
    if {!$found} {
	return {}
    }
    set tabfile [read $f]
    close $f

    set list {}
    foreach tab $tabfile {
	lappend list [lindex $tab 0]
    }
    set top [toplevel $top]
    wm title $top $name
    wm protocol $top WM_DELETE_WINDOW "code::PreferenceClose $top $pref"
    Preference General show$pref 1
    set n $top.nb

    set geometry [Preference Geometry $pref]
    setGeometry $top $geometry
    Notebook:create $n -pages $list -pad 5 -status code::status($top)

    grid $n -in $top -row 0 -column 0 
    label $top.l -text "" -textvariable code::status($top) \
	-relief sunken -bd 2 \
        -anchor w -font [Preference General fontstatus] \
	-bg [Preference General colorstatusbackground]
    PreferenceWhenChanged General $top "$top.l config \
        -font \[Preference General fontstatus] \
	-bg \[Preference General colorstatusbackground]"
    grid $top.l -in $top -row 1 -column 0 -sticky ew
    grid columnconfigure $top.l 0 -weight 1


    # initialize each tab
    foreach tab $tabfile {
	set tabfile [lindex $tab 1]
	set tabargs [lrange $tab 2 end]
	set tab [lindex $tab 0]
	if {$tabfile == {}} {
	    # no file 
	    set tabfile $tab
	}
        set w [Notebook:frame $n $tab]
        foreach dir $dirs {
	    # search the directories in order
  	    set file [file join $dir $tabfile.ui.tcl]
	    if {[file exists $file]} {
	        source $file
	        set help [eval ${tabfile}_ui $w $top $tabargs $args]
                Notebook:pageconfig $n $tab -status $help
	        help $top $w $help
		break
	    }
	}
    }

    # this code seems to get around a bug in Tk (wm geometry for
    # a hidden window)
    global tcl_platform
    if {$tcl_platform(platform) == "windows" && $geometry != {}} {
	update idletasks
        setGeometry $top $geometry
    }

    bind $top <Configure> "code::PreferenceMove $top $pref"
    return $top
}

#
# ConfigureEditMemory - Edit target memory map
#
proc code::ConfigureEditMemory {{check 0} {map {}}} {
    variable Configure
    variable ConfigureGroup

    set box .editmemory
    # check to see if window should be recreated for a new processor
    if {$check} {
        if {![catch {wm geometry $box} geometry]} {
            # save only the window's position, not its size
	    regexp {([0-9]*x[0-9]*)(.*)} $geometry foo foo geometry
            Preference Geometry editmemory $geometry
        }
	foreach device [info command .editcontents*] {
	    destroy $device
	}
	if {$map != {}} {
	    Preference System memory {} 1
	}
        if {[winfo exists $box] && [winfo ismapped $box]} {
	    # destroy old one and recreate
	    destroy $box
        } else {
	    # otherwise destroy a potential and return
	    catch {destroy $box}
	    return
        }
    }

    if {[Preference Build ldfile]} {
	# the ld file overrides this
        tk_messageBox \
	    -icon error \
	    -message "Memory Map editing disabled. You have selected a linker command file in the Project Tools window." \
	    -type ok
	return
    }
	
    set geometry [Preference Geometry editmemory]
    if {[winfo exists $box]} {
        if {$geometry != {}} {
	    # the window has had its geometry change saved
	    setGeometry $box $geometry
	}
        wm deiconify $box
	raise $box
        # this code seems to get around a bug in Tk (wm geometry for
        # a hidden window)
        global tcl_platform
        if {$tcl_platform(platform) == "windows" && $geometry != {}} {
            setGeometry $box $geometry
        }
	return
    }

    if {$map == {}} {
	# get the default map for the processor/variant
	set var [Preference Build variant]
	if {$var != {}} {
	    set map [Preference Build ${var}defaultmemory]
        }
	if {$map == {}} {
	    set map Default
	}
    }
    set value [ConfigureGetMap $map [Preference Build processor]]
    if {$value == {}} {
	# no value found or processor not set
	return
    }
    PreferenceSetIfChanged System memory $value
    set ConfigureGroup(map) $map
    set ConfigureGroup(error) 0
    set proc [Preference Build processor]

    toplevel $box 
    wm withdraw $box
    wm protocol $box WM_DELETE_WINDOW "code::ConfigureGroupClose $box $proc close"
    set f [frame $box.buttons]
    set tf [frame $box.tf]
    set t $tf.t
    set sx $tf.x
    set sy $tf.y
    canvas $t -xscrollcommand "$sx set" -yscrollcommand "$sy set" \
	-bg [Preference General colorbackground]
    bind $box <Button-3> "code::ConfigureGroupMenu %W %X %Y $proc"

    set ConfigureGroup(window) $box
    set ConfigureGroup(canvas) $t
    scrollbar $sx -command "$t xview" -orient h
    scrollbar $sy -command "$t yview" -orient v
    grid $t -in $tf -row 1 -column 0 -sticky nesw
    grid $sx -in $tf -row 2 -column 0 -sticky ew
    grid $sy -in $tf -row 1 -column 1 -sticky ns
    grid columnconfigure $tf 0 -weight 1
    grid rowconfigure $tf 1 -weight 1

    set device 0
    set x 0
    set y 0
    set ConfigureGroup(devices) {}
    set ConfigureGroup(spaces) {}
    set ConfigureGroup(near) {}
    foreach group $value {
        set name [lindex $group 0]
        set flags [lindex $group 1]
        set start [lindex $group 2]
        set end [lindex $group 3]
        set sections [lindex $group 4]
        append result "group $name $flags origin $start maxsize $end-$start;\n"
	ConfigureGroupSetDefaults $proc $device $name $start $end $sections
	set index 0
	set length [llength $flags]
	while {$index < $length} {
	    set flag [lindex $flags $index]
	    switch $flag {
	        ram -
	        rom -
	        eeprom -
	        io {
	            set ConfigureGroup($device,type) $flag
	        }
	        bss {
	            set ConfigureGroup($device,empty) 1
	        }
	        near {
	            set ConfigureGroup(near) $name
	        }
	        itemalign {
	    	    incr index
	            set ConfigureGroup($device,align) [lindex $flags $index]
		}
		fill {
		    incr index
	            set ConfigureGroup($device,fill) 1
	            set ConfigureGroup($device,fillvalue) [lindex $flags $index]
		}
	        window {
		    incr index
	            set ConfigureGroup($device,window) 1
	            set ConfigureGroup($device,windowcount) [lindex $flags $index]
	        }
	        dorigin {
		    incr index
	            set ConfigureGroup($device,dorigin) 1
	            set ConfigureGroup($device,doriginvalue) [lindex $flags $index]
	        }
  	    }
	    incr index
	}
	set nx [ConfigureMakeGroup $t $y $proc $device]
        incr y $ConfigureGroup($device,dy)
        # remember height for later movement
        if {$nx > $x} {
   	    set x $nx
	}
	lappend ConfigureGroup(devices) $device
	incr device
    }
    set ConfigureGroup(maxx) $x
    set ConfigureGroup(nextdevice) $device
    ConfigureGroupSpaces $proc
    $t xview moveto 0
    $t yview moveto 0

    set closeBtn [button $f.ok -text Close -width 6 \
        -command "code::ConfigureGroupClose $box $proc close"]
    help $box $closeBtn "Check changes for correctness, update project, and close window"
    if {[Preference Build [set proc]haschipselects]} {
        set csBtn [button $f.cs -text "Chip Selects" \
            -command "code::ConfigureEditChipSelects"]
        help $box $csBtn "Edit the $proc's chip selects"
    }
    set variablesBtn [button $f.var -text Variables \
        -command "code::ConfigureEditVariables $box"]
    help $box $variablesBtn "Linker variables and additional commands"
    set defaultBtn [menubutton $f.default \
        -textvariable code::ConfigureGroup(map) \
	-indicatoron 1 -relief raised \
	-direction flush]
    $defaultBtn config -menu [code::ConfigureMapMenu $box $proc $defaultBtn]
    help $box $defaultBtn "Reset memory to a pre-defined map"
    grid $closeBtn -in $f -row 0 -column 0 
    if {[Preference Build [set proc]haschipselects]} {
        grid $csBtn -in $f -row 0 -column 1 
    }
    grid columnconfigure $f 3 -weight 1
    grid $variablesBtn -in $f -row 0 -column 4 
    grid columnconfigure $f 5 -weight 1
    grid $defaultBtn -in $f -row 0 -column 6 
    status $box {}
    set l [label $box.l -textvariable code::status($box) \
	    -bd 2 -relief sunken \
	    -anchor w -font [Preference General fontstatus] \
	    -bg [Preference General colorstatusbackground] \
	    ]
    PreferenceWhenChanged General $box "$l config \
	-bg \[Preference General colorstatusbackground] \
        -font \[Preference General fontstatus]"
    set ll [label $box.ll -text \
	{Right click on an area to delete or perform other operations}]
    grid $ll -in $box -row 0 -column 0 -sticky ew -padx 5 -pady 5
    grid $tf -in $box -row 1 -column 0 -sticky nesw
    grid $l -in $box -row 2 -column 0 -sticky ew
    grid $f -in $box -row 3 -column 0 -sticky ew -padx 5 -pady 5
    grid columnconfigure $box 0 -weight 1
    grid rowconfigure $box 1 -weight 1

    if {$geometry != {}} {
        # the window has had its geometry change saved
        setGeometry $box $geometry
    }
    wm deiconify $box
    wm title $box "$proc Memory Map"
    # this code seems to get around a bug in Tk (wm geometry for
    # a hidden window)
    global tcl_platform
    if {$tcl_platform(platform) == "windows" && $geometry != {}} {
        setGeometry $box $geometry
    }

    focus $t
    bind $box <Configure> "code::PreferenceMove $box editmemory"
    bind $box <F1> "code::OpenHelp Applications Project MemoryMap.html"
}

#
# ConfigureMapMenu - put up a menu of valid memory maps
#
proc code::ConfigureMapMenu {w proc button {name menu}} {
    variable menustatus
    set m [menu $button.$name -tearoff 0]
    # build the menu entries
    foreach {name desc map} [ConfigureGetValue Configure memory $proc] {
	$m add radiobutton -label $name \
   	    -variable code::ConfigureGroup(map) \
	    -indicatoron 0 \
	    -command "code::ConfigureEditMemory 1 \"$name\""
	set menustatus($w,$name) $desc
    }
    return $m
}

#
# ConfigureGroupMenu - handle a right button press in the memory window
#
proc code::ConfigureGroupMenu {W X Y proc} {
    variable menustatus
    variable ConfigureGroup

    # parse the widget name to find out where we are
    regexp {\.editmemory([^.]*)(\.[^.]*)(\.[^.]*)\.([0-9]*).*} $W all junk tf t device
    if {![info exists proc] || ![info exists device] || $device == {}} {
	return
    }

    # have a valid processor and device: must be in memory window over a device

    set w $ConfigureGroup(window)
    set m $w.popup
    if {[info commands $m] == ""} {
        menu $m -tearoff 0
    }
    $m delete 0 end
    $m add command -label Expand \
	-command "code::ConfigureGroupExpand $w $proc $device"
    set menustatus($w,Expand) "Expand this memory area to its maximum size" 
    $m add command -label Delete \
	-command "code::ConfigureGroupDelete $w $proc $device"
    set menustatus($w,Delete) "Delete this memory area" 
    tk_popup $m $X $Y
}

#
# ConfigureGroupExpand - expand this area to its maximum size
#
proc code::ConfigureGroupExpand {w proc device} {
    variable ConfigureGroup

    if {![ConfigureGroupCheck $proc]} {
	return
    }
    set start 0x00
    set end [Preference Build [set proc]maxaddress]
    set index [lsearch $ConfigureGroup(devices) $device]
    set dev [lindex $ConfigureGroup(devices) [expr $index - 1]]
    if {$dev != {}} {
	# preceeding device
	set start [expr $ConfigureGroup($dev,end) + 1]
    }
    set dev [lindex $ConfigureGroup(devices) [expr $index + 1]]
    if {$dev != {}} {
	# later device
	set end $ConfigureGroup($dev,start)
    }
    set ConfigureGroup($device,start) $start
    set ConfigureGroup($device,end) [expr $end - 1]
    ConfigureGroupSpaces $proc
    status $w {}
}

#
# ConfigureGroupDelete - delete the given device
#
proc code::ConfigureGroupDelete {w proc device} {
    variable ConfigureGroup
    variable ConfigureContents

    # remove from the device list
    set ConfigureGroup(devices) [lremove $ConfigureGroup(devices) \
	$device]

    # destroy the widget
    destroy $ConfigureGroup($device,widget)
    foreach name [array names ConfigureGroup $device,*] {
	unset ConfigureGroup($name)
    }
    if {![catch {destroy .editcontents$device}]} {
	# a contents window exists
        foreach name [array names ConfigureContents $device,*] {
	    unset ConfigureContents($name)
        }
    }
    ConfigureGroupSpaces $proc
    status $w {}
}

#
# ConfigureGroupClose - handle the closing of a memory map window
#
proc code::ConfigureGroupClose {w proc command} {
    variable ConfigureGroup

    # check changes, show errors and return if bad

    if {![ConfigureGroupCheck $proc]} {
	return
    }

    # set new value of variable

    set all {}
    foreach device $ConfigureGroup(devices) {
	set this [list $ConfigureGroup($device,name)]
	set flags {}
	if {$ConfigureGroup($device,window)} {
	    set flags "window $ConfigureGroup($device,windowcount)"
	}
	if {$ConfigureGroup(near) != {} && $ConfigureGroup(near) == $this} {
	    lappend flags near
	}
	lappend flags $ConfigureGroup($device,type)
	if {$ConfigureGroup($device,empty)} {
	    lappend flags bss
	}
	if {$ConfigureGroup($device,fill)} {
	    lappend flags fill $ConfigureGroup($device,fillvalue)
	}
	if {$ConfigureGroup($device,dorigin)} {
	    lappend flags dorigin $ConfigureGroup($device,doriginvalue)
	}
	if {$ConfigureGroup($device,align) != 1} {
	    lappend flags itemalign $ConfigureGroup($device,align)
	}
	set end $ConfigureGroup($device,end)
        set endlen [string length $end]
        incr endlen -2
        set end [format 0x%0${endlen}X [expr $end + 1]]
	lappend this $flags $ConfigureGroup($device,start) \
	    $end $ConfigureGroup($device,contents)
	lappend all $this
    }

    set orig [Preference System memory]
    PreferenceSetIfChanged System memory $all

    # check for box close
    if {$command == "close"} {
        # close the box with the new value
        foreach name [array names ConfigureGroup *] {
	    unset ConfigureGroup($name)
        }
	destroy $w
    } elseif {$all != $orig} {
	# show the changes in the window
        ConfigureGroupSpaces $proc
    }
}

#
# ConfigureAdrCompareLT - compare two integers as unsigned values for LT
#
proc code::ConfigureAdrCompareLT {val1 val2} {
    if {$val1 < 0} {
	if {$val2 >= 0} {
	    return 0
	}
	# both negative
        return [expr $val1 < $val2]
    }
    if {$val2 < 0} {
	# val2 is negative (bigger)
	return 1
    }
    return [expr $val1 < $val2]
}

#
# ConfigureGroupCheck - check fields for validity in a group
#
proc code::ConfigureGroupCheck {proc} {
    variable ConfigureGroup

    # check syntax of start, end, name, fill, and window fields
    set names {}
    foreach device $ConfigureGroup(devices) {
	if {[catch {expr int($ConfigureGroup($device,start))}]} {
	    ConfigureGroupError $proc $device start "Illegal integer in start address"
	    return 0
	}
        $ConfigureGroup($device,widgetstart) configure -fg black
	if {[catch {expr int($ConfigureGroup($device,end))}]} {
	    ConfigureGroupError $proc $device end "Illegal integer in end address"
	    return 0
	}
        $ConfigureGroup($device,widgetend) configure -fg black
	set name $ConfigureGroup($device,name)
	if {$name == {}} {
	    ConfigureGroupError $proc $device name "No name given for area"
	    return 0
	}
	regexp {[A-Za-z_?.][A-Za-z0-9_$?.: ]*} $name all
	if {![info exists all] || $all != $name} {
	    ConfigureGroupError $proc $device name "$name contains illegal characters"
	    return 0
	}
	if {[lsearch -exact $names $name] != -1} {
	    ConfigureGroupError $proc $device name "$name is used previously"
	    return 0
	}
        $ConfigureGroup($device,widgetname) configure -fg black
	lappend names $name
	if {$ConfigureGroup($device,fill)} {
	    if {[catch {expr int($ConfigureGroup($device,fillvalue))} val]} {
	        ConfigureGroupError $proc $device fillvalue "Illegal integer in fill value"
	        return 0
	    }
	    if {$val < -128 || $val > 255} {
	        ConfigureGroupError $proc $device fillvalue "Fill value must fit in a byte"
	        return 0
	    }
	}
        $ConfigureGroup($device,widgetfillvalue) configure -fg black

	if {$ConfigureGroup($device,window)} {
	    if {[catch {expr int($ConfigureGroup($device,windowcount))} val]} {
	        ConfigureGroupError $proc $device windowcount "Illegal integer in window count"
	        return 0
	    }
	    if {$val < 0 || $val > 4096} {
	        ConfigureGroupError $proc $device windowcount "Window count must be between 0 and 4096"
	        return 0
	    }
            $ConfigureGroup($device,widgetwindowcount) configure -fg black
	}

	if {$ConfigureGroup($device,dorigin)} {
	    if {[catch {expr int($ConfigureGroup($device,doriginvalue))} val]} {
	        ConfigureGroupError $proc $device doriginvalue "Illegal integer in dorigin value"
	        return 0
	    }
	}
        $ConfigureGroup($device,widgetdoriginvalue) configure -fg black
    }

    # make sure devices are in sorted order by start address
    set ConfigureGroup(devices) \
	[lsort -command "code::ConfigureGroupSort $proc" \
	    $ConfigureGroup(devices)]

    # make sure all areas have an end > start and that no areas overlap
    set lastend 0
    set lastdevice 0
    foreach device $ConfigureGroup(devices) {
	if {[ConfigureAdrCompareLT $ConfigureGroup($device,start) $lastend]} {
	    ConfigureGroupError $proc $device start "Start of area must be greater than end of previous area"
	    ConfigureGroupError $proc $lastdevice end "Start of area must be greater than end of previous area"
	    return 0
	}
        $ConfigureGroup($device,widgetstart) configure -fg black
	if {[ConfigureAdrCompareLT $ConfigureGroup($device,end) \
	  $ConfigureGroup($device,start)]} {
	    ConfigureGroupError $proc $device start "End address must be greater than start address"
	    ConfigureGroupError $proc $device end "End address must be greater than start address"
	    return 0
	}
        $ConfigureGroup($device,widgetend) configure -fg black
	set lastdevice $device
	set lastend [expr $ConfigureGroup($device,end) + 1]
    }

    # make sure address space of processor isn't exceeded
    if {$lastend > [Preference Build [set proc]maxaddress]} {
	    ConfigureGroupError $proc $device end "End address is greater than address space of the $proc"
	    return 0
    }
    $ConfigureGroup($device,widgetend) configure -fg black
    # everything is ok, clear error
    set ConfigureGroup(error) 0
    status $ConfigureGroup(window) {}
    return 1
}

#
# ConfigureGroupSort - comparison function for group start addresses
#
proc code::ConfigureGroupSort {proc d1 d2} {
    variable ConfigureGroup

    if {[ConfigureAdrCompareLT $ConfigureGroup($d1,start) \
      $ConfigureGroup($d2,start)]} {
	return -1
    } elseif {$ConfigureGroup($d1,start) - $ConfigureGroup($d2,start)} {
	return 0
    }
    return 1
}

#
# ConfigureGroupError - display an error message and color error
#
proc code::ConfigureGroupError {proc device what msg} {
    variable ConfigureGroup

    $ConfigureGroup($device,widget$what) configure -fg red

    if {$ConfigureGroup(error)} {
	return
    }
    set ConfigureGroup(error) 1
    $ConfigureGroup($device,widget$what) configure -fg red
    tk_messageBox \
        -icon error \
        -message "$msg" \
        -type ok
}

#
# ConfigureMakeGroup - make a group modification entry
#
proc code::ConfigureMakeGroup {t y proc device} {
    variable ConfigureGroup

    set gf [frame $t.$device -borderwidth 2 -relief raised]
    ConfigureGroupInfo $gf $proc $device
    set id [$t create window 0 $y -window $gf -anchor nw]

    set ConfigureGroup($device,id) $id
    set ConfigureGroup($device,widget) $gf
    update idletasks
    set bbox [$t bbox $id]
    set dy [expr [lindex $bbox 3] - [lindex $bbox 1]]
    set ConfigureGroup($device,dy) $dy
    set nx [expr [lindex $bbox 2] - [lindex $bbox 0]]
    return $nx
}

#
# ConfigureGroupSetDefaults - set group default values
#
proc code::ConfigureGroupSetDefaults {proc device name start end sections} {
    variable ConfigureGroup

    set ConfigureGroup($device,name) $name
    set ConfigureGroup($device,start) $start
    set endlen [string length $end]
    incr endlen -2
    set end [format 0x%0${endlen}X [expr $end - 1]]
    set ConfigureGroup($device,end) $end
    set ConfigureGroup($device,type) ram
    set ConfigureGroup($device,empty) 0
    set ConfigureGroup($device,align) 1
    set ConfigureGroup($device,fill) 0
    set ConfigureGroup($device,fillvalue) {}
    set ConfigureGroup($device,dorigin) 0
    set ConfigureGroup($device,doriginvalue) {}
    set ConfigureGroup($device,window) 0
    set ConfigureGroup($device,windowcount) {}
    set ConfigureGroup($device,contents) $sections
}

#
# ConfigureGroupSpaces - add empty regions to memory map canvas
#
proc code::ConfigureGroupSpaces {proc} {
    variable ConfigureGroup

    set c $ConfigureGroup(canvas)

    # delete old spaces
    $c delete spaces
    foreach space $ConfigureGroup(spaces) {
	destroy $space
    }
    set ConfigureGroup(spaces) {}

    set lastend 0
    set y 0
    set hasone 0
    foreach device $ConfigureGroup(devices) {
        if {$lastend != $ConfigureGroup($device,start)} {
	    incr y [ConfigureGroupAddSpace $c $proc $y $lastend \
		$ConfigureGroup($device,start)]
	}
	set hasone 1
	set lastend [expr $ConfigureGroup($device,end) + 1]
	$c coords $ConfigureGroup($device,id) 0 $y
	incr y $ConfigureGroup($device,dy)
    }
    if {(!$hasone || $lastend != 0) && $lastend != [Preference Build [set proc]maxaddress]} {
	incr y [ConfigureGroupAddSpace $c $proc $y $lastend \
	    [Preference Build [set proc]maxaddress]]
    }

    $c config -scrollregion "0 0 $ConfigureGroup(maxx) $y" \
	-height $y -width $ConfigureGroup(maxx)
}

#
# ConfigureGroupAddSpace - add a space filler to a memory map
#
proc code::ConfigureGroupAddSpace {c proc y from to} {
    variable ConfigureGroup

    set sf [frame $c.space$y -borderwidth 2 -relief flat \
	-bg [Preference General colorbackground]]
    lappend ConfigureGroup(spaces) $sf
    set id [$c create window 0 $y -window $sf -anchor nw -tags spaces \
	-width $ConfigureGroup(maxx)]
    set fromlen [string length $to]
    incr fromlen -2
    set from [format 0x%0${fromlen}X $from]
    set l [button $sf.l -text "Empty area: $from - $to" \
	-command "code::ConfigureGroupAdd $c $proc $from $to"]
    help $ConfigureGroup(window) $l "Press here to create a new memory area"
    grid $l -in $sf -row 1 -column 1
    update idletasks
    set bbox [$c bbox $id]
    set dy [expr [lindex $bbox 3] - [lindex $bbox 1]]
    return $dy
}

#
# ConfigureGroupAdd - add a new group to a memory map
#
proc code::ConfigureGroupAdd {c proc from to} {
    variable ConfigureGroup

    set index 0
    set found 0
    foreach device $ConfigureGroup(devices) {
        if {[ConfigureAdrCompareLT $to $ConfigureGroup($device,start)] || \
	  $to == $ConfigureGroup($device,start)} {
	    # add a new device here
	    set found 1
	    break
	}
	incr index
    }
    # insert in list or add at end
    set next $ConfigureGroup(nextdevice)
    ConfigureGroupSetDefaults $proc $next newgroup $from $to {}
    ConfigureMakeGroup $c 0 $proc $next
    set list $ConfigureGroup(devices)
    set ConfigureGroup(devices) \
	[concat [lrange $list 0 [expr $index - 1]] \
	    $next [lrange $list $index end]]
    incr ConfigureGroup(nextdevice)
    ConfigureGroupSpaces $proc
}

#
# ConfigureGroupEditContents - Edit group contents
#
proc code::ConfigureGroupEditContents {w proc device} {
    variable ConfigureGroup 
    variable ConfigureContents

    set box .editcontents$device

    set geometry [Preference Geometry editcontents$device]
    if {[winfo exists $box]} {
        if {$geometry != {}} {
	    # the window has had its geometry change saved
	    setGeometry $box $geometry
	}
        wm deiconify $box
	raise $box
        # this code seems to get around a bug in Tk (wm geometry for
        # a hidden window)
        global tcl_platform
        if {$tcl_platform(platform) == "windows" && $geometry != {}} {
            setGeometry $box $geometry
        }
	return
    }

    toplevel $box
    wm withdraw $box
    wm title $box "$ConfigureGroup($device,name) Contents"
    # get contents of device
    set contents $ConfigureGroup($device,contents)
    # get predefined section names
    set sections [ConfigureGetSections $proc]

    wm protocol $box WM_DELETE_WINDOW "code::ConfigureContentsClose $box $proc $device close"
    set f [frame $box.buttons]
    set tf [frame $box.tf]
    set t $tf.t
    set sx $tf.x
    set sy $tf.y
    canvas $t -xscrollcommand "$sx set" -yscrollcommand "$sy set" \
	-bg [Preference General colorbackground]

    scrollbar $sx -command "$t xview" -orient h
    scrollbar $sy -command "$t yview" -orient v
    grid $t -in $tf -row 1 -column 0 -sticky nesw
    grid $sx -in $tf -row 2 -column 0 -sticky ew
    grid $sy -in $tf -row 1 -column 1 -sticky ns
    grid columnconfigure $tf 0 -weight 1
    grid rowconfigure $tf 1 -weight 1

    set row 0
    set x 0
    set y 0
    set vf [frame $t.v]
    set id [$t create window 0 0 -window $vf]
    set widest 0
    set highest 0
    # create all the names
    set names {}
    set uniques {}
    set id [$t create text 0 2 -text "In use:" -anchor nw]
    $t bind $id <Enter> "code::status $box {Program elements used in this area}"
    $t bind $id <Leave> "code::status $box {}"
    foreach "name unique description value" $sections {
        set id [$t create text $x $y -text $description -tags "$name label all"]
	lappend names $name
	lappend uniques $unique
	set ConfigureContents($device,x,$name) $x
	set ConfigureContents($device,y,$name) $y
	set ConfigureContents($device,id,$name) $id
        update idletasks
        set bbox [$t bbox $id]
        set width [expr [lindex $bbox 2] - [lindex $bbox 0]]
        set height [expr [lindex $bbox 3] - [lindex $bbox 1]]
	incr y $height
	if {$width > $widest} {
	    set widest $width
	}
	if {$height > $highest} {
	    set highest $height
	}
    }

    # create rectangles
    set x 80
    set y 2
    incr widest 4
    incr highest 4
    set ConfigureContents($device,yoffsets) {}
    set ConfigureContents($device,highest) $highest
    foreach name $names {
        set id [$t create rectangle \
	    $x $y [expr $widest + $x + 1] [expr $highest + $y] \
	    -tags "$name rect all"]
	set ConfigureContents($device,y,$name) [expr $y - 1]
	lappend ConfigureContents($device,yoffsets) [expr $y - 1]
        update idletasks
        set bbox [$t bbox $id]
        set width [expr [lindex $bbox 2] - [lindex $bbox 0]]
        set height [expr [lindex $bbox 3] - [lindex $bbox 1]]
	set centerx [expr ($x + 1) + ($width / 2)]
	set centery [expr [lindex $bbox 1] + ($height / 2)]
	$t coords $ConfigureContents($device,id,$name) $centerx $centery
	lappend ConfigureContents($device,id,$name) $id
	incr y $height
	incr y 2
    }
    incr widest [expr $x + 10]
    set ConfigureContents($device,midpoint) [expr $y + 5]
    $t create line 0 $y $widest $y
    set id [$t create text 2 [expr $y + 2] -text "Not in use:" -anchor nw]
    $t bind $id <Enter> "code::status $box {Program elements not in this area}"
    $t bind $id <Leave> "code::status $box {}"
    set y [expr ($y * 2) + 10]

    # raise all the labels
    $t raise label

    # find which are used, unused
    set ConfigureContents($device,used) {}
    set ConfigureContents($device,unused) {}
    set ordered {}
    foreach name $names unique $uniques {
	if {[regexp -indices .*\\$\{($name)\}.* $ConfigureGroup($device,contents) all index]} {
	    set ConfigureContents($device,name,$name) 1
	    set ConfigureContents($device,orig,$name) 1
	    lappend ordered $index

	} else {
	    set ConfigureContents($device,name,$name) 0
	    set ConfigureContents($device,orig,$name) 0
	    lappend ConfigureContents($device,unused) $name
	}
	set id [lindex $ConfigureContents($device,id,$name) 1]
	$t bind $name <Leave> \
	     "$t itemconfigure $id -fill \[code::Preference General colorbackground]; \
              code::status $box {}"
        $t bind $name <1> "code::ConfigureContentsDown $device $name $t %x %y"
        $t bind $name <ButtonRelease-1> "code::ConfigureContentsRelease $device $name $t $proc $box"
        $t bind $name <B1-Motion> "code::ConfigureContentsMove $device $name $t %x %y"

	if {$unique} {
	    set rest ", only one allowed in memory map"
	} else {
	    set rest ", any number allowed in memory map"
	}
	$t bind $name <Enter> "$t itemconfigure $id -fill \[code::Preference General colorselection]; \
            if \{\$code::ConfigureContents($device,name,$name)\} \{ \
   	        code::status $box \"Drag below to remove$rest\" \
	    \} else \{ \
	        code::status $box \"Drag above to add$rest\" \
	    \}"
    }

    if {$ordered != {}} {
        # sort the contents into memory order
        set ordered [lsort -integer -index 0 $ordered]
        foreach index $ordered {
            set name [eval string range \{$ConfigureGroup($device,contents)\} $index]
            lappend ConfigureContents($device,used) $name
        }
    }
    # remember starting order
    set ConfigureContents($device,usedorig) $ConfigureContents($device,used)

    # show the members in their proper places
    ConfigureContentsRedraw $t $device
    set ConfigureContents($device,end) $y
    $t config -scrollregion "0 0 $widest $y" -width $widest -height $y
    $t xview moveto 0
    $t yview moveto 0

    set closeBtn [button $f.ok -text Close -width 6 \
        -command "code::ConfigureContentsClose $box $proc $device close"]
    help $box $closeBtn "Update project and close window"
    grid $closeBtn -in $f -row 0 -column 0 
    grid columnconfigure $f 3 -weight 1
    status $box {}
    set l [label $box.l -textvariable code::status($box) \
	    -bd 2 -relief sunken \
	    -anchor w -font [Preference General fontstatus] \
	    -bg [Preference General colorstatusbackground] \
	    ]
    PreferenceWhenChanged General $box "$l config \
	-bg \[Preference General colorstatusbackground] \
        -font \[Preference General fontstatus]"
    grid $tf -in $box -row 1 -column 0 -sticky nesw
    grid $l -in $box -row 2 -column 0 -sticky ew
    grid $f -in $box -row 3 -column 0 -sticky ew -padx 5 -pady 5
    grid columnconfigure $box 0 -weight 1
    grid rowconfigure $box 1 -weight 1

    if {$geometry != {}} {
        setGeometry $box $geometry
    }
    wm deiconify $box
    raise $box
    # this code seems to get around a bug in Tk (wm geometry for
    # a hidden window)
    global tcl_platform
    if {$tcl_platform(platform) == "windows" && $geometry != {}} {
        setGeometry $box $geometry
    }
    bind $box <Configure> "code::PreferenceMove $box editcontents$device"
    bind $box <F1> "code::OpenHelp Applications Project MemoryMapContents.html"
}

#
# ConfigureContentsDown - the mouse has been pressed over a member
#
proc code::ConfigureContentsDown {device name w x y} {
    variable ConfigureContents

    $w dtag selected
    $w addtag selected withtag $name
    $w raise $name
}

#
# ConfigureContentsMove - the mouse is moving
#
proc code::ConfigureContentsMove {device name w x y} {
    variable ConfigureContents

    # allow vertical movement only and restrict to canvas
    if {$y < 0} {
	set y 0
    }
    if {$y > $ConfigureContents($device,end)} {
	set y $ConfigureContents($device,end)
    }
    # compute the first visible part of the canvas and adjust y
    set reg [$w cget -scrollregion]
    set y [expr {$y + (([lindex $reg 3] - [lindex $reg 1]) * [lindex [$w yview] 0])}]
    $w move selected 0 [expr $y - $ConfigureContents($device,y,$name)]
    set ConfigureContents($device,y,$name) $y
}
#
# ConfigureContentsRelease - the mouse has released the item
#
proc code::ConfigureContentsRelease {device name w proc box} {
    variable ConfigureContents

    $w dtag selected

    # remove the tag from it's current list
    if {$ConfigureContents($device,name,$name)} {
        set ConfigureContents($device,used) \
	    [lremove $ConfigureContents($device,used) $name]
    } else {
        set ConfigureContents($device,unused) \
	    [lremove $ConfigureContents($device,unused) $name]
    }

    # find where we have moved to
    set y $ConfigureContents($device,y,$name)
    if {$y > $ConfigureContents($device,midpoint)} {
	# past the midpoint
	set used 0
	set y [expr $y - $ConfigureContents($device,midpoint)]
    } else {
	set used 1
    }
    set index 0
    # get the height of one entry
    set height $ConfigureContents($device,highest)
    # add some wiggle
    incr height 4
    foreach offset $ConfigureContents($device,yoffsets) {
	if {$y >= $offset && $y <= [expr $offset + $height]} {
	    # in here
	    break
	}
	incr index
    }
    set ConfigureContents($device,name,$name) $used
    if {$used} {
	set ConfigureContents($device,used) \
	    [linsert $ConfigureContents($device,used) $index $name]
    } else {
	set ConfigureContents($device,unused) \
	    [linsert $ConfigureContents($device,unused) $index $name]
    }
    ConfigureContentsRedraw $w $device
    ConfigureContentsClose $box $proc $device apply
}

#
# ConfigureContentsRedraw - put the contents in their proper places
#
proc code::ConfigureContentsRedraw {t device} {
    variable ConfigureContents

    set index 0
    foreach name $ConfigureContents($device,used) {

	set oldy $ConfigureContents($device,y,$name)
	set newy [lindex $ConfigureContents($device,yoffsets) $index]
	$t move $name 0 [expr $newy - $oldy]
	set ConfigureContents($device,y,$name) $newy
	incr index
    }
    set index 0
    foreach name $ConfigureContents($device,unused) {

	set oldy $ConfigureContents($device,y,$name)
	set newy [lindex $ConfigureContents($device,yoffsets) $index]
	incr newy $ConfigureContents($device,midpoint)
	$t move $name 0 [expr $newy - $oldy]
	set ConfigureContents($device,y,$name) $newy
	incr index
    }
}

#
# ConfigureContentsClose - update the contents
#
proc code::ConfigureContentsClose {w proc device command} {
    variable ConfigureContents
    variable ConfigureGroup

    # set new contents value, if any
    set changed 0
    set names [array names ConfigureContents $device,name,*]
    if {$ConfigureContents($device,used) != $ConfigureContents($device,usedorig)} {
	set ConfigureGroup($device,contents) {}
        foreach name $ConfigureContents($device,used) {
	    append ConfigureGroup($device,contents) "\$\{$name\}\n"
	}
	# apply changes
        ConfigureGroupClose {} $proc apply
        set ConfigureContents($device,usedorig) $ConfigureContents($device,used)
    }

    # check for box close
    if {$command == "close"} {
        # close the box with the new value
        foreach name [array names ConfigureContents $device,*] {
	    unset ConfigureContents($name)
        }
	destroy $w
    } 
}

#
# ConfigureGroupInfo - generate a group information window into a frame
#
proc code::ConfigureGroupInfo {base proc device} {
    variable ConfigureGroup

    set w $ConfigureGroup(window)

    frame $base.typeframe 
    frame $base.miscframe
    frame $base.alignframe
    frame $base.addressframe
    frame $base.startframe
    frame $base.endframe
    frame $base.fillframe
    frame $base.doriginframe

    entry $base.start -textvariable code::ConfigureGroup($device,start) \
	-width 10

    bind $base.start <Leave> "code::ConfigureGroupClose $w $proc apply"
    bind $base.start <Return> "code::ConfigureGroupClose $w $proc apply"
    set ConfigureGroup($device,widgetstart) $base.start
    help $w $base.start {The memory area start address}
    entry $base.name -textvariable code::ConfigureGroup($device,name) \
	-width 30
    bind $base.name <Leave> "code::ConfigureGroupClose $w $proc apply"
    bind $base.name <Return> "code::ConfigureGroupClose $w $proc apply"
    set ConfigureGroup($device,widgetname) $base.name
    help $w $base.name {The memory area name (must be unique)}
    entry $base.end -textvariable code::ConfigureGroup($device,end) \
	-width 10
    bind $base.end <Leave> "code::ConfigureGroupClose $w $proc apply"
    bind $base.end <Return> "code::ConfigureGroupClose $w $proc apply"
    set ConfigureGroup($device,widgetend) $base.end
    help $w $base.end {The address of the end of the memory area}

    checkbutton $base.dorigin -text Dorigin: \
	-variable code::ConfigureGroup($device,dorigin) \
        -command "code::ConfigureGroupValue $proc $device dorigin doriginvalue"
    help $w $base.dorigin {Set a device origin for this area}
    entry $base.doriginvalue -textvariable code::ConfigureGroup($device,doriginvalue) \
	-width 10
    bind $base.doriginvalue <Leave> "code::ConfigureGroupClose $w $proc apply"
    bind $base.doriginvalue <Return> "code::ConfigureGroupClose $w $proc apply"
    set ConfigureGroup($device,widgetdoriginvalue) $base.doriginvalue
    help $w $base.doriginvalue {The device origin for this area}

    label $base.type -text Type:
    help $w $base.type {The type of this memory area}
    menubutton $base.typemenu -indicatoron 1 -menu $base.typemenu.p \
	-direction flush -textvariable code::ConfigureGroup($device,typename) \
	-relief raised -width 8
    help $w $base.typemenu {The type of this memory area}
    set m [menu $base.typemenu.p -tearoff 0]
    foreach "name value" {RAM ram ROM rom EEPROM eeprom I/O io} {
        $m add radiobutton -label $name\
            -variable code::ConfigureGroup($device,type) \
	    -value $value \
	    -command "set code::ConfigureGroup($device,typename) $name; \
    		code::ConfigureGroupClose $w $proc apply"
	if {$value == $ConfigureGroup($device,type)} {
	    set ConfigureGroup($device,typename) $name
	}
    }

    checkbutton $base.empty -text Empty \
	-variable code::ConfigureGroup($device,empty) \
        -command "code::ConfigureGroupClose $w $proc apply"
    help $w $base.empty {This area has no information in the object file (i.e. bss)}
    checkbutton $base.fill -text Fill: \
	-variable code::ConfigureGroup($device,fill) \
        -command "code::ConfigureGroupValue $proc $device fill fillvalue"
    help $w $base.fill {Fill unused space in this area}
    entry $base.fillvalue -textvariable code::ConfigureGroup($device,fillvalue) \
	-width 3
    bind $base.fillvalue <Leave> "code::ConfigureGroupClose $w $proc apply"
    bind $base.fillvalue <Return> "code::ConfigureGroupClose $w $proc apply"
    set ConfigureGroup($device,widgetfillvalue) $base.fillvalue
    help $w $base.fillvalue {The byte value to fill unused space with}
    if {[Preference Build [set proc]canwindow]} {
	# this processor can use windows
        frame $base.windowframe
        checkbutton $base.window -text Windows: \
	    -variable code::ConfigureGroup($device,window) \
	    -command "code::ConfigureGroupValue $proc $device window windowcount"
	entry $base.windowcount -width 3 -textvariable code::ConfigureGroup($device,windowcount)
        bind $base.windowcount <Leave> "code::ConfigureGroupClose $w $proc apply"
        bind $base.windowcount <Return> "code::ConfigureGroupClose $w $proc apply"
        set ConfigureGroup($device,widgetwindowcount) $base.windowcount
        help $w $base.window {Make this area a window for bank switching}
        help $w $base.windowcount {The number of banks in this window}
    }
    
    if {[Preference Build [set proc]hasnear]} {
	# this processor needs a near definition
        radiobutton $base.near -text Near \
	    -value $code::ConfigureGroup($device,name) \
	    -variable code::ConfigureGroup(near) \
            -command "code::ConfigureGroupValue $proc $device fill fillvalue"
        help $w $base.near {Make this area the near bank}
    }
    button $base.contents -text Contents \
	-command "code::ConfigureGroupEditContents $base $proc $device"
    help $w $base.contents {Configure the contents (sections) in this area}
    label $base.alignment -text Alignment:
    help $w $base.alignment {The alignment between parts in this area}
    menubutton $base.alignmenu -indicatoron 1 -menu $base.alignmenu.p \
	-direction flush -textvariable code::ConfigureGroup($device,align) \
	-relief raised
    help $w $base.alignmenu {The alignment between parts in this area}
    set m [menu $base.alignmenu.p -tearoff 0]
    foreach number {1 2 4 8} {
        $m add radiobutton -label $number\
            -variable code::ConfigureGroup($device,align) \
	    -value $number \
            -command "code::ConfigureGroupValue $proc $device fill fillvalue"
    }

    # Geometry management
    grid $base.addressframe -in $base -row 1 -column 1  -rowspan 3 -sticky nesw
    grid $base.startframe -in $base.addressframe -row 1 -column 1  -sticky n
    grid $base.start -in $base.startframe -row 1 -column 1 
    grid rowconfigure $base.addressframe 2 -weight 1
    grid $base.endframe -in $base.addressframe	-row 3 -column 1  -sticky s
    grid $base.end -in $base.endframe	-row 1 -column 1 

    grid $base.name -in $base -row 1 -column 2 -sticky ew
    grid $base.doriginframe -in $base -row 1 -column 3 -sticky nesw
    grid $base.dorigin -in $base.doriginframe -row 1 -column 1 
    grid $base.doriginvalue -in $base.doriginframe -row 1 -column 2 -sticky w
    grid $base.alignframe -in $base -row 1 -column 5 -sticky nesw
    grid $base.alignment -in $base.alignframe -row 1 -column 1 
    grid $base.alignmenu -in $base.alignframe -row 1 -column 2  -sticky w
    grid $base.miscframe -in $base -row 2 -column 2 -sticky ew -columnspan 5
    grid $base.typeframe -in $base.miscframe -row 1 -column 0 -sticky nesw
    grid $base.type -in $base.typeframe	-row 1 -column 1 
    grid $base.typemenu -in $base.typeframe -row 1 -column 2 -sticky w
    grid $base.empty -in $base.miscframe -row 1 -column 1 -sticky w
    grid $base.fillframe -in $base.miscframe -row 1 -column 2 -sticky w
    grid $base.fill -in $base.fillframe	-row 1 -column 1 -sticky w
    grid $base.fillvalue -in $base.fillframe -row 1 -column 2 
    if {[Preference Build [set proc]canwindow]} {
        grid $base.windowframe -in $base.miscframe -row 1 -column 3 -sticky w
        grid $base.window -in $base.windowframe	-row 1 -column 1 -sticky w
        grid $base.windowcount -in $base.windowframe -row 1 -column 2 
    }
    if {[Preference Build [set proc]hasnear]} {
        grid $base.near -in $base.miscframe -row 1 -column 4 -sticky w
    }
    grid $base.contents -in $base.miscframe -row 1 -column 5 -sticky e
    grid columnconfigure $base.miscframe 1 -weight 1
    grid columnconfigure $base.miscframe 2 -weight 1
    grid columnconfigure $base.miscframe 3 -weight 1
    grid columnconfigure $base.miscframe 4 -weight 1
    grid columnconfigure $base.miscframe 5 -weight 1
}

#
# ConfigureGroupValue - if a value is deselected, make it blank
#
proc code::ConfigureGroupValue {proc device bool value} {
    variable ConfigureGroup

    if {!$ConfigureGroup($device,$bool)} {
	set ConfigureGroup($device,$value) {}
    }
    ConfigureGroupClose $ConfigureGroup(window) $proc apply
}

#
# ConfigureFilesDown - the mouse has been pressed over a file
#
proc code::ConfigureFilesDown {name w x y} {
    variable FilesContents

    $w dtag selected
    $w addtag selected withtag $name
    $w raise $name
    set FilesContents(y0,$name) $FilesContents(y,$name)
}

#
# ConfigureFilesRight - the right mouse has been pressed over a file
#
proc code::ConfigureFilesRight {name c x y X Y} {
    variable FilesContents
    variable CODEdocument
    variable CODEcurrentfiles

    $c dtag selected
    $c addtag selected withtag $name
    $c raise $name

    set m $c.right
    if {[info commands $m] == ""} {
        menu $m -tearoff 0
    }
    $m delete 0 end
    if {[catch {file stat $name stat}]} {
	# file is inaccessable for some reason
	set stat(type) file
    }
    set extension [string tolower [file extension $name]]
    set type $stat(type)
    set Type [Preference FileTypes $type$extension]
    if {$Type == {}} {
        set Type [Preference FileTypes $type]
        if {$Type == {}} {
            error "File type botch."
        }
    }
    set cmds [lindex $Type 2]
    set CODEcurrentfiles [list $name]
    ToolGenerate popup $cmds $CODEdocument . $x $y $X $Y
}

#
# ConfigureFilesMove - the mouse is moving with a file
#
proc code::ConfigureFilesMove {name w x y} {
    variable FilesContents

    # allow vertical movement only and restrict to canvas
    if {$y < $FilesContents(start)} {
	set y $FilesContents(start)
    }
    if {$y > $FilesContents(end)} {
	set y $FilesContents(end)
    }
    # compute the first visible part of the canvas and adjust y
    set reg [$w cget -scrollregion]
    set y [expr {$y + (([lindex $reg 3] - [lindex $reg 1]) * [lindex [$w yview] 0])}]
    $w move selected 0 [expr {$y - $FilesContents(y,$name)}]
    set FilesContents(y,$name) $y
}

#
# ConfigureFilesRelease - the mouse has released the file
#
proc code::ConfigureFilesRelease {name w box} {
    variable FilesContents

    $w dtag selected

    # find where we have moved to
    set y $FilesContents(y,$name)
    if {$y == $FilesContents(y0,$name)} {
	# not moved
	return
    }
    # get the height of one entry
    set height $FilesContents(highest)
    # add some wiggle
    incr height 4
    set index 0
    foreach offset $FilesContents(yoffsets) {
	if {$y >= $offset && $y <= [expr $offset + $height]} {
	    # in here
	    break
	}
	incr index
    }

    # remove the tag from the current list
    set FilesContents(names) [lremove $FilesContents(names) $name]

    set FilesContents(names) [linsert $FilesContents(names) $index $name]
    ConfigureFilesRedraw $w
    ConfigureFilesUpdate $box
}

#
# ConfigureFilesRedraw - put the files in their proper places
#
proc code::ConfigureFilesRedraw {t} {
    variable FilesContents

    set index 0
    foreach name $FilesContents(names) {
	set oldy $FilesContents(y,$name)
	set newy [lindex $FilesContents(yoffsets) $index]
	$t move $name 0 [expr $newy - $oldy]
	set FilesContents(y,$name) $newy
	incr index
    }
}

#
# ConfigureFilesUpdate - update the file contents
#
proc code::ConfigureFilesUpdate {w} {
    variable FilesContents

    # set new contents value, if any
    if {$FilesContents(names) != $FilesContents(namesorig)} {
	# list order has changed
        set FilesContents(namesorig) $FilesContents(names)
	set files [Preference Build files]
	set newfiles {}
	foreach name $FilesContents(names) {
	    # put the files in this order
	    set index 0
	    foreach file $files {
		if {[string equal $name [lindex $file 0]]} {
		    # matches, add to new list
		    lappend newfiles $file
		    # remove from old list
		    set files [lreplace $files $index $index]
		    break
		}
		incr index
	    }
	}
	Preference Build files $newfiles
    }
}

#
# FILESstartup - return the command to open the files window
#
proc code::FILESstartup {doc} {
    return "REGISTERS"
}

#
# FILES - create the Files window
#
proc code::FILES {args} {
    set type "Files"
    set name ""
    set w [CODEfind $type $name]
    if {$w != {}} {
	# have a command prompt, raise and return
	if {   [$w cget -state] == "minimized" 
	    || [$w cget -state] == "withdrawn"} {
	    $w configure -state normal
	}
	$w raise
	return $w
    }
    set w [eval document .work.projectfiles -type \{$type\} \
	-raiseproc code::DOCraise \
	-startupproc code::FILESstartup $args]
    # redo files if a new file is loaded
    bind $w <<State>> "if {\$code::Debugger(newfile)} \"code::FILESsetup $w\"; break"
    FILESsetup $w 1
    return $w
}

#
# FILESsetup - show all the files in the project list
#
proc code::FILESsetup {w {force 0}} {
    variable FilesContents
    variable FileIcons

    # build a list of file names
    set names {}
    set list [Preference Build files]
    foreach file $list {
        set name [lindex $file 0]
        lappend names $name
    }
    if {[catch {DebuggerDbg files} plist]} {
	set plist {}
    } else {
	# find the names of these files
	set newlist {}
	foreach file $plist {
	    if {![file exists $file]} {
		if {[catch {DebuggerDbg path -find $file} file]} {
		    # could not find
		    continue
		}
	    }
        lappend newlist $file
	}
	set plist $newlist
    }

    set extralist [concat $plist [Build $w projectfiles]]
    set extranames {}
    foreach one $extralist {
	set index [lsearch $names $one]
	if {$index == -1} {
	    # this is not project file
	    lappend extranames $one
	}
    }
    set extranames [MainMenuList $extranames]
    if {!$force} {
	# we have already created the window for this project
        if {   [info exists FilesContents(names)] && $FilesContents(names) == $names
	    && [info exists FilesContents(extranames)] && $FilesContents(extranames) == $extranames} {
	    # unchanged
	    return
        } 
    }

    set c $w.c
    if {![winfo exists $c]} {
        scrollcanvas $c -scrollbar off
        $w pack $c -fill both -expand 1
    }
    set image $FileIcons(codefile.icon)
    $w configure -image $image -icontext Files -title  "Project Files"
    # delete any pre-existing files
    $c delete all
    $c xview moveto 0
    $c yview moveto 0
    set height 0
    set widest 0
    set highest 0
    foreach name [concat $names $extranames] {
	if {$name == {}} {
	    continue
	}
	set realname [BuildFileName $name] 
        if {[string match {${PROJECT}/*} $realname]} {
	    # we remove the project prefix, for readability
	    regexp {\$\{PROJECT\}/(.*)} $realname all realname
        } 
        if {[string match {${INTROL}/*} $realname]} {
	    # we remove the INTROL prefix, for readability
	    regexp {\$\{INTROL\}/(.*)} $realname all realname
        } 
        set realname [file nativename [BuildSubstitute $realname]]
        set id [$c create text 10000 10000 -text $realname -tags "$name name all" -anchor nw]
        set FilesContents(id,$name) $id
        update idletasks
        set bbox [$c bbox $id]
        set width [expr [lindex $bbox 2] - [lindex $bbox 0]]
        set height [expr [lindex $bbox 3] - [lindex $bbox 1]] 
        if {$width > $widest} {
            set widest $width
        }
        if {$height > $highest} {
            set highest $height
        }
    }

    # create boxes
    set x 4
    set id [$c create text 0 2 -text "The source files in this project, in the order they are linked. Click to move, double click to open, right click to perform other operations." -anchor nw]
    set FilesContents(start) 20
    set y $FilesContents(start)
    incr widest 4
    incr highest 4
    set FilesContents(yoffsets) {}
    set FilesContents(highest) $highest
    foreach name $names {
        set id [$c create rectangle \
                $x $y [expr $widest + $x + 1] [expr $highest + $y] \
                -tags "$name rect all"]
        set FilesContents(y,$name) [expr $y - 1]
        lappend FilesContents(yoffsets) [expr $y - 1]
        update idletasks
        set bbox [$c bbox $id]
        set width [expr [lindex $bbox 2] - [lindex $bbox 0]]
        set height [expr [lindex $bbox 3] - [lindex $bbox 1]]
        set centerx [expr ($x + 3)]
        set centery [expr [lindex $bbox 1] + 3]
        $c coords $FilesContents(id,$name) $centerx $centery
        lappend FilesContents(id,$name) $id
        $c bind $name <Leave> \
             "$c itemconfigure $id -fill \[code::Preference General colorbackground]; \
              code::status $w {}"
        $c bind $name <1> "code::ConfigureFilesDown \{$name\} $c %x %y"
        $c bind $name <Double-1> "code::OpenFile \{$name\}"
        $c bind $name <ButtonRelease-1> "code::ConfigureFilesRelease \{$name\} $c $w"
        $c bind $name <B1-Motion> "code::ConfigureFilesMove \{$name\} $c %x %y"
        $c bind $name <Enter> "$c itemconfigure $id -fill \[code::Preference General colorselection]"    
	$c bind $name <Button-3> "code::ConfigureFilesRight \{$name\} $c %x %y %X %Y"
        incr y $height
        incr y 2
    }

    set FilesContents(end) $y
    incr y 4
    if {$extranames != {}} {
        set id [$c create text 0 $y -text "Other files used in this project. Double click to open. Right click to perform file operations." -anchor nw]
    }
    incr y 20
    foreach name $extranames {
	if {$name == {}} {
	    continue
	}
        set id [$c create rectangle \
                $x $y [expr $widest + $x + 1] [expr $highest + $y] \
                -tags "$name rect all"]
        set FilesContents(y,$name) [expr $y - 1]
        lappend FilesContents(yoffsets) [expr $y - 1]
        update idletasks
        set bbox [$c bbox $id]
        set width [expr [lindex $bbox 2] - [lindex $bbox 0]]
        set height [expr [lindex $bbox 3] - [lindex $bbox 1]]
        set centerx [expr ($x + 3)]
        set centery [expr [lindex $bbox 1] + 3]
        $c coords $FilesContents(id,$name) $centerx $centery
        lappend FilesContents(id,$name) $id
        $c bind $name <Double-1> "code::OpenFile \{$name\}"
        $c bind $name <Leave> \
             "$c itemconfigure $id -fill \[code::Preference General colorbackground]; \
              code::status $w {}"
        $c bind $name <Enter> "$c itemconfigure $id -fill \[code::Preference General colorselection]"    
	$c bind $name <Button-3> "code::ConfigureFilesRight \{$name\} $c %x %y %X %Y"
        incr y $height
        incr y 2
    }
    $c raise name
    set FilesContents(names) $names
    set FilesContents(extranames) $extranames
    set FilesContents(namesorig) $names
    set FilesContents(height) [expr $height + 2]
    incr widest [expr $x + 10]
    update idletasks
    $c config -scrollregion [$c bbox all]
    $c xview moveto 0
    $c yview moveto 0
    set FilesContents(top) $w
    set FilesContents(canvas) $c
    PreferenceWhenChanged Build $w "code::FILESsetup $w"
    $c configure -scrollbar auto
}
