#
#	Ddf - process Device Definition Files (.ddf)
#

#
# DdfParse - parse a .ddf file
#
# Returns prefix in "theprefix", variants in "thevariants", modifier
# in "themodifier".
# Returns defined registers in "thedefines", Additional info in "theai".
#
proc code::DdfParse {ddf theprefix thevariants themodifier thedefines theai} {
    upvar $theprefix prefix
    upvar $thevariants variants
    upvar $themodifier modifier
    upvar $thedefines defines
    upvar $theai ai

    # initialize output
    set prefix {}
    set variants {}
    set modifier {}
    set defines {}
    array set ai {}

    set ifile [open $ddf r]
    set indefines 1
    set lastblank 0
    while {$indefines && [gets $ifile line] != -1} {
	# get rid of leading, trailing whitespace
	set line [string trim $line]
	set end [string wordend $line 0]
	if {[string index $line $end] == "."} {
	    incr end 2
	}
	set command [string range $line 0 [expr $end -1]]
	set line [string trim [string range $line $end end]]
	switch -exact $command {
	    "" {
		# blank line, return only one
		if {!$lastblank} {
	            lappend defines [list {}]
		    set lastblank 1
		}
	    }
	    ; -
	    * {
		# output a comment
	        lappend defines [list comment $line]
		set lastblank 0
	    }
	    end {
		# end of defines
		set indefines 0
		set lastblank 0
	    }
	    defines {
		# The devices this ddf defines, get the list

		while {1} {
	            set end [string wordend $line 0]
	            set command [string range $line 0 [expr $end -1]]
	            set line [string trim [string range $line $end end]]
		    if {$command == "" || $command == ";" || $command == "*" } {
			break;
		    }
		    if {$command == ","} {
			continue;
		    }
		    lappend variants $command
		}
		set lastblank 0
	    }
	    prefix {
		# get the prefix and near/far (if any)
	        set end [string wordend $line 0]
	        set prefix [string range $line 0 [expr $end -1]]
	        set line [string trim [string range $line $end end]]
		if {[string index $line 0] == ","} {
		    # process a near or far
	            set end [string wordend $line 1]
	            set command [string range $line 1 [expr $end -1]]
		    if {$command == "near"} {
			set modifier "near"
		    } elseif {$command == "far"} {
			set modifier "far"
		    }
		}
		set lastblank 0
	    }
	    space.l -
	    space.b -
	    space.w {
		# Handle an array definition

		set arg 0
		set defline ""
		set comment ""
		while {1} {
	            set end [string wordend $line 0]
	            set reg [string range $line 0 [expr $end -1]]
	            set line [string trim [string range $line $end end]]
		    if {$reg == "" || $reg == ";" || $reg == "*" } {
			break;
		    }
		    if {$reg == ","} {
			continue;
		    }
		    incr arg
		    if {$arg == 1} {
			set size $reg
		    } elseif {$arg == 2} {
		        # reg contains the register name
			set name $reg
		    }
	            set ch [string index $line 0]
	            if {$ch == ";" || $ch == "*"} {
	                set comment [string range $line 1 end]
	                break;
	            }
	        }
	        lappend defines [list $command $size $name $comment]
		set lastblank 0
	    }
	    long -
	    label.l -
	    word -
	    label.w -
	    byte -
	    pbyte -
	    label.b {
		# Handle a definition

		set arg 0
		set defline ""
		set initial {}
		set hex 0
		set comment ""
		while {1} {
	            set end [string wordend $line 0]
	            set reg [string range $line 0 [expr $end -1]]
	            set line [string trim [string range $line $end end]]
		    if {$reg == "" || $reg == ";" || $reg == "*" } {
			break;
		    }
		    if {$reg == ","} {
			continue
		    }
		    if {$reg == "$"} {
			set hex 1
			continue
		    }
		    incr arg
		    if {$arg == 1} {
		        # reg contains the register name
			set name $reg
		    } elseif {$arg == 2} {
		        # reg contains the initial value
			set initial $reg
			if {$initial != {}} {
			    # make into a standard format
			    if {$hex} {
			        set initial [format "0x%02X" 0x$initial]
			    } else {
				# decimal
			        set initial [format "0x%02X" $initial]
			    }
			}
			set hex 0
		    }
	            set ch [string index $line 0]
	            if {$ch == ";" || $ch == "*"} {
	                set comment [string range $line 1 end]
	                break;
	            }
	        }
	        lappend defines [list $command $initial $name $comment]
		set lastblank 0
	    }
	}
    }

    # get User interface definitions

    while {[gets $ifile line] != -1} {
	# get rid of leading, trailing whitespace
	set line [string trim $line]
	set end [string wordend $line 0]
	set command [string range $line 0 [expr $end -1]]
	set line [string trim [string range $line $end end]]
	switch -exact $command {
	    "" -
	    # -
	    ; -
	    * {
		# a comment or blank line
		continue
	    }
	    default {
		# get a name/value pair
		# command is the name, the value is the next word/list
		while {![info complete $line]} {
		    if {[gets $ifile next] == -1} {
			error "Syntax error in $ddf"
		    }

		    append line "\n$next"
		}
		eval set ai($command) $line
	    }
	}
    }
    close $ifile
}

#
# DdfMakeHeader - process a ddf file and return the C header
#
proc code::DdfMakeHeader {ddf thefiles} {
    upvar $thefiles files

    # parse the .ddf file
    DdfParse $ddf prefix variants modifier defines ai

    # return a list of header file names
    set files {}
    foreach variant $variants {
        lappend files "$variant.h"
    }

    # adjust the modifier for C
    if {$modifier == "near"} {
	set modifier "__base"
    } elseif {$modifier == "far"} {
	set modifier "__far"
    }

    set file "// Definitions of registers
#undef __WORDreg__
#undef __BYTEreg__
#define __WORDreg__ extern volatile $modifier unsigned short
#define __BYTEreg__ extern volatile $modifier unsigned char
"
    if {[Preference Build [Preference Build processor]setlong] != {}} {
	append file "#define __LONGreg__ extern volatile $modifier unsigned long\n"
    }

    append file "\n"

    # output the definitions
    foreach element $defines {
	set command [lindex $element 0]
	switch -exact $command {
	    "" {
		# blank line
	        append file "\n"
	    }
	    comment {
		# comment
		append file "// [lindex $element 1]\n"
	    }
	    space.b {
		# Handle a byte array definition

		set size [lindex $element 1]
		set name [lindex $element 2]
		set comment [lindex $element 3]

	        append file "__BYTEreg__ $name\[$size\];\t\t//$comment\n"
	    }
	    space.w {
		# Handle a word array definition

		set size [lindex $element 1]
		set name [lindex $element 2]
		set comment [lindex $element 3]

	        append file "__WORDreg__ $name\[$size\];\t\t//$comment\n"
	    }
	    space.l {
		# Handle a long array definition

		set size [lindex $element 1]
		set name [lindex $element 2]
		set comment [lindex $element 3]

	        append file "__LONGreg__ $name\[$size\];\t\t//$comment\n"
	    }
	    long -
	    label.l -
	    word -
	    label.w -
	    byte -
	    pbyte -
	    label.b {
		# Handle a definition

		set initial [lindex $element 1]
		set name [lindex $element 2]
		set comment [lindex $element 3]

		switch -exact $command {
	            long -
	            label.l {
		        set defline "__LONGreg__ $name;"
		    }
	            word -
	            label.w {
		        set defline "__WORDreg__ $name;"
		    }
	            byte -
	            pbyte -
	            label.b {
		        set defline "__BYTEreg__ $name;"
		    }
		}
	        append file "$defline\t\t//$comment\n"
	    }
	}
    }

    if {[string compare $prefix ""] != 0} {
        append file "#if !${prefix}NODEFINES\n"
        foreach element $defines {
	    set command [lindex $element 0]
	    switch -exact $command {
	        "" -
	        comment {
		    # skip
	        }
	        space.b -
	        space.w -
	        space.l -
	        long -
	        label.l -
	        word -
	        label.w -
	        byte -
	        pbyte -
	        label.b {
		    # make a define for the register name

		    set name [lindex $element 2]

	            append file "#define $prefix$name $name\n"
	        }
	    }
	}
        append file "#endif // !${prefix}NODEFINES\n"
    }
    return $file
}

#
# DdfMakeAsmHeader - process a ddf file and return the assembly header
#
proc code::DdfMakeAsmHeader {ddf thefiles} {
    upvar $thefiles files

    # parse the .ddf file
    DdfParse $ddf prefix variants modifier defines ai

    # return a list of header file names
    set files {}
    foreach variant $variants {
        lappend files "$variant.lib"
    }

    # adjust the modifier for assembly
    if {$modifier == "near"} {
	set modifier ".s"
    } elseif {$modifier == "far"} {
	set modifier ""
    }

    set file "; Definitions of registers\n"
    append file "\n"

    # output the definitions
    foreach element $defines {
	set command [lindex $element 0]
	switch -exact $command {
	    "" {
		# blank line
	        append file "\n"
	    }
	    comment {
		# comment
		append file "; [lindex $element 1]\n"
	    }
	    space.b -
	    space.w -
	    space.l -
	    long -
	    label.l -
	    word -
	    label.w -
	    byte -
	    pbyte -
	    label.b {
		# Handle a definition

		set name [lindex $element 2]
		set comment [lindex $element 3]
	        append file "\timport$modifier\t$name\t\t;$comment\n"
	    }
	}
    }
    return $file
}

#
#	DdfMakeVariantHeader - make the header for a variant
#
proc code::DdfMakeVariantHeader {hfile ddf {asm 0}} {
    if {$asm} {
	set comment ";"
    } else {
	set comment "//"
    }
    return "$comment
$comment $hfile - Device definitions.
$comment
$comment This file was created by CODE from $ddf.
$comment DO NOT EDIT. Edit $ddf instead and re-generate this file.
$comment

"
}

#
# DdfGetRegisters - process a ddf file and define registers
#
proc code::DdfGetRegisters {ddf} {
    variable DdfCommands
    variable DdfValues
    variable DdfDescriptions
    variable DdfAi
    variable DdfPrefix
    variable DdfVariants
    variable DdfModifier
    variable DdfOrder
    variable Configure

    # parse the .ddf file
    DdfParse $ddf DdfPrefix DdfVariant DdfModifier defines DdfAi

    foreach element $defines {
	set command [lindex $element 0]
	switch -exact $command {
	    long -
	    space.l -
	    space.b -
	    space.w -
	    word -
	    label.w -
	    byte -
	    pbyte -
	    label.b {
		# Handle a definition

		set value [lindex $element 1]
		set name [lindex $element 2]
		set comment [lindex $element 3]
		set DdfCommands($name) $command
		set DdfValues($name) $value
		set DdfDescriptions($name) $comment
		set Configure(Registers,$name) [code::Preference Registers $name]
		if {$Configure(Registers,$name) == {} && $value != {}} {
		    # set default value
		    set Configure(Registers,$name) $value
		}
		lappend DdfOrder $name
	    }
	}
    }
}

#
# DdfLoad - load the ddf file for the current processor variant
#
proc code::DdfLoad { {newproc 1} } {
    variable INTROL
    variable Preferences
    variable PreferenceChangedList
    variable DdfCommands
    variable DdfValues
    variable DdfDescriptions
    variable DdfAi
    variable DdfPrefix
    variable DdfVariants
    variable DdfModifier
    variable DdfOrder

    # clear any old values
    foreach name [array names DdfCommands] {
        set DdfCommands($name) {}
    }
    foreach name [array names DdfValues] {
        set DdfValues($name) {}
    }
    foreach name [array names DdfDescriptions] {
        set DdfDescriptions($name) {}
    }
    foreach name [array names DdfAi] {
        set DdfAi($name) {}
    }
    if {$newproc} {
        foreach name [array names Preferences Registers,*] {
            if {[info exists PreferenceChangedList($name)]} {
                unset PreferenceChangedList($name)
	    }
	    unset Preferences($name)
        }
        Preference System vectors {} 1
        Preference System devtable {} 1
	Preference System devstdin 0
	Preference System devstdout 0
	Preference System devstderr 0
    }
    set DdfPrefix {}
    set DdfVariants {}
    set DdfModifier {}
    set DdfOrder {}

    Preference Build ddf {} 1
    Preference Build import {} 1
    Preference Build memory {} 1
    Preference Build defaultmemory {} 1
    Preference Build module {} 1
    Preference Build sections {} 1
    Preference Build variables {} 1
    Preference Build vectors {} 1
    Preference Build vectorsimport {} 1

    set proc [Preference Build processor]
    set variant [Preference Build variant]
    if {$variant == {}} {
	return
    }
    set gen $Preferences(Build,${proc}gen)
    set ddf $Preferences(Build,${variant}ddf)

    Preference Build ddf $Preferences(Build,${variant}ddf) 1
    Preference Build import $Preferences(Build,${variant}import) 1
    Preference Build memory $Preferences(Build,${variant}memory) 1
    Preference Build defaultmemory $Preferences(Build,${variant}defaultmemory) 1
    Preference Build module $Preferences(Build,${variant}module) 1
    Preference Build sections $Preferences(Build,${variant}sections) 1
    Preference Build variables $Preferences(Build,${variant}variables) 1
    Preference Build vectors $Preferences(Build,${variant}vectors) 1
    Preference Build vectorsimport $Preferences(Build,${variant}vectorsimport) 1

    # search locally first
    set file $ddf
    if {$ddf == {}} {
	# no ddf file specified
	return
    }
    if {![file exists $file]} {
	# no project file, try standard place
        set file [file join $INTROL Libraries Assembly $gen $ddf]
    }
    if {![file exists $file]} {
	# no ddf file specified for variant or it doesn't exist
	return
    }
    # get the ddf info
    DdfGetRegisters $file
}

#
# DdfGetStartup - get the startup configuration for the current processor
# 	and variant
#
proc code::DdfGetStartup {} {
    variable DdfAi
    variable Preferences 

    if {[info exists DdfAi(Startup)] && $DdfAi(Startup) != {}} {
	return $DdfAi(Startup)
    }
    return $Preferences(Build,$Preferences(Build,processor)startup)
}

#
# DdfGetTabs - get the addition tabs for the current processor
# 	and variant
#
proc code::DdfGetTabs {} {
    variable DdfAi

    if {[info exists DdfAi(Tabs)] && $DdfAi(Tabs) != {}} {
	return $DdfAi(Tabs)
    }
    return {}
}

#
# DdfGetCS - get the CS info for the current processor
# 	and variant
#
proc code::DdfGetCS {} {
    variable DdfAi

    if {[info exists DdfAi(CS)] && $DdfAi(CS) != {}} {
	return $DdfAi(CS)
    }
    return {}
}

#
# DdfGet - get some info for the current processor
# 	and variant
#
proc code::DdfGet {name} {
    variable DdfAi

    if {[info exists DdfAi($name)] && $DdfAi($name) != {}} {
	return $DdfAi($name)
    }
    return {}
}

#
# DdfHasRegs - return TRUE if the current processor and variant has registers
#
proc code::DdfHasRegs {} {
    variable DdfOrder

    if {[info exists DdfOrder] && $DdfOrder != {}} {
	return 1
    }
    return 0
}

#
# DdfGetRegisterList - get a list of defined registers
#
proc code::DdfGetRegisterList {} {
    variable DdfOrder
    return $DdfOrder
}

#
# DdfGetRegisterInfo - get information about a register
#
proc code::DdfGetRegisterInfo {name} {
    variable DdfCommands
    variable DdfValues
    variable DdfDescriptions

    return [list $DdfCommands($name) $DdfValues($name) $DdfDescriptions($name)]
}

#
# DdfGetStartFile - generate the startup code
#
proc code::DdfGetStartFile {startfile} {
    variable INTROL
    variable DdfAi
    variable DdfModifier
    variable DdfValues

    # get the program entry point
    set entry [Preference System entry]
    set name [Preference Build name]
    # build the startup code
    set result ";
; $startfile - startup file for the project $name.
; This file was automatically generated by Introl-CODE. DO NOT EDIT.
"

    if {[Preference System notesinstartfile]} {
	# add project notes to the startfile
	set notes [Preference Build notes]
	if {$notes != {}} {
	    set notes [split $notes \n]
	    foreach note $notes {
	        append result "; $note\n"
	    }
	}
    }

    append result ";

		section		.start0
; The program entry point (at RESET)
__start:
"
    if {[Preference Module chipinit] != {} && [DdfHasRegs]} {
	# generate chip initialization code

	# get priority initialization order
	if {[info exists DdfAi(Order)]} {
	    set order $DdfAi(Order)
	} else {
	    set order {}
	}
	# get the list of registers
	set registers [DdfGetRegisterList]

	# get special pre-initialization code
	if {[info exists DdfAi(Precode)]} {
	    set precode $DdfAi(Precode)
	} else {
	    set precode {}
	}
	if {$DdfModifier == "near"} {
	    # use base page addressing
	    set import "import.s"
	} else {
	    set import "import"
	}
	if {$precode != {}} {
	    set uses [lindex $precode 0]
	    set sets [lindex $precode 1]
	    set code [lindex $precode 2]
	    foreach var $uses {
		# set variables values
		set $var [Preference Registers $var]
		if {[set $var] == {}} {
		    # use default
		    set $var $DdfValues($var)
		}
	    }
	    # add code, replacing variables
	    append result [subst -nocommands $code]
	    # remove set variables from register list
	    foreach var $sets {
		set registers [lremove $registers $var]
	    }
	}
	# do priority registers first
	foreach reg $order {
	    # initialize the register
	    DdfInitReg result $reg $import
	    # remove from non-priority list
	    set registers [lremove $registers $reg]
	}
	append result "__fastend:\t; Define the end of the fast initializations\n"
	# do the rest of the registers
	foreach reg $registers {
	    # initialize the register
	    DdfInitReg result $reg $import
	}
    }

    # grab initialization code from the library

    set proc [Preference Build processor]
    set gen [Preference Build ${proc}gen]
    foreach module {bssinit datainit fbssinit fdatainit stackinit interruptinit load_k} {
        if {![catch {Preference Module $module} symbol] && $symbol != {}} {
            regexp __(.*) $symbol all name
            set file [file join $INTROL Libraries Assembly $gen $name.s]
	    set f [open $file]
	    append result [read $f]
	    close $f
        }
    }
    if {[Preference System defineio]} {
	# build the calls to open stdin, stdout, and stderr
	append result "\n; Open stdio streams\n"
	append result "\t\tsection\t\t.startX\n"

	set DEV 0
        foreach one [DdfGetDevTab] {
	    # output calls
	    if {$one == {}} {
		# empty. This only happens when the last device was deleted
		continue
	    }
	    # get the name of the device
	    set name [lindex $one 0]

	    set TARGET [lindex $one 5]
	    if {$TARGET == {}} {
		# no open function
		incr DEV
		continue
	    } 

	    # get stdin, stdout stderr labels
	    if {$DEV == [Preference System devstdin] || \
		$DEV == [Preference System devstdout] || \
		$DEV == [Preference System devstderr]} {
		set COMMENT "Open the $name stream"
		set NAME rwi$DEV
	        set code [Preference Build ${proc}push1ptr]
	        append result [subst -nocommands $code]
	        # have all parameters, output an open call
	        if {[Preference Build largemodel]} {
	            set code [Preference Build ${proc}farcall]
	        } else {
	            set code [Preference Build ${proc}funccall]
	        }
	        append result [subst -nocommands $code]
	        set code [Preference Build ${proc}pop1ptr]
		if {$code != {}} {
	            append result [subst -nocommands $code]
		}
	    }
	    incr DEV
        }
    }

    if {$entry != {}} {
	append result "\t\tsection\t\t.startX\n"
	append result "\t\timport\t\t$entry\n"
	set TARGET $entry
	set COMMENT "Call the program entry point"
	if {[Preference Build largemodel]} {
	    set code [Preference Build ${proc}farcall]
	} else {
	    set code [Preference Build ${proc}funccall]
	}
	append result [subst -nocommands $code]
    }
    if {![catch {Preference Module onexit} symbol] && $symbol != {}} {
        regexp __(.*) $symbol all name
        set file [file join $INTROL Libraries Assembly $gen $name.s]
	set f [open $file]
	append result [read $f]
	close $f
    }

    if {[Preference System definevectors]} {
	# build the vector table
	append result "\n; The processor vector table\n"
	append result "\t\tsection\t\t.vectors\n"
	append result "vectors\n"
        set vector 0
	set traps 0
	set hastraps 0
	set imports {}
	set maxvector [Preference System maxvector]
	if {$maxvector == {}} {
	    # undefined, do them all
	    set maxvector 99999
	}
	set code [Preference Build ${proc}vector]
        set list [ConfigureGetValue System vectors]
        foreach "foo NAME DESC" $list {
	    # output a vector table entry

	    # check for maximum vector number
	    if {$traps > $maxvector} {
		break
	    }
	    # split possible expressions
	    set entry [split $NAME {()|+&->< }]
	    if {$NAME == "__exit" && [Preference System unhandledtraps]} {
		set hastraps [Preference System unhandledtraps]
		set NAME "trap$traps"
	    } else {
	        foreach name $entry {
		    if {$name == {} || ![catch  {expr int($name)}]} {
		        # weed out nulls and integers
		        continue
		    }
		    if {$name != "__start" && [lsearch $imports $name] == -1} {
		        lappend imports $name
		    }
	        }
	    }
	    if {$DESC == {}} {
		set DESC "Reserved"
	    }
	    append result [subst -nocommands $code]
	    incr vector
	    incr traps
        }
	append result "vectorsend\n"
	append result "vectorsize\tequ\t\tvectorsend-vectors\n"
	foreach name $imports {
	    append result "\t\timport\t\t$name\n"
	}
	if {$hastraps} {
	    if {$hastraps == 1} {
	        set code [Preference Build ${proc}unhandledtrap]
	        append result "\n; Unhandled exception traps\n"
	    } else {
	        set code [Preference Build ${proc}revectortrap]
	        append result "\n; Revectored exception traps\n"
	        set REVECTORS [Preference System revectors]
		if {[string compare $REVECTORS ""] == 0} {
		    set REVECTORS __revectors
		}
	        set entry [split $REVECTORS {()|+&->< }]
	        set needrevectors 1
	        foreach name $entry {
		    if {$name == {} || ![catch  {expr int($name)}] \
		      || [string compare $name "vectorsize"] == 0} {
		        # weed out nulls and integers, and the symbol vectorsize
		        continue
		    }
		    if {[string compare $name "__revectors"] == 0} {
			# the symbol __revectors is imported
			set needrevectors 0
		    }
		    if {$name != "__start" && [lsearch $imports $name] == -1} {
	                append result "\t\timport\t\t$name\n"
		    }
	        }
		if {$needrevectors} {
		    append result "__revectors:\tequ\t\t$REVECTORS\n"
		}
		set REVECTORS __revectors
		append result "__revectorsend:\tequ\t\t__revectors+vectorsize\n"
	    }
	    set traps 0
	    append result "\t\tsection\t\t.text\n"
            foreach "foo NAME DESC" $list {
	        # check for maximum vector number
	        if {$traps > $maxvector} {
		    break
	        }
		if {$NAME != "__exit"} {
		    incr traps
		    continue
		}
	        if {$DESC == {}} {
		    set DESC "Reserved"
	        }
		set TRAP "trap$traps"
		set TRAPNO $traps
	        append result [subst -nocommands $code]
		incr traps
	    }
	}
    }

    if {[Preference System defineio]} {
	# build the device table
	append result "\n; The program device table\n"
	append result "\t\tsection\t\t.const\n"
	append result "__devtab:\n"
	set imports {}
	set names {}
	set io_read {}
	set io_write {}
	set code [Preference Build ${proc}devtab]
	set funcptrmod [Preference Build ${proc}funcptrmod]

	set DEV 0
        foreach one [DdfGetDevTab] {
	    # output FILE data
	    if {$one == {}} {
		# empty. This only happens when the last device was deleted
		continue
	    }
	    # output a device entry
	    set name [lindex $one 0]
	    set NAME dev$DEV
	    lappend names $name $NAME
	    set MODE [lindex $one 1]
	    if {$MODE == {}} {
		set MODE 0
	    }
	    set FLAGS [lindex $one 2]
	    if {$FLAGS == {}} {
		set FLAGS 0
	    }
	    set char [lindex $one 3]
	    if {[catch {expr int($char)} DELETE]} {
		# must be a ^x or del
		if {$char == "del"} {
		    set DELETE 0x7F
		} elseif {[string index $char 0] == "^"} {
		    scan [string index $char 1] %c DELETE
		    set DELETE [expr $DELETE & 0x1F]
		} else {
		    set DELETE 0
		}
	    }

	    set char [lindex $one 4]
	    if {[catch {expr int($char)} KILL]} {
		# must be a ^x or del
		if {$char == "del"} {
		    set KILL 0x7F
		} elseif {[string index $char 0] == "^"} {
		    scan [string index $char 1] %c KILL
		    set KILL [expr $KILL & 0x1F]
		} else {
		    set KILL 0
		}
	    }

	    if {$DEV != 0} {
	        append result "\t\tsection\t\t.const\n"
	    }
	    # get stdin, stdout stderr labels
	    if {$DEV == [Preference System devstdin]} {
		append result "__stdin:\n"
	    }
	    if {$DEV == [Preference System devstdout]} {
		append result "__stdout:\n"
	    }
	    if {$DEV == [Preference System devstderr]} {
		append result "__stderr:\n"
	    }

	    set OPEN [lindex $one 5]
	    if {$OPEN == {}} {
		set OPEN 0
	    } else {
	        if {[lsearch $imports $OPEN] == -1} {
		    lappend imports $OPEN
	        }
		append OPEN $funcptrmod
	    }
	    set CLOSE [lindex $one 6]
	    if {$CLOSE == {}} {
		set CLOSE 0
	    } else {
	        if {[lsearch $imports $CLOSE] == -1} {
		    lappend imports $CLOSE
	        }
		append CLOSE $funcptrmod
	    }
	    set READ [lindex $one 7]
	    if {$READ == {}} {
		set READ 0
	    } else {
	        if {[lsearch $imports $READ] == -1} {
		    lappend imports $READ
	        }
	        if {$DEV == [Preference System devstdin]} {
		    set io_read $READ
		}
		append READ $funcptrmod
	    }
	    set WRITE [lindex $one 8]
	    if {$WRITE == {}} {
		set WRITE 0
	    } else {
	        if {[lsearch $imports $WRITE] == -1} {
		    lappend imports $WRITE
	        }
	        if {$DEV == [Preference System devstdout]} {
		    set io_write $WRITE
		}
		append WRITE $funcptrmod
	    }
	    set IOCTL [lindex $one 9]
	    if {$IOCTL == {}} {
		set IOCTL 0
	    } else {
	        if {[lsearch $imports $IOCTL] == -1} {
		    lappend imports $IOCTL
	        }
		append IOCTL $funcptrmod
	    }
	    set DESC [lindex $one 10]

	    set bufsize [lindex $one 11]
	    if {$DEV == [Preference System devstdin] && ($FLAGS & 2)} {
		# set a resonable default for buffered stdin
		set bufsize 80
	    }
	    if {[string compare $bufsize {}] != 0} {
		# allocate a buffer
		set BUFFER buf$DEV
	        append result "\t\tsection\t\t.bss\n"
	        append result "$BUFFER\t\tds.b\t\t$bufsize\t\t; buffer for $name\n"
	        append result "\t\tsection\t\t.const\n"
	    } else {
		set BUFFER 0
	    }

	    # have all parameters, output a table entry
	    append result [subst -nocommands $code]

	    incr DEV
        }

	# define the end of the device table
	append result "\t\tsection\t\t.const\n"
	set code [Preference Build ${proc}ptrvar]
	set LABEL __devtabend:
	set NAME *
	set DESC "Pointer to the end of device table"
        append result [subst -nocommands $code]


	set code [Preference Build ${proc}string]
	append result "\n; The program device name table\n"
	append result "\t\tsection\t\t.const\n"
	foreach "STRING LABEL" $names {
	    append result [subst -nocommands $code]
	}

	# import the device handler functions
	foreach name $imports {
	    append result "\t\timport\t\t$name\n"
	}

	append result "\n; Set up labels for trapping stdin/out for debugging\n"
        if {$io_read != {}} {
	    append result "__io_read:\tequ\t\t$io_read\n"
	}
        if {$io_write != {}} {
	    append result "__io_write:\tequ\t\t$io_write\n"
	}
    }

    append result "\t\tend\n"
}

#
# DdfStartFile - generate the startup code and save to a file
#
proc code::DdfStartFile {} {
    # add the startup code to the project
    set startfile [subst -nocommands [BuildSubstitute [Preference System startfilename]]]
    set result [DdfGetStartFile $startfile]
    ConfigureAddProject [list $startfile]
    set file [open $startfile w]
    puts -nonewline $file $result
    close $file
}

#
# DdfGetDevTab - get the device table
#
proc code::DdfGetDevTab {} {
    # get the default device

    set devtab [Preference System devtable]
    if {$devtab == {}} {
	# get the default
        set devtab [string trim [DdfGet DEVTAB]]
    }
    if {$devtab == {}} {
	# nothing defined
        set devtab [list [DdfDefaultDevice]]
    }
    return $devtab
}

#
# DdfDefaultDevice - the default device entry
#
proc code::DdfDefaultDevice {} {
    return { DEBUG 3 0 ^h ^x __DEBUG_open __DEBUG_close __DEBUG_read
	  __DEBUG_write __DEBUG_ioctl "The debug (null) device" }
}
#
# DdfInitReg - initialize a processor register
#
proc code::DdfInitReg { to name import } {
    variable DdfCommands
    variable DdfValues
    variable DdfDescriptions
    variable DdfPrefix

    upvar $to result
    # get the current value
    set VAL [Preference Registers $name]
    if {$VAL == {} || $DdfValues($name) == $VAL} {
	return
    }
    set REG $name
    if {[string length $import] >= 8} {
        append result "\t\t$import\t$REG\t\t; $DdfDescriptions($name)\n"
    } else {
        append result "\t\t$import\t\t$REG\t\t; $DdfDescriptions($name)\n"
    }
    switch $DdfCommands($name) {
	    long -
	    label.l {
		set code [Preference Build [Preference Build processor]setlong]
	        append result [subst -nocommands $code]
	    }
	    word -
	    label.w {
		set code [Preference Build [Preference Build processor]setword]
	        append result [subst -nocommands $code]
	    }
	    byte -
	    pbyte -
	    label.b {
		set code [Preference Build [Preference Build processor]setbyte]
	        append result [subst -nocommands $code]
	    }
    }
}

#
# DdfSetRegister - set the value of a register
#
proc code::DdfSetRegister {name thevar} {
    upvar $thevar var

    set value $var
    if {$value != {}} {
	# check for a valid integer
	if {[catch {expr int($value)}]} {
	    tk_messageBox \
		-icon error \
	        -message "Invalid integer constant" \
	        -type ok
	    set var {}
	    return
	}
	# put in a standard format
	set value [format 0x%02X $value]
	set var $value
    }
    # change the project database, if necessary
    PreferenceSetIfChanged Registers $name $value
}
