#
#	Debugger - the CODE debugger
#

# This is the set of debugger commands available from the command prompt

array set code::DebuggerCommands {
 HELP "define balloon help for a widget"
 BYTE "read/write a byte from memory"
 WORD "read/write a word from memory"
 LONG "read/write a long from memory"
 BR "set a processor breakpoint"
 REG "get or set a processor register"
 VAR "get or set a debugger variable"
 VADR "get the address of a variable in an object file"
 EXPR "evaluate a target expression"
 MODULE "load debugger modules"
 PROCESSOR "return the current processor"
 VARIANT "return the current processor variant"
 TARGET "return the current debugger target"
 RESET "reset the target processor"
 PREFERENCE "Get or set a preference"
 PORT "set the target parallel i/o port"
 SERIAL "set the target serial i/o port"
 WHERE "return the program position for an address"
 TK "load Tk into the module"
 WINDOW "create a window for a debugger module"
 TITLE "set a module window title"
 WINDOWS "return the list of windows for this module"
 DESTROY "destroy a window created by this module"
 UI "load a user-interface into a module"
 BUTTON "create a button for a module"
 DELETEBUTTON "delete button(s) for a module"
 STDIN "set up a stdin handler for a module"
 STDOUT "show target output"
 UPDATE "update the debugger state"
 INFO "dummy command for module info"
}

# the main Debugger options
array set code::Debugger {
    source {}
    line 1.0
    pc 0
    pcchanged 1
    notstate 1
    newfile 0
    tag cursrc
    tagstart {}
    tagend {}
    mixed 0
    domixed 0
    automixed 0
    disassembly {}
    opened {}
    loaded 0
    mtime {}
    changed 0
    running 0
    stopping 0
    where "No current context"
    context ""
    PC 0
    server {}
    serverport {}
    serverip {}
    client {}
    breakexit {}
    breakio {}
    stdio {}
    stdioinput {}
    stdioinputhandler {}
    registers {}
    registersproc {}
    trace {}
    counters {}
    tid 0
    traceenable 0
    tracesize 4096
    modules {}
    moduletools 0
    targetmodule {}
    stopmsg {}
    processor {}
    serial {}
    serialbaud {}
    serialparity {}
    serialdata {}
    serialstop {}
}

#
# DebuggerOpen - open an object file in the debugger
#
proc code::DebuggerOpen {{file {}} {noload 0}} {
    variable Debugger

    if {$file == ""} {
        # open from the object directory by default
	set dir [Preference Build OBJECT]
	if {$dir == {}} {
	    # otherwise the project directory
	    set dir [pwd]
	}
	set ext [Preference Build [Preference Build processor]E]
	if {[Preference DebuggerInternal lastfile] != {}} {
	    set file [tk_getOpenFile \
	        -filetypes {{{Executable files}
	    	    {.e[code::Preference Build [code::Preference Build processor]E] .out}}
	  	    {{Other files} {.*}}} \
	        -initialdir [BuildRealFileName $dir] \
		-initialfile [BuildRealFileName [Preference DebuggerInternal lastfile]]]
	} else {
	    set file [tk_getOpenFile -initialdir [BuildRealFileName $dir] \
	        -filetypes {{{Executable files}
		    {.e[code::Preference Build [code::Preference Build processor]E] .out}}
		     {{Other files} {.*}}} ]
	}
	if {$file == ""} {
	        return 0
	}
    }
    if {[catch {DebuggerDbg object $file} od]} {
        # error opening file
        tk_messageBox \
  	    -parent . \
	    -icon error \
	    -message "Cannot open $file.\n\`$od'" \
	    -type ok
	return 0
    }

    set Debugger(opened) $file
    set Debugger(loaded) 0
    set Debugger(mtime) [file mtime $file]
    Preference DebuggerInternal lastfile [BuildFileName $file]
    set Debugger(newfile) 1
    if {!$noload && [Preference Debugger autoload]} {
        # autoload
	status . "Auto Load"
	if {![DebuggerLoad]} {
	    return 0
	}
    }
    return 1
}

#
# DebuggerLoad - load or reload the object file
#
proc code::DebuggerLoad {} {
    variable Debugger

    # always reset first
    if {![DebuggerDoReset 0 0]} {
	return 0
    }

    if {$Debugger(opened) == {}} {
        # no object file opened yet
	set file [BuildProgramName]
	set dir [Preference Build OBJECT]
	if {$dir != {}} {
	    set file [BuildRealFileName [file join $dir $file]]
	}
	if {$file != "" && [file exists $file]} {
	    status . "Opening [file nativename $file]"
	    update idletasks
	} else {
	    set file {}
	}
	if {![DebuggerOpen $file]} {
	    return 0
	}
    }
    status . "Loading..."
    update idletasks
    if {[catch {DebuggerDbg download} msg]} {
	# error downloading file
	tk_messageBox \
	    -parent . \
	    -icon error \
	    -message "Can't download: $msg" \
	    -type ok
	status . ""
	return 0
    }

    # mark file as loaded
    set Debugger(loaded) 1

    # initialize standard breakpoints
    DebuggerSetupBreakpoints .

    # initialize modules
    foreach module $Debugger(modules) {
        if {[info commands ${module}loaded] != {}} {
            ${module}loaded
        }
    }

    status . ""
    update idletasks
    if {[Preference Debugger autoreset]} {
        # autoreset
	DebuggerDoReset
    }
    return 1
}


#
# DebuggerReset - reset the processor with a rebuild check
#
proc code::DebuggerReset {{go 0}} {
    variable Debugger

    if {   $Debugger(opened) != {}
	&& [Preference Debugger autobuild]} {
	# see if project needs a rebuild
	status . "Build Check"
	set result [Build . build]
	if {$result == {}} {
	    # build failed
	    return 0
	}
	if {$result != $Debugger(opened)} {
	    DebuggerOpen $result
	}
    }
    after idle code::DebuggerDoReset $go
}

#
# DebuggerDoReset - reset the processor
#
proc code::DebuggerDoReset {{go 0} {autoload 1}} {
    variable Debugger

    if {   $autoload
	&& !$Debugger(loaded)
	&& [Preference Debugger autoload]} {
        # autoload
	status . "Auto Load"
	if {![DebuggerLoad]} {
	    return 0
	}
    }

    status . "Reseting"
    update idletasks
    if {[catch {DebuggerDbg target -id} target]} {
	tk_messageBox \
	    -parent . \
	    -icon error \
	    -message "Can't reset: no processor or target set." \
	    -type ok
	return 0
    }

    if {!$go} {
	# plain reset
        if {[catch {DebuggerDbg reset} msg]} {
	    DebuggerCommError $msg
	    status . ""
	    return 0
        }
        set Debugger(pc) 0
	if {$autoload} {
            # stop in case we were running
            DebuggerStop
            foreach module $Debugger(modules) {
	        # issue a reset to all modules that need it
	        if {[info commands ${module}reset] != {}} {
	            ${module}reset
	        }
            }
            DebuggerState
	}
    } else {
	DebuggerDbg go -reset
        set Debugger(pc) 0
	if {$autoload} {
            DebuggerState
	}
    }

    status . ""
    return 1
}

#
# DebuggerCheckTime - check to see if the disk copy of the exectuable is newer
#
proc code::DebuggerCheckTime {result} {
    variable Debugger

    set file $Debugger(opened)
    if {$file != {} || $file != $result} {
	# check for a newer file in the disk
	if {![file exists $result]} {
	    # file was lost
	    return
	}
	if {$Debugger(mtime) != {} && [file mtime $result] > $Debugger(mtime)} {
	    # need a reload

	    set Debugger(mtime) {}
	    # get rid of the cancel box
            BuildCancelBox . {}
	    after idle code::DebuggerCheckReload $result
	}
    }
}

#
# DebuggerCheckReload - reload object file and update modified time
#
proc code::DebuggerCheckReload {file} {
    variable Debugger

    if {![Preference Debugger autocheck]} {
	# autocheck is not enabled
        set result [tk_messageBox \
	    -parent . \
            -icon question \
            -message "'$file' has changed on the disk. Reload?" \
            -default yes \
            -type yesno]
        if {$result == "no"} {
	    # don't reload, but remember new time
            set Debugger(mtime) [file mtime $file]
	    return
        }
    }
    status . "Reloading [file nativename $file]"
    DebuggerOpen $file
}

#
# DebuggerPreferences - set up environment according to preferences
#
proc code::DebuggerPreferences {} {
    variable Debugger
    variable DebuggerRegisters
    variable DebuggerValues
    variable DebuggerCommands

    # export the basic debugging commands
    foreach command [array names DebuggerCommands] {
	namespace export $command
        namespace eval :: namespace import -force code::$command
    }

    set Debugger(domixed) [Preference Debugger mixed]
    set Debugger(automixed) [Preference Debugger automixed]
    set Debugger(traceenable) [Preference DebuggerInternal traceenable]
    if {[Preference DebuggerInternal server]} {
	# run as an execute server
	if {[Preference DebuggerInternal serverport] != $Debugger(serverport)} {
            if {$Debugger(server) != {}} {
	        close $Debugger(server)
	    }
	    set Debugger(serverport) [Preference DebuggerInternal serverport]
	    if {[catch {socket -server code::DebuggerRemote $Debugger(serverport)} fd]} {
                set result [tk_messageBox \
		    -parent . \
    	            -icon error \
	            -message $fd \
	            -type ok]
	        set Debugger(server) {}
	        set Debugger(serverport) {}
	        Preference DebuggerInternal server 0
            } else {
                set Debugger(server) $fd
	    }
	}
    } elseif {$Debugger(server) != {}} {
	# not a server any more
	close $Debugger(server)
	set Debugger(server) {}
    }

    if {[Preference DebuggerInternal serverclient]} {
	# run as an execute client
	if {[Preference DebuggerInternal serverip] != $Debugger(serverip) \
	    || [Preference DebuggerInternal serverport] != $Debugger(serverport)} {
            if {$Debugger(client) != {}} {
	        close $Debugger(client)
	    }
	    set Debugger(serverip) [Preference DebuggerInternal serverip]
	    set Debugger(serverport) [Preference DebuggerInternal serverport]
	    set try 1
	    while {$try} {
	        if {[catch {socket $Debugger(serverip) $Debugger(serverport)} fd]} {
                    set result [tk_messageBox \
		        -parent . \
    	                -icon error \
	                -message $fd \
	                -type retrycancel]
		    if {$result == "retry"} {
			continue
		    }
	            set Debugger(client) {}
	            set Debugger(serverip) {}
	            set Debugger(serverport) {}
		    Preference DebuggerInternal serverclient 0
		    break
	        } else {
	            set Debugger(client) $fd
		    break
		}
	    }
	}
    } elseif {$Debugger(client) != {}} {
	# not a client any more
	close $Debugger(client)
	set Debugger(client) {}
	set Debugger(serverip) {}
	set Debugger(serverport) {}
    }

    set proc [Preference Build processor]
    if {$proc == {}} {
	# no processor set
	return
    }
    if {$Debugger(processor) != $proc} {
	# The processor has changed
        DebuggerDbg processor [Preference Build processor]
	set Debugger(processor) $proc
	set changed 1
    } else {
	set changed 0
    }
    if {[DebuggerDbg target -id] != [Preference DebuggerInternal target] && \
        [catch {DebuggerDbg target [Preference DebuggerInternal target]} msg]} {
	# processor must have changed - set default target
	if {!$changed} {
	    tk_messageBox \
		-parent . \
	        -icon error \
	        -message "Cannot set target [Preference DebuggerInternal target]: $msg.\nUsing simulator." \
	        -type ok
	}
	Preference DebuggerInternal target [DebuggerDbg target -id]
    }
    if {$changed} {
	DebuggerModuleDeleteAll
    }
    # load the target script module, if any
    set target [DebuggerDbg target -id]
    if {$Debugger(targetmodule) != $target} {
	# new or changing
	if {$Debugger(targetmodule) != {}} {
	    # delete the old one
	    DebuggerModuleDelete $Debugger(targetmodule)
	}
	set Debugger(targetmodule) $target
        if {[DebuggerFindModule $target] != {}} {
	    DebuggerModule $target
        }
    }

    if {$changed} {
        # clear old registers
	catch {destroy $Debugger(registers)}
	set Debugger(registers) {}
        catch {unset DebuggerRegisters}

	foreach counter $Debugger(counters) {
            unset Debugger([lindex $counter 0])
	}
	set Debugger(counters) {}

        catch {unset DebuggerValues}

	# rebuild the module window, if necessary
	if {[winfo exists .modules] && [winfo ismapped .modules]} {
	    destroy .modules
	    PreferenceCallbackForget .modules
	    DebuggerModuleSelect
	} else {
	    # not, mapped, just destroy if it exists
	    catch {destroy .modules}
	    PreferenceCallbackForget .modules
	}
    }

    # add subproject directories to the search path
    foreach subproject [Preference Build subprojects] {
        set subproject [file dirname [BuildRealFileName [lindex $subproject 0]]]
        set current [DebuggerDbg path]
        if {[lsearch $current $subproject] == -1} {
    	    DebuggerDbg path $subproject
	}
    }

    if {$changed} {
        # load any debugger scripts
        foreach module [Preference DebuggerInternal modules] {
            if {[DebuggerFindModule $module] != {}} {
	        DebuggerModule $module
	    }
        }
        if {[Preference Debugger autoopen]} {
	    # make sure Build is initialized
	    Build . setproc $proc
            set file [BuildSubstitute [Preference DebuggerInternal lastfile]]
	    if {$file != {}} {
		if {[file exists $file]} {
                    DebuggerOpen $file 1
		}
	    }
        }
    }
}

#
# REGISTERSstartup - return the command to open the registers window
#
proc code::REGISTERSstartup {doc} {
    return "REGISTERS"
}

#
# REGISTERS - create the register window
#
proc code::REGISTERS {args} {
    if {[catch {DebuggerDbg registers} regs]} {
	# error getting registers
        tk_messageBox \
            -parent . \
    	    -icon error \
	    -message "Registers are not available yet: $regs" \
	    -type ok
	return
    }
    set type "Registers"
    set name ""
    set w [CODEfind $type $name]
    if {$w != {}} {
	# have a register window, raise and return
	if {   [$w cget -state] == "minimized" 
	    || [$w cget -state] == "withdrawn"} {
	    $w configure -state normal
	}
	$w raise
	return $w
    }
    set w [eval document .work.registers -type $type \
	-raiseproc code::DOCraise \
	-startupproc code::REGISTERSstartup $args]
    bind $w <<State>> "code::REGISTERSsetup $w; break"
    REGISTERSsetup $w 1
    return $w
}

#
# REGISTERSsetup - set up the contents of the registers window
#
proc code::REGISTERSsetup {w {force 0}} {
    variable Debugger
    variable DebuggerRegisters

    if {!$Debugger(running)} {
        DebuggerGetRegisters
    }
    if {!$force && $Debugger(registersproc) == [Preference Build processor]} {
	# we have already created the window for this processor
	return
    }
    catch {unset DebuggerRegisters}
    catch {destroy $w.c}

    set c [scrollcanvas $w.c -scrollbar auto]
    $w pack $c -fill both -expand 1
    set f [frame $c.f]
    $c configure -background [$f cget -background]
    $c create window 0 0 -window $f -anchor nw
    DebuggerGetRegisters
    set Debugger(registersproc) [Preference Build processor]
    set image [CODEicon efile.icon]
    $w configure -image $image -icontext Registers \
	-title  "$Debugger(registersproc) Registers"
    # first, find the max widths
    set maxvalue 0
    set regs [DebuggerDbg registers]
    foreach reg $regs {
        set valuewid [expr {[string length [lindex $reg 1]] + 2}]
        if {$valuewid > $maxvalue} {
    	    set maxvalue $valuewid
	}
    }
    # give a little room
    incr maxvalue
    set row 0
    set col 0
    foreach reg $regs {
	set name [lindex $reg 0]
	if {$name == {}} {
	    set col 0
	    incr row
	    continue
	}
	set l [label $f.n$name -text $name] 
	set e [entry $f.e$name -width $maxvalue -textvariable code::DebuggerRegisters($name)]
	grid $l -in $f -row $row -column $col -sticky e
	incr col
	grid $e -in $f -row $row -column $col -sticky w
	incr col
	bind $f.e$name <FocusOut> "code::DebuggerRegisterCheck $name"
	bind $f.e$name <Return> "code::DebuggerRegisterCheck $name"
	set format [lindex $reg 2]
	if {$format == {}} {
	    # no format information
	    continue
	}
	# set up controls using the given format
	set valuewid [string length [lindex $reg 1]]
	if {$valuewid >= 8} {
	    set bit 31
	} elseif {$valuewid >= 4} {
	    set bit 15
	} else {
	    set bit 7
	}
	set ff [frame $f.ff$name]
	grid $ff -in $f -row $row -column $col -sticky w -columnspan 10
	set l [label $ff.l -text "$name fields:"] 
	grid $l -in $ff -row 0 -column 0 -sticky w
	set bcol 1
	foreach field $format {
	    # display a field
	    set bitname [lindex $field 0]
	    set help [lindex $field 1]
	    if {$bitname == {}} {
	        # undefined
		incr bit -1
	    } elseif {[llength $bitname] == 1} {
	        # a single bit
	        set r [checkbutton $ff.cb$bitname \
		    -text $bitname \
		    -variable code::DebuggerRegisters($name) \
		    -maskvalue [expr {1 << $bit}] \
		    -onvalue [expr {1 << $bit}] \
		    -command "code::DebuggerRegisterCheck $name"]
		.ttip add $r $help
		grid $r -in $ff -row 0 -column $bcol -sticky w
		incr bcol
		incr bit -1
	    } else {
	        # a bit field
		set len [lindex $bitname 1]
		set values [lindex $bitname 2]
		set bitname [lindex $bitname 0]
		# find the widest
		set width 0
		foreach fieldtext $values {
		    set strlen [string length $fieldtext]
		    if {$strlen > $width} {
		        set width $strlen
		    }
		}
		set end [expr {$bit - $len + 1}]
		set mask [expr {1 << $bit}]
		set start $bit
		# get the mask for this field
		while {$bit > $end} {
		    set mask [expr {$mask | ($mask >> 1)}]
		    incr bit -1
		}
		set value [expr {1 << $end}]
		set fieldvalue 0
		set DebuggerRegisters($name,$bitname,mask) $mask
		set DebuggerRegisters($name,$bitname) {}
		foreach fieldtext $values {
		    set DebuggerRegisters($name,$bitname,$fieldtext) $fieldvalue
		    set DebuggerRegisters($name,$bitname,$fieldvalue) $fieldtext
		    incr fieldvalue $value
	        }
	        set cf [frame $ff.cbf$bitname]
	        grid $cf -in $ff -row 0 -column $bcol -sticky w
	        incr bcol
	        set l [label $cf.l -text $bitname:]
	        grid $l -in $cf -row 0 -column 0 -sticky e
	        .ttip add $l $help
	        set DebuggerRegisters($name,$bitname) [lindex $values 0]
	        set r [combobox $cf.b$bitname \
		    -maxheight 0 \
		    -textvariable code::DebuggerRegisters($name,$bitname) \
			-width $width \
			-editable false \
			-command "code::DebuggerSetField $name $bitname ; #"]
		.ttip add $r $help
		grid $r -in $cf -row 0 -column 1 -sticky w
		foreach fieldtext $values {
		    $r list insert end $fieldtext
		}

		DebuggerGetField $name $bitname
		trace variable DebuggerRegisters($name) w "code::DebuggerGetField $name $bitname; #"
		incr bit -1
	    }
	}
    }
    $c configure -scrollbar auto
}

#
# DebuggerSetField - a register field has been changed, update the register value
#
proc code::DebuggerSetField {name fieldname} {
    variable DebuggerRegisters

    set current $DebuggerRegisters($name,$fieldname)
    if {   [info exists DebuggerRegisters($name,$fieldname,old)]
	&& $DebuggerRegisters($name,$fieldname,old) == $current} {
	# not changed
	return
    }
    set mask $DebuggerRegisters($name,$fieldname,mask)
    set value $DebuggerRegisters($name,$fieldname,$current)
    set DebuggerRegisters($name) [expr {($DebuggerRegisters($name) & ~ $mask) | $value}]
    DebuggerRegisterCheck $name
}

#
# DebuggerGetField - get the value of a register field
#
proc code::DebuggerGetField {name fieldname} {
    variable DebuggerRegisters

    set mask $DebuggerRegisters($name,$fieldname,mask)
    set value [expr {$DebuggerRegisters($name) & $mask}]
    set new $DebuggerRegisters($name,$fieldname,$value)
    set DebuggerRegisters($name,$fieldname,old) $DebuggerRegisters($name,$fieldname)
    set DebuggerRegisters($name,$fieldname) $new
}

#
# DebuggerEvaluate - evaluate a selected expression and puts its value the status bar
#
proc code::DebuggerEvaluate {t} {
    if {   ![catch {winfo parent [selection own -displayof $t]} tmp]
	&& $tmp == "$t.text"
	&& ![catch {selection get -displayof $t} selection]
	&& $selection != {}} {
    } else {
        return
    }
    if {[catch {DebuggerDbg expression $selection} value]} {
	# an error occured
        status . $value
	return
    }
    if {![catch {expr int($value)} int] && [string match $int $value]} {
	# display value in hex also
        status . "$selection = $value ([format "0x%02X" $value])"
    } else {
        status . "$selection = $value"
    }
}

#
# WATCHstartup - return the command to open a watch window
#
proc code::WATCHstartup {doc} {
    return "WATCH [list [$doc.watch cget -contents]]"
}

#
# WATCH - create a watch window
#
proc code::WATCH {{exprs {}} args} {
    variable CODEID
    variable Debugger
    variable WATCHcount

    if {![info exists WATCHcount]} {
	set WATCHcount 1
    }
    set type Watch
    set title "Watch $WATCHcount"
    incr WATCHcount
    set name $exprs
    set w .work.doc$CODEID
    incr CODEID
    
    eval document $w -type \"$type\" -raiseproc code::WATCHraise \
	-startupproc code::WATCHstartup $args
    bind $w <<State>> "code::WATCHupdate $w; break"
    set c [scrollcanvas $w.watch -scrollbar auto]
    $w pack $c -fill both -expand 1
    set f [frame $c.f]
    $c configure -background [$f cget -background]
    $c create window 0 0 -window $f -anchor nw
    set image [CODEicon efile.icon]
    $w configure -image $image -icontext $title -title $title
    set Debugger(watches$w) 0
    WATCHraise $w
    foreach expr $exprs {
	# add any predefined watches
        WATCHadd $w $expr
    }
    WATCHupdate $w
    $c configure -scrollbar auto
    return $w
}

#
# WATCHraise - raise a watch window
#
proc code::WATCHraise {doc} {
    variable WATCHlast
    DOCraise $doc
    # remember last watch window
    set WATCHlast $doc
}

#
# WATCHupdate - update the contents of a watch window
#
proc code::WATCHupdate {w} {
    variable Debugger

    foreach watch [$w.watch cget -contents] {
	WATCHvalue $w $watch
    }
}

#
# WATCHadd - add a watch to the watch window
#
proc code::WATCHadd {{doc {}} {expr {}}} {
    variable Debugger
    variable WATCHlast

    if {$doc == {} && [info exists WATCHlast]} {
	# use last raised watch window
	set doc $WATCHlast
    }
    # watch current selection, if present
    set hassel 0
    if {$expr == {} && ![catch {selection get} sel] && $sel != {}} {
	set hassel 1
	set expr $sel
    }
    if {$expr == {} || ($hassel && [catch {DebuggerDbg expression $expr}])} {
	set expr [WATCHget New $doc $expr]
        if {$expr == {}} {
	    return
	}
        if {$doc == {}} {
	    # no watch window exists, create one
	    set doc [WATCH]
        }
    }
    if {$doc == {}} {
	# no watch window exists, create one
	WATCH $expr
	return
    }
    $doc raise
    set row [lindex $expr 4]
    set gridwidget [lindex $expr 3]
    set realexpr [lindex $expr 2]
    set format [lindex $expr 1]
    set expr [lindex $expr 0]
    set f $doc.watch.f$gridwidget
    # create the expand frame, if needed
    if {![catch {frame $f} exf]} {
        grid $exf -row 1 -column 0
    }

    if {$row == {}} {
        set row $Debugger(watches$doc)
        incr Debugger(watches$doc)
    } elseif {$gridwidget == {} && $row >= $Debugger(watches$doc)} {
	set Debugger(watches$doc) [expr $row + 1]
    }

    set Debugger(expr$f.$row) $expr
    set Debugger(format$f.$row) $format
    if {$realexpr == {}} {
	set realexpr $expr
    }
    regsub -all % $realexpr %% myexpr
    set el [label $f.l$row -textvariable code::Debugger(expr$f.$row) -anchor e]
    bind $el <Button-3> "code::WATCHmenu %X %Y $doc \{$gridwidget\} $row \{$myexpr\}" 
    help . $el "Right click to delete or edit the watch expression"
    grid $el -in $f -row $row -column 0 -sticky ne

    set ef [frame $f.f$row]
    grid $ef -in $f -row $row -column 1 -sticky nsew
    grid columnconfigure $f 1 -weight 1
    # create the expand frame, if needed
    if {![catch {frame $ef$gridwidget} exf]} {
        grid $exf -in $ef -row 1 -column 0 -sticky nw
    }

    set ee [entry $f.f$row.e -width 0 -textvariable "code::Debugger(watch$f.$row)"]
    grid $ee -in $ef -row 0 -column 0 -sticky nsew
    grid columnconfigure $ef 0 -weight 1
    # remember watched values
    set list [$doc.watch cget -contents]
    set newlist {}
    set found 0
    foreach element $list {
	if {$row == [lindex $element 4] && $gridwidget == [lindex $element 3]} {
	    # remove old one
	    set found 1
	    set element [list $expr $format $realexpr $gridwidget $row]
	}
	lappend newlist $element
    }
    if {!$found} {
        lappend newlist [list $expr $format $realexpr $gridwidget $row]
    }
    set newlist [lsort -integer -index 4 $newlist]
    set newlist [lsort -dictionary -index 3 $newlist]
    $doc.watch configure -contents $newlist -scrollbar auto
    WATCHvalue $doc [list $expr $format $realexpr $gridwidget $row]
}

#
# WATCHregisters - watch all processor registers
#
proc code::WATCHregisters {} {
    variable CODEdocument
    variable DebuggerRegisters

    if {[catch {DebuggerDbg registers} regs]} {
	DebuggerCommError $regs
	return
    }
    foreach list $regs {
	set name [lindex $list 0]
	if {$name == {}} {
	    continue
	}
	WATCHadd $CODEdocument %$name
    }
}

#
# WATCHvalue - set the value of a watched expression
#
proc code::WATCHvalue {doc expr} {
    variable Debugger
    variable DebuggerRegisters

    set row [lindex $expr 4]
    set gridwidget [lindex $expr 3]
    # get the name of the entry
    set f $doc.watch.f$gridwidget
    set ee $f.f$row.e

    set realexpr [lindex $expr 2]
    set format [lindex $expr 1]
    set expr [lindex $expr 0]

    regexp %(.*) $expr all reg
    if {   [info exists reg]
	&& $reg != $expr
        && ![catch {DebuggerDbg expression %$reg}]} {
        # this is a register
	DebuggerGetRegisters
        set width [expr [string length $DebuggerRegisters($reg)] - 2]
	set format "0x%0${width}X"
	set Debugger(expr$f.$row) $reg
        set list [$doc.watch cget -contents]
        set newlist {}
        set found 0
        foreach element $list {
	    if {$row == [lindex $element 4] && $gridwidget == [lindex $element 3]} {
	        # remove old one
	        set found 1
	        set element [list $reg $format $expr $gridwidget $row]
	    }
	    lappend newlist $element
        }
        if {!$found} {
            lappend newlist [list $reg $format $expr $gridwidget $row]
        }
        set newlist [lsort -integer -index 4 $newlist]
        set newlist [lsort -dictionary -index 3 $newlist]
        $doc.watch configure -contents $newlist
    }
    if {$format != {} && [catch {DebuggerDbg expression $realexpr $format} value]} {
	# can't access, make into a pseudo label
	$ee configure -state disabled -relief flat 
	# in case this was a button
	bind $ee <Button-1> {}
	bind $ee <FocusIn> {}
	bind $ee <FocusOut> {}
	bind $ee <ButtonRelease-1> {}
	bind $ee <Leave> {}
    } elseif {$format == {} && [catch {DebuggerDbg expression $realexpr} value]} {
	# can't access, make into a pseudo label
	$ee configure -state disabled -relief flat 
	# in case this was a button
	bind $ee <Button-1> {}
	bind $ee <FocusIn> {}
	bind $ee <FocusOut> {}
	bind $ee <ButtonRelease-1> {}
	bind $ee <Leave> {}
    } elseif {[llength $value] > 1} {
	set type [lindex $value 0]
	# make into a pseudo button
	$ee configure -state disabled -relief raised 
	set Debugger(watch$f.down$row) 0
	bind $ee <Button-1> \
	    "set code::Debugger(watch$f.down$row) 1; $ee configure -relief sunken"
	regsub -all % $realexpr %% realexpr
	regsub -all % $format %% newformat
	# in case this was an entry
	bind $ee <FocusIn> {}
	bind $ee <FocusOut> {}
	bind $ee <Return> {}
	bind $ee <Leave> \
	    "set code::Debugger(watch$f.down$row) 0; $ee configure -relief raised"
	switch $type {
	    function {
	        set value "NOT IMPLIMENTED Press to execute $type"
	        bind $ee <ButtonRelease-1> {}
	    }
	    pointer -
	    array -
	    union -
	    structure {
		set done 0
		if {$type == "pointer"} {
	            if {[lindex $value 1] == 0} {
	                # NULL, make into a pseudo label
	                $ee configure -state disabled -relief flat
	                # in case this was a button
	                bind $ee <Button-1> {}
	                bind $ee <Leave> {}
	                bind $ee <ButtonRelease-1> {}
		        set value "NULL pointer"
		        # treat as a simple value
                        $ee configure -state normal -relief sunken 
	                bind $ee <Return> "code::WATCHchange \{$realexpr\} $doc \{$gridwidget\} $row \{$newformat\}"
	                bind $ee <FocusIn> "set code::Debugger(watch$f.$row.orig) \
	                    \$code::Debugger(watch$f.$row); \
	                set code::Debugger(format$f.$row.orig) \
	                    \$code::Debugger(format$f.$row)"
	                bind $ee <FocusOut> "code::WATCHchange \{$realexpr\} $doc \{$gridwidget\} $row \{$newformat\}"
			set done 1
	            } else {
		        set type "*$realexpr (*[format 0x%04X [lindex $value 1]])"
		    }
		}
		if {!$done && [llength [info commands $f.f$row.f*]] > 1} {
	            bind $ee <ButtonRelease-1> \
	                "$ee configure -relief raised; \
		        if {\$code::Debugger(watch$f.down$row)} \
			    { \
			    code::WATCHunnest $ee \{$type\} $doc \{$gridwidget\} $row \
			        \{$realexpr\} \{[lrange $value 1 end]\} \{$newformat\} \
			    }"
	            set value "Press to hide $type"
		} elseif {!$done} {
	            bind $ee <ButtonRelease-1> \
	                "$ee configure -relief raised; \
		        if {\$code::Debugger(watch$f.down$row)} \
			    { \
			    code::WATCHnest $ee \{$type\} $doc \{$gridwidget\} $row \
			        \{$realexpr\} \{[lrange $value 1 end]\} \{$newformat\} \
			    }"
	            set value "Press to view $type"
		}
	    }
	}
    } else {
	# a simple value
        $ee configure -state normal -relief sunken 
	regsub -all % $realexpr %% realexpr
	regsub -all % $format %% newformat
	bind $ee <Return> "code::WATCHchange \{$realexpr\} $doc \{$gridwidget\} $row \{$newformat\}"
	bind $ee <FocusIn> "set code::Debugger(watch$f.$row.orig) \
	    \$code::Debugger(watch$f.$row); \
	    set code::Debugger(format$f.$row.orig) \
	    \$code::Debugger(format$f.$row)"
	bind $ee <FocusOut> "code::WATCHchange \{$realexpr\} $doc \{$gridwidget\} $row \{$newformat\}"
	bind $ee <Leave> {}
    }
    set Debugger(watch$f.$row) $value
    set Debugger(format$f.$row) $format
}

#
# WATCHunnest - unnest a watch window
#
proc code::WATCHunnest {button type doc gridwidget row expr values format} {
    variable Debugger

    set f $doc.watch.f$gridwidget
    bind $button <ButtonRelease-1> \
        "$button configure -relief raised; \
             if {\$code::Debugger(watch$f.down$row)} \
		{ \
		code::WATCHnest $button \{$type\} $doc \{$gridwidget\} $row \
		    \{$expr\} \{$values\} \{$format\} \
		}"
    set value "Press to view $type"
    set Debugger(watch$f.$row) $value
    WATCHdelete $doc $gridwidget $row $expr 1
}

#
# WATCHnest - nest a watch window
#
proc code::WATCHnest {button type doc gridwidget row expr values format} {
    variable Debugger

    set f $doc.watch.f$gridwidget
    bind $button <ButtonRelease-1> \
        "$button configure -relief raised; \
             if {\$code::Debugger(watch$f.down$row)} \
		{ \
		code::WATCHunnest $button \{$type\} $doc \{$gridwidget\} $row \
		    \{$expr\} \{$values\} \{$format\} \
		}"
    set value "Press to hide $type"
    set Debugger(watch$f.$row) $value

    # get the row frame
    set gridwidget $gridwidget.f$row.f
    set index 0
    if {$type == "structure"} {
        foreach "field value" $values {
	    WATCHadd $doc [list $field $format ($expr).$field $gridwidget $index]
	    incr index
        }
    } elseif {$type == "union"} {
        foreach field $values {
	    WATCHadd $doc [list $field $format ($expr).$field $gridwidget $index]
	    incr index
        }
    } elseif {$type == "array"} {
	# array
	set end [lindex $values 0]
	if {$end == 0} {
	    # array of unknown size
	    set end 1
	}
	for {set index 0} {$index < $end} {incr index} {
	    WATCHadd $doc [list \[$index\] $format ($expr)\[$index\] $gridwidget $index]
	}
    } else {
	# pointer
	WATCHadd $doc [list *$expr $format *$expr $gridwidget 0]
    }
}

#
# WATCHchange - update the value of a watch variable if it was changed
#
proc code::WATCHchange {expr doc gridwidget row format} {
    variable Debugger

    set f $doc.watch.f$gridwidget
    if {   $Debugger(watch$f.$row) == $Debugger(watch$f.$row.orig) \
        && $Debugger(format$f.$row) == $Debugger(format$f.$row.orig)} {
	# no change
        return
    }
    if {   $format != {}
	&& ![catch {DebuggerDbg expression "$expr=$Debugger(watch$f.$row)" $format} msg]} {
	    set Debugger(watch$f.$row) [DebuggerDbg expression $expr $format]
            set Debugger(watch$f.$row.orig) $Debugger(watch$f.$row)
	    DebuggerState
    } elseif {   $format == {}
	      && ![catch {DebuggerDbg expression "$expr=$Debugger(watch$f.$row)"} msg]} {
	    set Debugger(watch$f.$row) [DebuggerDbg expression $expr]
            set Debugger(watch$f.$row.orig) $Debugger(watch$f.$row)
	    DebuggerState
    } else {
	# an error occured
        tk_messageBox \
            -parent . \
    	    -icon error \
	    -message "Error changing watch value: $msg." \
	    -type ok
        set Debugger(watch$f.$row) $Debugger(watch$f.$row.orig)
    }
}

#
# WATCHmenu - generate the Watch menu
#
proc code::WATCHmenu {X Y doc gridwidget row expr} {
    variable Debugger
    variable menustatus

    set f $doc.watch.f$gridwidget
    set m .watchpopup
    if {[info commands $m] == ""} {
        menu $m -tearoff 0
	bind $m <Leave> "code::status . {}"
    }
    $m delete 0 end
    $m add command -label "Edit" \
        -command "code::WATCHedit $doc \{$gridwidget\} $row \{$expr\}"
    set menustatus($m,"Edit") \
        "Edit the watch expression $Debugger(expr$f.$row)"
    set menustatus($m,Edit,top) .
    $m add command -label "Delete" \
        -command "code::WATCHdelete $doc \{$gridwidget\} $row \{$expr\}"
    set menustatus($m,"Delete") \
                "Delete the watch expression $Debugger(expr$f.$row)"
    set menustatus($m,Delete,top) .
    tk_popup $m $X $Y
}

#
# WATCHedit - edit a watch expression
#
proc code::WATCHedit {doc gridwidget row {realexpr {}}} {
    variable Debugger

    set f $doc.watch.f$gridwidget
    set both [WATCHget Edit $doc $Debugger(expr$f.$row) \
	$Debugger(format$f.$row) $realexpr]
    set expr [lindex $both 0]
    set format [lindex $both 1]
    if {$both == {} || \
	($expr == $Debugger(expr$f.$row) && $format == $Debugger(format$f.$row))} {
	# canceled or the same
	return
    }
    WATCHdelete $doc $gridwidget $row $expr
    WATCHadd $doc [list $expr $format {} $gridwidget $row]
}

#
# WATCHdelete - delete a watch from the watch window
#
proc code::WATCHdelete {doc gridwidget row expr {parent 0}} {
    variable Debugger

    set f $doc.watch.f$gridwidget
    if {!$parent} {
        unset Debugger(expr$f.$row)
        unset Debugger(format$f.$row)
        destroy $f.l$row
        destroy $f.f$row
    }
    # get gridwidget of children
    set children $gridwidget.f$row.f
    eval destroy [info commands $doc.watch.f$children*]
    set list [$doc.watch cget -contents]
    set newlist {}
    foreach element $list {
	if {!$parent && $row == [lindex $element 4] && $gridwidget == [lindex $element 3]} {
	    continue
	} elseif {[string match $children* [lindex $element 3]]} {
	    # remove children
	    continue
	}
	lappend newlist $element
    }
    set newlist [lsort -integer -index 4 $newlist]
    set newlist [lsort -dictionary -index 3 $newlist]
    $doc.watch configure -contents $newlist
}

#
# WATCHget - get a Debugger watch expression
#
proc code::WATCHget {type doc {expr {}} {format {}} {realexpr {}}} {
    variable Debugger

    if {$doc != {} && [string match Watch* [$doc cget -type]]} {
	set in " in [$doc cget -type]"
    } else {
	set in ""
    }
    set box .newwatch
    # check for a current selection
    if {$expr != {}} {
	set Debugger(watchentry) $expr
	set Debugger(watchformat) $format
    } else {
	set Debugger(watchentry) {}
	set Debugger(watchformat) {}
    }
    set title "$type watch expression$in"

    toplevel $box 
    wm transient $box .
    set f [frame $box.buttons]
    set okBtn [button $f.ok -text Ok -width 6 \
        -command code::WATCHcheck]
    set cancelBtn [button $f.cancel -text Cancel -width 6 \
        -command "set code::Debugger(watchset) {}"]
    grid $okBtn -in $f -row 0 -column 0 
    grid $cancelBtn -in $f -row 0 -column 1 
    grid $f -in $box -row 2 -column 0 -columnspan 2 -sticky ew -padx 5 -pady 5
    if {$realexpr == {} || $realexpr == $expr} {
        set e [entry $box.entry -textvariable code::Debugger(watchentry) -width 30]
        bind $e <Key-Return> code::WATCHcheck
        grid $e -in $box -row 1 -column 0 -sticky ew -padx 5 -pady 5
        help $box $e "Enter or edit the debugger watch expression here"
    } else {
        set e [label $box.entry -text $realexpr]
	grid $e -in $box -row 1 -column 0 -sticky w -padx 5 -pady 5
	help $box $e "The debugger watch expression"
    }

    set e [entry $box.format -textvariable code::Debugger(watchformat) -width 10]
    bind $e <Key-Return> code::WATCHcheck
    grid $e -in $box -row 1 -column 2 -sticky ew -padx 5 -pady 5
    help $box $e "Enter an optional printf() like format string here"
    set l [label $box.lf -text "Format:"]
    help $box $l "An optional printf() like format string"
    grid $l -in $box -row 1 -column 1 -sticky ew -padx 5 -pady 5

    placewindow $box widget .
    wm title $box $title

    set oldFocus [focus]
    set oldGrab [grab current $box]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $box
    focus $e

    tkwait variable code::Debugger(watchset)
    catch {focus $oldFocus}
    grab release $box
    destroy $box
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    if {$Debugger(watchset) == {}} {
	# cancelled
	return {}
    }
    return [list $Debugger(watchset) $Debugger(watchformat)]
}

#
# WATCHcheck - check a watch expression
#
proc code::WATCHcheck {} {
    variable Debugger

    set Debugger(watchset) $Debugger(watchentry)
}

#
# DebuggerAddPath - add a path to the directory search list
#
proc code::DebuggerAddPath {} {
    variable CODEdocument

    set path {}
    if {$CODEdocument != {} && [$CODEdocument cget -type] == "Folder"} {
	set path [$CODEdocument.dir cget -contents]
    } elseif {![catch {selection get} sel]} {
        # use current selection
	set path $sel
    }
    set path [tk_chooseDirectory -initialdir $path -parent . \
	-title "Add a Debugger Search Path"]
    if {$path == {}} {
	return
    }
    DebuggerDbg path $path
}

#
# DebuggerCheckPath - check a path name for validity
#
proc code::DebuggerCheckPath {w} {
    variable Debugger

    if {$Debugger(pathentry) != {} && ![file isdirectory $Debugger(pathentry)]} {
        tk_messageBox \
            -parent $w \
    	    -icon error \
	    -message "Directory does not exist: $Debugger(pathentry)" \
	    -type ok
	return
    }
    set Debugger(path) $Debugger(pathentry)
}

#
# DebuggerPathMenu - create the Path menu
#
proc code::DebuggerPathMenu {m} {
    if {![winfo exists $m]} {
        menu $m -tearoff 0 -postcommand "code::DebuggerPathMenuBuild $m"
    }
    return $m
}

#
# DebuggerPathMenuBuild - generate the Path menu
#
proc code::DebuggerPathMenuBuild {m} {
    variable menustatus

    $m delete 0 end
    set name "Add Path..."
    $m add command -label $name -command "code::DebuggerAddPath"
    set menustatus(.,$name) "Add a folder to the debugger search list"
    set list [DebuggerDbg path]
    if {$list == {}} {
        return
    }
    $m add separator
    foreach path $list {
	set name [file nativename $path]
        $m add command -label $name -command "code::DIRECTORY [list $path]"
        set menustatus(.,$name) "Open this folder"
    }
}
        
#
# DebuggerComplexMenu - build create the Complex Breakpoint Menu
#
proc code::DebuggerComplexMenu {m} {
    if {![winfo exists $m]} {
        menu $m -tearoff 0 -postcommand "code::DebuggerComplexMenuBuild $m"
    }
    return $m
}

#
# DebuggerComplexMenuBuild - generate the Complex Breakpoint menu
#
proc code::DebuggerComplexMenuBuild {m} {
    variable Debugger 
    variable DebuggerBreakpoints

    $m delete 0 end
    set list [lsort -dictionary [array names DebuggerBreakpoints]]
    if {$list == {}} {
	$m add command \
	    -label "No complex breakpoints are defined."
        return $m
    }
    foreach break $list {
	set value $DebuggerBreakpoints($break)
	set name "[lindex $value 0] [lindex $value 1]"
	if {[lindex $value 2] != {} || [lindex $value 3] != {}} {
	    append name { (more)}
	}
	if {![lindex $value 5]} {
	    $m add command -label "$name..." \
	        -command "code::DebuggerComplexBreakpoint $break 0" \
		-foreground grey
	} else {
	    $m add command -label "$name..." \
	        -command "code::DebuggerComplexBreakpoint $break 0"
	}
    }
    return $m
}
        
#
# DebuggerComplexBreakpoint - set/edit an complex breakpoint
#
proc code::DebuggerComplexBreakpoint {{id {}} {hit 1}} {
    variable Debugger
    variable DebuggerBreakpoints

    set box .breakpoint
    toplevel $box 
    wm transient $box .
    wm withdraw $box

    if {$id != {}} {
	set value $DebuggerBreakpoints($id)
        set Debugger(breakpoint,type) [lindex $value 0]
        set Debugger(breakpoint,address) [lindex $value 1]
        set Debugger(breakpoint,condition) [lindex $value 2]
        set Debugger(breakpoint,command) [lindex $value 3]
	set Debugger(breakpoint) [lindex $value 4]
        set Debugger(breakpoint,enabled) [lindex $value 5]
        set Debugger(breakpoint,stop) [lindex $value 6]
	if {$hit} {
	    set mode "Stopped At"
	    set Debugger(pc) 0
	    DebuggerState 0 "Complex breakpoint [lindex $value 1] occured"
	} else {
	    set mode Edit
	}
    } else {
	set mode New
	set Debugger(breakpoint,enabled) 1
        set Debugger(breakpoint,type) at
        set Debugger(breakpoint,address) {}
        set Debugger(breakpoint,condition) {}
        set Debugger(breakpoint,command) {}
	set Debugger(breakpoint) {}
        set Debugger(breakpoint,stop) 1
    }
    if {$Debugger(breakpoint,stop)} {
	# remove added stopping code
	regexp {(.*)\n# CODE.*# CODE\n} $Debugger(breakpoint,command) all \
	    Debugger(breakpoint,command)
    }
    # remove error catching code
    regexp {if \{\[catch \{\n(.*)\n\} code::breakpointerror\]\}.*} \
	$Debugger(breakpoint,command) all Debugger(breakpoint,command)
    
    bind $box <F1> "code::OpenHelp Applications Debugger Breakpoint.html"
    set f [frame $box.buttons]
    set okBtn [button $f.ok -text Ok -width 6 \
        -command "code::DebuggerCheckBreakpoint \"$id\""]
    grid $okBtn -in $f -row 0 -column 0 
    set cancelBtn [button $f.cancel -text Cancel -width 6 \
        -command "set code::Debugger(breakpoint) {}"]
    grid $cancelBtn -in $f -row 0 -column 1 
    if {$mode != "New"} {
        set deleteBtn [button $f.delete -text Delete -width 6 \
            -command "code::DebuggerDeleteBreakpoint \"$id\""]
        grid $deleteBtn -in $f -row 0 -column 2 
    }
    grid $f -in $box -row 2 -column 0 -sticky ew -padx 5 -pady 5 

    set f [frame $box.content -bd 2 -relief groove]
    set label [label $f.lab]
    grid $label -in $f -row 5 -column 1 -columnspan 3 -sticky ew -pady 5

    set b $f.m
    set m [tk_optionMenu $b \
	code::Debugger(breakpoint,type) at read write access clock enabled]
    grid $b -in $f -row 1 -column 1 -sticky ew -pady 5
    bind $b <Enter> "$label config -text {Select the breakpoint type}"
    bind $b <Leave> "$label config -text {}"
    set e [entry $f.adr -textvariable code::Debugger(breakpoint,address)]
    grid $e -in $f -row 1 -column 2 -sticky ew -columnspan 2
    bind $e <Enter> "$label config -text {Enter the breakpoint target}"
    bind $e <Leave> "$label config -text {}"

    set l $f.l
    set l [label $f.l -text If:]
    grid $l -in $f -row 2 -column 1 -sticky e -pady 5
    set e [entry $box.cond -textvariable code::Debugger(breakpoint,condition)]
    grid $e -in $f -row 2 -column 2 -sticky ew -columnspan 2
    bind $e <Enter> "$label config \
	-text {Enter the breakpoint condition (optional)}"
    bind $e <Leave> "$label config -text {}"

    set l $f.l
    set l [label $f.lcmd -text Command: -width 12]
    grid $l -in $f -row 3 -column 1 -sticky e -pady 5
    set e [entry $box.cmd -width 40 \
	-textvariable code::Debugger(breakpoint,command)]
    bind $e <Enter> "$label config -text {Enter the breakpoint command (optional)}"
    bind $e <Leave> "$label config -text {}"
    grid $e -in $f -row 3 -column 2 -sticky ew
    set title "Complex Breakpoint Command"
    set b [button $f.b -text Edit \
	-command "code::EDITvariable code::Debugger(breakpoint,command) {} \{$title\}"]
    grid $b -in $f -row 3 -column 3 -sticky ew -padx 5 -pady 5
    bind $b <Enter> "$label config -text {Use the Editor to edit the command}"
    bind $b <Leave> "$label config -text {}"

    set cf [frame $f.cf]
    grid $cf -in $f -row 4 -column 1 -columnspan 3 -sticky ew -pady 5

    set c [checkbutton $cf.c -text "Enabled" \
	-variable code::Debugger(breakpoint,enabled)]
    bind $c <Enter> "$label config -text {Enable or disable this breakpoint}"
    bind $c <Leave> "$label config -text {}"
    grid $c -in $cf -row 0 -column 1 -sticky ew

    set c [checkbutton $cf.s -text "Stop" \
	-variable code::Debugger(breakpoint,stop)]
    bind $c <Enter> "$label config -text {Stop program if this breakpoint occurs}"
    bind $c <Leave> "$label config -text {}"
    grid $c -in $cf -row 0 -column 2 -sticky ew

    grid columnconfigure $f 1 -weight 1
    grid $f -in $box -row 1 -column 0 -padx 5 -pady 5

    placewindow $box widget .
    wm title $box "$mode Complex Breakpoint"
    wm protocol $box WM_DELETE_WINDOW "set code::Debugger(breakpoint) {}"

    while {1} {
        tkwait variable code::Debugger(breakpoint)
	set doc [CODEfind "Edit Value" $title]
	if {$doc != {}} {
	    $doc.contents destroy $doc
	} else {
            destroy $box
	    break
	}
    }
    if {$hit} {
	ToolGenerate toolbar
    }
}

#
# DebuggerDeleteBreakpoint - delete a breakpoint
#
proc code::DebuggerDeleteBreakpoint {id {remember 1}} {
    variable Debugger
    variable DebuggerBreakpoints

    if {$id == {}} {
	return
    }
    set value $DebuggerBreakpoints($id)
    if {[lindex $value 5]} {
	# breakpoint is enabled
        DebuggerDbg breakpoint delete [lindex $value 4]
    }
    unset DebuggerBreakpoints($id)
    set Debugger(breakpoint) {}
    if {$remember} {
        PreferenceSetIfChanged DebuggerInternal breakpoints [array get DebuggerBreakpoints]
    }
}

#
# DebuggerCheckBreakpoint - check a breakpoint for validity
#
proc code::DebuggerCheckBreakpoint {id} {
    variable Debugger

    if {$Debugger(breakpoint,address) == {}} {
	# nothing here
	return
    }
    set result [DebuggerSetBreakpoint $Debugger(breakpoint,type) \
        $Debugger(breakpoint,address) 0 $Debugger(breakpoint,enabled) \
	$Debugger(breakpoint) $Debugger(breakpoint,condition) \
	$Debugger(breakpoint,command) $id $Debugger(breakpoint,stop)]
    if {$result == "error"} {
        # do nothing
    } else {
        set Debugger(breakpoint) $result
    }
}

#
# DebuggerSetBreakpoint - add a breakpoint to the breakpoint list,
#   set if enabled
#
proc code::DebuggerSetBreakpoint {type address {simpleat 1} {enabled 1} 
        {dbgid {}} {condition {}} {command {}} {id {}} {stop 1} {noedit 0}} {
    variable Debugger
    variable DebuggerBreakpoints
    variable DebuggerAtBreakpoints

    if {$type == "widget"} {
	# breakpoint a specific line in an editor
	set t $address
	set index [$t index insert]
	# check for debugger context
        if {[lsearch [$t tag names insert] "disassembly"] != -1 } {
            # a disassembly line, get address
	    set address &0x[$t get "insert linestart" "insert lineend"]
	    regexp {([^:]*):.*} $address all address
	} else {
	    regexp {([0-9]+)\.[0-9]+} $index all line
	    set line [EDITrealline $t $line]
            set address '[$t cget -contents]'@$line
	    if {[catch {DebuggerDbg where $address}]} {
	        # not a context
	        return
	    }
	}
	# always an execution breakpoint
	set type at
    } elseif {$type == "selection"} {
	# breakpoint a selection
	set t $address
	if {   ![catch {winfo parent [selection own -displayof $t]} tmp]
	    && $tmp == "$t.text"
	    && ![catch {selection get -displayof $t} selection]
	    && $selection != {}} {
	} else {
	    return
	}
	set loc {}
	# search static context
	catch {set loc [DebuggerDbg where '$Debugger(source)'.$selection]}
	if {$loc == {}} {
	    # search global context
	    catch {set loc [DebuggerDbg where $selection]}
	}
	set selfile [lindex $loc 0]
	set selline [lindex $loc 1]
	# check for debugger context
	if {$selfile == {}} {
	    return
	}
        set address '$selfile'@$selline
	if {[catch {DebuggerDbg where $address}]} {
	    # not a context
	    return
	}
	# always an execution breakpoint
	set type at
    } elseif {$type == "line"} {
	# breakpoint a specific line
        set t [lindex $address 0]
	set line [lindex $address 1]
	# check for debugger context
	regexp {([0-9]+)\.[0-9]+} $line all line
	set line [EDITrealline $t $line]
        set address '[$t cget -contents]'@$line
	if {[catch {DebuggerDbg where $address}]} {
	    # not a context
	    return
	}
	# always an execution breakpoint
	set type at
    }
    if {$type != "at"} {
	set simpleat 0
    } 
    set cmd "breakpoint $type $address "
    if {$condition != {}} {
	set simpleat 0
        append cmd "-if [list $condition] "
    }
    if {$command != {} || (!$simpleat && $stop)} {
	set simpleat 0
        if {$stop} {
	    # add the stopping code
	    if {$condition != {}} {
		set cond "\nif $condition"
	    } else {
		set cond ""
	    }
            if {$id == {}} {
	        # get a unique id
	        set id 0
	        while {[info exists DebuggerBreakpoints($id)]} {
		    incr id
	        }
            }
	    if {$command != {}} {
		    regsub {\$} $cond {\\$} mycond
		    set command "if {\[catch {\n$command\n} code::breakpointerror]} {\n    tk_messageBox -parent . -icon error -message \"Error in Complex Breakpoint command:\n$address$mycond\n\$code::breakpointerror\" -type ok\n}\n"
	    }
	    set message "code::DebuggerDbg stop; after idle code::DebuggerComplexBreakpoint $id"
	    append command "\n# CODE\n$message\n# CODE\n"
        }
        append cmd "-command [list $command] "
    }
    if {$simpleat} {
	# make sure we have a 'file'@line form
	set where [DebuggerDbg where $address]
	set file [lindex $where 0]
	set line [lindex $where 1]
	if {![file exists $file]} {
	    catch {DebuggerDbg path -find $file} file
	}
        if {[file dirname $file] == "."} {
            set file [file join [pwd] $file]
        } 
	set address "'$file'@$line"
	# toggle a simple at breakpoint
	# toggles from set to clear to remove
	if {[info exists DebuggerAtBreakpoints($address)]} {
	    # this one exists
	    set dbgid $DebuggerAtBreakpoints($address)
	    if {![string equal $dbgid {}]} {
		set enabled 0
	    } else {
	       # was disabled, remove
	       unset DebuggerAtBreakpoints($address)
	       DebuggerUpdateBreakpoints $address {}
	       EDITshowbreak $address
	       return
	    }
	}
        DebuggerUpdateBreakpoints $address $enabled
        # otherwise fall through to set
    }
    if {$dbgid != {}} {
         DebuggerDbg breakpoint delete $dbgid
         set dbgid {}
    }
    if {!$enabled} {
        set dbgid {}
    } elseif {[catch {eval DebuggerDbg $cmd} msg]} {
        tk_messageBox \
	    -parent . \
	    -icon error \
	    -message "Can't set breakpoint: $msg" \
	    -type ok
        return  error
    } else {
	set dbgid $msg
    }
    if {!$simpleat} {
        if {$id == {}} {
	    # get a unique id
	    set id 0
	    while {[info exists DebuggerBreakpoints($id)]} {
		incr id
	    }
        }
        set info [DebuggerDbg where $address]
        set file [lindex $info 0]
        set line [lindex $info 1]
        set function [lindex $info 5]
        set symbol [lindex $info 6]
        set offset [lindex $info 7]
        set pc [lindex $info 8]
	if {$type == "at"} {
            if {   $function != {}
	        && ![catch {DebuggerDbg where $function} myinfo]
	        && $myinfo == $info} {
	        # this is at a global function symbol
	        set where $function
            } elseif {   $function != {}
	        && ![catch {DebuggerDbg where '$file'.$function} myinfo]
	        && $myinfo == $info} {
	        # this is at a static function symbol
	        set where '$file'.$function
            } elseif {$symbol != {} && $offset == 0} {
	        set where $symbol
            } elseif {$file != {}} {
	        # use simplest form of name
	        set where '$file'@$line
            }
	}
        set DebuggerBreakpoints($id) \
	    [list $type $address $condition $command $dbgid $enabled $stop]
        PreferenceSetIfChanged DebuggerInternal breakpoints [array get DebuggerBreakpoints]
	if {!$noedit && [info commands .breakpoint] == {}} {
	    # edit this new complex breakpoint
	    DebuggerComplexBreakpoint $id 0
	}
    } else {
        set DebuggerAtBreakpoints($address) $dbgid
	EDITshowbreak $address
    }
    return $dbgid
}

#
# DebuggerUpdateBreakpoints - update the at breakpoint list
#
proc code::DebuggerUpdateBreakpoints {where newstate} {
    set info [DebuggerDbg where $where]
    set file [lindex $info 0]
    set line [lindex $info 1]
    set function [lindex $info 5]
    set symbol [lindex $info 6]
    set offset [lindex $info 7]
    set pc [lindex $info 8]
    if {   $function != {}
	&& ![catch {DebuggerDbg where $function} myinfo]
	&& $myinfo == $info} {
	# this is at a global function symbol
	set where $function
    } elseif {   $function != {}
	&& ![catch {DebuggerDbg where '$file'.$function} myinfo]
	&& $myinfo == $info} {
	# this is at a static function symbol
	set where '$file'.$function
    } elseif {$symbol != {} && $offset == 0} {
	set where $symbol
    } elseif {$file != {}} {
	# use simplest form of name
	set where '$file'@$line
    }
    set done 0
    set newlist {}
    foreach "break state" [Preference DebuggerInternal atbreakpoints] {
	if {$break == $where} {
	    # this is it
	    set done 1
	    if {$newstate == {}} {
		# remove
		continue
	    }
	    set state $newstate
	}
	lappend newlist $break $state
    }
    if {!$done && $newstate != {}} {
       # add new entry
       lappend newlist $where $newstate
    }
    PreferenceSetIfChanged DebuggerInternal atbreakpoints $newlist
}

#
# DebuggerAtBreakpoints - actions performed on all debugger at breakpoints
#
proc code::DebuggerAtBreakpoints {command {remember 1}} {
    variable DebuggerAtBreakpoints

    if {$remember && [Preference DebuggerInternal atmain]} {
        # don't modify breakpoint at main
	set where [DebuggerDbg where main]
	set file [lindex $where 0]
	set line [lindex $where 1]
	if {![file exists $file]} {
	    catch {DebuggerDbg path -find $file} file
	}
        if {[file dirname $file] == "."} {
            set file [file join [pwd] $file]
        } 
	set main "'$file'@$line"
    } else {
	set main {}
    }
    foreach address [array names DebuggerAtBreakpoints] {
	set id $DebuggerAtBreakpoints($address)
        switch $command {
	    remove {
		if {$remember && $address == $main} {
		    continue
		}
	        if {![string equal $id {}]} {
                    DebuggerDbg breakpoint delete $id
	        }
	        unset DebuggerAtBreakpoints($address)
		if {$remember} {
	            DebuggerUpdateBreakpoints $address {}
		}
	        EDITshowbreak $address
	    }
	    inactivate {
		if {$address == $main} {
		    continue
		}
	        if {![string equal $id {}]} {
                    DebuggerDbg breakpoint delete $id
		    set DebuggerAtBreakpoints($address) {}
	            EDITshowbreak $address
		    if {$remember} {
	                DebuggerUpdateBreakpoints $address 0
		    }
	        }
	    }
	    activate {
	        if {[string equal $id {}]} {
                    set cmd "breakpoint at $address"
                    if {[catch {eval DebuggerDbg $cmd} dbgid]} {
                        tk_messageBox \
			    -parent . \
	                    -icon error \
	                    -message "Can't set breakpoint: $dbgid" \
	                    -type ok
			return
                    }
                    set DebuggerAtBreakpoints($address) $dbgid
	            EDITshowbreak $address
		    if {$remember} {
	                DebuggerUpdateBreakpoints $address 1
		    }
		}
	    }
	}
    }
}

#
# DebuggerComplexBreakpoints - actions performed on all debugger breakpoints
#
proc code::DebuggerComplexBreakpoints {mycommand {remember 1}} {
    variable DebuggerBreakpoints

    foreach breakid [array names DebuggerBreakpoints] {
	set info $DebuggerBreakpoints($breakid)
	set type [lindex $info 0]
	set address [lindex $info 1]
	set condition [lindex $info 2]
	set command [lindex $info 3]
	set dbgid [lindex $info 4]
	set enabled [lindex $info 5]
	set stop [lindex $info 6]
        switch $mycommand {
	    remove {
               DebuggerDeleteBreakpoint $breakid $remember
	    }
	    inactivate {
		if {$dbgid != {}} {
	            DebuggerSetBreakpoint $type $address 0 0 $dbgid \
	                $condition $command $breakid $stop 1
		}
	    }
	    activate {
		if {$dbgid == {}} {
	            DebuggerSetBreakpoint $type $address 0 0 {} \
	                $condition $command $breakid $stop 1
		}
	    }
	}
    }
    if {$remember} {
        PreferenceSetIfChanged DebuggerInternal breakpoints [array get DebuggerBreakpoints]
    }
}

#
# DebuggerGoto - go to a specified target
#
proc code::DebuggerGoto {} {
    variable Debugger

    set box .gototarget

    # check for a current selection
    if {![catch {selection get} sel]} {
	set Debugger(addressentry) $sel
    }
    toplevel $box 
    wm transient $box .
    wm protocol $box WM_DELETE_WINDOW "set code::Debugger(address) {}"
    set f [frame $box.buttons]
    set okBtn [button $f.ok -text Ok -width 6 \
        -command "code::DebuggerCheckAddress $box"]
    set cancelBtn [button $f.cancel -text Cancel -width 6 \
        -command "set code::Debugger(address) {}"]
    grid $okBtn -in $f -row 0 -column 0 
    grid $cancelBtn -in $f -row 0 -column 1 
    grid $f -in $box -row 2 -column 0 -sticky ew -padx 5 -pady 5
    set e [entry $box.entry -textvariable code::Debugger(addressentry) -width 30]
    bind $e <Key-Return> "code::DebuggerCheckAddress $box"
    grid $e -in $box -row 1 -column 0 -sticky ew -padx 5 -pady 5
    set l [label $box.l -text "Enter the goto target"]
    grid $l -in $box -row 0 -column 0 -sticky ew -padx 5 -pady 5

    placewindow $box widget .
    wm title $box Goto
    bind $box <F1> "code::OpenHelp Applications Debugger Goto.html"

    set oldFocus [focus]
    set oldGrab [grab current $box]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $box
    focus $e

    tkwait variable code::Debugger(address)
    catch {focus $oldFocus}
    grab release $box
    destroy $box
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    if {$Debugger(address) == {}} {
	return
    }
    set Debugger(stopmsg) "Stopped at $Debugger(address)"
    DebuggerGo go $Debugger(address)
}

#
# DebuggerCheckAddress - check an address for validity
#
proc code::DebuggerCheckAddress {w} {
    variable Debugger

    if {$Debugger(addressentry) != {} && \
	[catch {DebuggerDbg where $Debugger(addressentry)} msg]} {
        tk_messageBox \
	    -parent $w \
    	    -icon error \
	    -message "$Debugger(addressentry) is not valid: $msg" \
	    -type ok
	return
    }
    set Debugger(address) $Debugger(addressentry)
}

#
# DebuggerSetPC - set the value of the PC
#
proc code::DebuggerSetPC {type t} {
    variable Debugger

    if {$type == "widget"} {
	# setting PC to a specific line
	set index [$t index insert]
	# check for debugger context
        if {[lsearch [$t tag names insert] "disassembly"] != -1 } {
            # a disassembly line, get address
	    set where 0x[$t get "insert linestart" "insert lineend"]
	    regexp {([^:]*):.*} $where all where
	} else {
	    # a source line
	    regexp {([0-9]+)\.[0-9]+} $index all line
	    set line [EDITrealline $t $line]
            set where '[$t cget -contents]'@$line
	    if {[catch {DebuggerDbg where $where}]} {
	        # not a context
	        return
	    }
	}
    } elseif {$type == "selection"} {
	# breakpoint a selection
	if {   ![catch {winfo parent [selection own -displayof $t]} tmp]
	    && $tmp == "$t.text"
	    && ![catch {selection get -displayof $t} selection]
	    && $selection != {}} {
	} else {
	    return
	}
	set loc {}
	# search static context
	catch {set loc [DebuggerDbg where '$Debugger(source)'.$selection]}
	if {$loc == {}} {
	    # search global context
	    catch {set loc [DebuggerDbg where $selection]}
	}
	set selfile [lindex $loc 0]
	set selline [lindex $loc 1]
	# check for debugger context
	if {$selfile == {}} {
	    return
	}
        set where [list '$selfile'@$selline]
	if {[catch {DebuggerDbg where $where}]} {
	    # not a context
	    return
	}
    }
    DebuggerDbg expression %pc=$where
    DebuggerState
}

#
# DebuggerGo - start or single-step the target
#
proc code::DebuggerGo {type {args {}}} {
    variable Debugger

    if {!$Debugger(loaded) && [Preference Debugger autoload]} {
        # autoload
	status . "Auto Load"
	if {![DebuggerLoad]} {
	    return 0
	}
    }
    if {$type == "widget"} {
	# going to a specific line
	set t [lindex $args 0]
	set index [$t index insert]
	# check for debugger context
        if {[lsearch [$t tag names insert] "disassembly"] != -1 } {
            # a disassembly line, get address
	    set where &0x[$t get "insert linestart" "insert lineend"]
	    regexp {([^:]*):.*} $where all where
	} else {
	    regexp {([0-9]+)\.[0-9]+} $index all line
	    set line [EDITrealline $t $line]
            set where '[$t cget -contents]'@$line
	    if {[catch {DebuggerDbg where $where}]} {
	        # not a context
	        return
	    }
	}
	set type go
	set args [list $where]
    } elseif {$type == "selection"} {
	# breakpoint a selection
	set t $args
	if {   ![catch {winfo parent [selection own -displayof $t]} tmp]
	    && $tmp == "$t.text"
	    && ![catch {selection get -displayof $t} selection]
	    && $selection != {}} {
	} else {
	    return
	}
	set loc {}
	# search static context
	catch {set loc [DebuggerDbg where '$Debugger(source)'.$selection]}
	if {$loc == {}} {
	    # search global context
	    catch {set loc [DebuggerDbg where $selection]}
	}
	set selfile [lindex $loc 0]
	set selline [lindex $loc 1]
	# check for debugger context
	if {$selfile == {}} {
	    return
	}
        set args [list '$selfile'@$selline]
	if {[catch {DebuggerDbg where $args}]} {
	    # not a context
	    return
	}
	# always an execution breakpoint
	set type go
    }
    DebuggerState 1 "Program running"
    if {[catch "eval DebuggerDbg $type $args" msg]} {
        tk_messageBox \
	    -parent . \
    	    -icon error \
	    -message "Can't $type: $msg." \
	    -type ok
	set msg {}
    }
    if {$msg == {}} {
	set msg $Debugger(stopmsg)
    }
    set Debugger(stopmsg) {}
    DebuggerState 0 $msg
}

#
# DebuggerStop - stop the target
#
proc code::DebuggerStop {} {
    variable Debugger

    set Debugger(stopping) 1
    DebuggerDbg stop
}

#
# DebuggerState - update the state of the target
#
proc code::DebuggerState {{running {}} {msg {}}} {
    variable Debugger

    set Debugger(notstate) 0
    ToolGenerate toolbar
    if {$running != {}} {
        set Debugger(running) $running
    } else {
	set Debugger(changed) 0
        set running $Debugger(running)
    }

    if {!$running} {
	# target is stopped

	set Debugger(stopping) 0
	if {![catch {DebuggerDbg where &%pc} where]} {
	    DebuggerWhere $where
	    set pc [lindex $Debugger(context) 8]
	} else {
	    # an error occured getting the pc
	    set pc $Debugger(pc)
	}
	if {$pc != $Debugger(pc)} {
	    set Debugger(pc) $pc
	    set file [lindex $Debugger(context) 0]
	    set line [lindex $Debugger(context) 1]
	    if {$file != ""} {
	        if {[catch {DebuggerDbg path -find $file} file]} {
	            set file ""
	        }
	    } 
	    set Debugger(mixed) 0
	    if {$file == ""} {
	        # no source file available
	        set Debugger(source) ""
	        set Debugger(line) 1.0
	        set Debugger(tagstart) {}
	        set Debugger(tagend) {}
	        set Debugger(disassembly) {}
	        if {$Debugger(opened) != {}} {
                    tk_messageBox \
	                -parent . \
    	                -icon info \
	    	        -message "No source file exists for PC=[format 0x%04X $Debugger(pc)]." \
	    	        -type ok
	        }
	    } else {
                if {[file dirname $file] == "."} {
                    set file [file join [pwd] $file]
                } 
	        set Debugger(mixed) $Debugger(domixed)
	        set Debugger(source) $file
	        set Debugger(line) "$line.0"
	        set Debugger(tagstart) "$line.0"
	        set Debugger(tagend) "$line.end + 1c"
	        set Debugger(disassembly) {}
	        set start [lindex $where 3]
		if {$start != $Debugger(PC)} {
		    # auto mixed
		    set Debugger(mixed) $Debugger(automixed)
		}
		if {$Debugger(mixed)} {
	            set end [lindex $where 4]
	            if {$end != $start} {
	                catch {DebuggerDbg disassemble $start $end} Debugger(disassembly)
	            }
		}
	        # open an editor window with the source
	        set doc [EDITexisting $file $line.0]
	    }
        } else {
	    # the pc hasn't changed
	    set Debugger(pcchanged) 0
        }

	# XXX update counters
	# foreach counter $Debugger(counters) {
	 #    if {[catch {eval [lindex $counter 1]} value]} {
	        # can't get this counter anymore (probably changed
	        # targets)

	  #       set value "NA"
	   #  }
	    # set Debugger([lindex $counter 0]) $value
	# }
	# update other state related things
	foreach module $Debugger(modules) {
	    if {[info commands [set module]update] != {}} {
	        [set module]update
	    }
	}
    } else {
	# running
	set Debugger(pc) 0
    }

    # inform the interface of a state change
    event generate . <<State>>
    set Debugger(notstate) 1
    ToolGenerate toolbar
    set Debugger(pcchanged) 1
    set Debugger(newfile) 0
    status . $msg
    CODEtitle
}

#
# DebuggerSetAddress - set the address of a display window
#
proc code::DebuggerSetAddress {} {
    variable Debugger
    variable CODEdocument

    set doc $CODEdocument
    set box .setaddress
    set info [$doc.contents cget -contents]
    set Debugger(displayauto) [lindex $info 0]
    set Debugger(addressentry) [lindex $info 1]
    toplevel $box 
    wm transient $box .
    wm protocol $box WM_DELETE_WINDOW "set code::Debugger(address) {}"
    set f [frame $box.buttons]
    set okBtn [button $f.ok -text Ok -width 6 \
        -command "code::DebuggerCheckAddress $doc"]
    set cancelBtn [button $f.cancel -text Cancel -width 6 \
        -command "set code::Debugger(address) {}"]
    grid $okBtn -in $f -row 0 -column 0 
    grid $cancelBtn -in $f -row 0 -column 1 
    grid $f -in $box -row 3 -column 0 -sticky ew -padx 5 -pady 5
    set e [entry $box.entry -textvariable code::Debugger(addressentry) -width 30]
    bind $e <Key-Return> "code::DebuggerCheckAddress $doc"
    grid $e -in $box -row 1 -column 0 -sticky ew -padx 5 -pady 5
    set l [label $box.l -text "Enter the [$doc cget -type] window address (constant or expression)"]
    grid $l -in $box -row 0 -column 0 -sticky ew -padx 5 -pady 5
    set c [checkbutton $box.auto -variable code::Debugger(displayauto) \
	-text "Auto reload on state change"]
    grid $c -in $box -row 2 -column 0 -sticky ew -padx 5 -pady 5

    placewindow $box widget .
    wm title $box "Address"

    set oldFocus [focus]
    set oldGrab [grab current $box]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $box
    focus $e

    tkwait variable code::Debugger(address)
    catch {focus $oldFocus}
    grab release $box
    destroy $box
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    if {$Debugger(address) == {}} {
	return
    }
    $doc.contents configure \
	-contents [list $Debugger(displayauto) $Debugger(address)]
    DEBUGupdate $doc 1
}

#
# DISASSEMBLEstartup - return a command to recreate a disassembly window
#
proc code::DISASSEMBLEstartup {doc} {
    return "DISASSEMBLE [list [$doc.contents cget -contents]]"
}

#
# DISASSEMBLE - create a disassembly window
#
proc code::DISASSEMBLE {{state {1 %pc}} args} {
    variable Debugger

    set title "Disassembly"
    set doc [eval TEXT $args]
    set t $doc.contents
    $doc configure -type Disassembly -startupproc "code::DISASSEMBLEstartup"
    set image [CODEicon efile.icon]
    $doc configure -image $image -icontext $title -title $title
    SearchUpdate $doc $doc.contents $title $title 0
    $t configure -contents $state -positionproc "code::EDITposition $doc" \
	-scrollbar on -insertproc code::DEBUGnull -deleteproc code::DEBUGnull
    $t.scrv configure -command "code::DEBUGscroll $doc"
    $t.text configure -yscroll "code::DEBUGscroll $doc set"
    bind $doc <<State>> "code::DEBUGupdate $doc; break"
    update idletasks
    DEBUGupdate $doc 1
    bind $doc <Configure> "code::DEBUGupdateschedule $doc"
    $doc raise
    return $doc
}

#
# TRACEcontrol - turn tracing on and off
#
proc code::TRACEcontrol {} {
    variable Debugger

    if {$Debugger(traceenable)} {
	 set Debugger(tracesize) [DebuggerDbg tracebuffer on $Debugger(tracesize)]
    } else {
	 DebuggerDbg tracebuffer off
    }
    PreferenceSetIfChanged DebuggerInternal traceenable $Debugger(traceenable)
}

#
# TRACEtoolbar - add a trace command toolbar
#
proc code::TRACEtoolbar {} {
    set toolbar {
        top {
            TraceGoBack TraceStepBack TraceNextBack TraceIstepBack {}
	    TraceGoForward TraceStepForward TraceNextForward TraceIstepForward
	    }
    }

    Preference Control toolbars [concat $toolbar [Preference Control toolbars]]
}

#
# TRACEstartup - return a command to recreate the trace window
#
proc code::TRACEstartup {doc} {
    return "TRACE [list [$doc.contents cget -contents]]"
}

#
# TRACE - create a trace window
#
proc code::TRACE {{state {1 current}} args} {
    variable Debugger

    set type "Trace"
    set name "Trace Buffer"
    set doc [CODEfind $type $state]
    if {$doc != {}} {
	# have a trace window, raise and return
	if {   [$doc cget -state] == "minimized" 
	    || [$doc cget -state] == "withdrawn"} {
	    $doc configure -state normal
	}
	$doc raise
	return
    }
    set doc [eval TEXT $args]
    set t $doc.contents
    $doc configure -type $type -startupproc "code::TRACEstartup"
    set image [CODEicon efile.icon]
    $doc configure -image $image -icontext $type -title $name
    SearchUpdate $doc $doc.contents $name $name 0
    $t configure -contents $state -positionproc "code::EDITposition $doc" \
	-scrollbar on -insertproc code::DEBUGnull -deleteproc code::DEBUGnull
    $t.scrv configure -command "code::DEBUGscroll $doc"
    $t.text configure -yscroll "code::DEBUGscroll $doc set"
    bind $doc <<State>> "code::DEBUGupdate $doc; break"
    update idletasks
    DEBUGupdate $doc 1
    bind $doc <Configure> "code::DEBUGupdateschedule $doc"
    $doc raise
    return $doc
}

#
# DEBUGnull - a donothing insert or delete proc (makes window readonly)
#
proc code::DEBUGnull {args} {
    return 0; # don't update widget
}

#
# DEBUGaddress - get the address represended by a cursor position in a display
# window
#
proc code::DEBUGaddress {doc where {inbyte {}}} {
    # get the address at the beginning of the line
    set where [$doc.contents index $where]
    set line [$doc.contents get "$where linestart" "$where lineend"]
    set endadr [string first ":" $line]
    set enddata [string first "   " $line]
    set address {}
    regexp {([^:]*):.*} 0x$line all address
    if {$address == {}} {
	return {}
    }
    if {[$doc cget -type] == "Memory"} {
	# add the byte offset
        regexp {([^.]*)\.(.*)} $where all junk column
	if {$inbyte != {}} {
	    # let caller know byte number
	    upvar $inbyte indata
	}
	# not data byte yet
	set indata {}
	set eoc 0; # end of column flag
	if {$column > ($endadr + 1) && $column <= $enddata} {
	    # in HEX area
	    if {$column == $enddata} {
		set eoc 1
	    }
	    incr endadr 2; # point past ": "
	    set indata [expr ($column - $endadr) / 3]
	    # calculate byte column (0..2)
	    set bcol [expr 2 - (($column - $endadr) % 3)]
	    set len [expr [string length $address] - 2]
	    set address [format "0x%0${len}X" [expr $address + $indata]]
	    # inform inserter
	    set indata [list $address $bcol $eoc HEX]
	} elseif {$column >= $enddata + 3} {
	    # in ASCII area
	    set indata [expr ($column - ($enddata + 3))]
	    set len [expr [string length $address] - 2]
	    # byte column (0..1)
	    set bcol 2
	    if {[$doc.contents compare $where == "$where lineend"]} {
		# last byte in line
		set bcol 0
		set eoc 1
		incr address -1
	    }
	    set address [format "0x%0${len}X" [expr $address + $indata]]
	    # inform inserter
	    set indata [list $address $bcol $eoc ASCII]
	}
    }
    return $address
}

#
# DEBUGupdate - update a display window
#
proc code::DEBUGupdate {doc {force 0} {first {}} {last {}}} {
    variable Debugger

    set type [$doc cget -type]
    set state [$doc.contents cget -contents]
    set reload [lindex $state 0]
    $doc.contents tag remove cursrc 1.0 end
    if {!$force && (!$reload || $Debugger(running))} {
	# no auto reload
	return
    }
    set address [lindex $state 1]
    if {$reload} {
	set auto " \[Auto Reload]"
    } else {
	set auto ""
    }
    $doc configure -title "$type at $address $auto"
    set charheight [font metrics [$doc.contents cget -font] -linespace]
    set count [expr int([winfo height $doc.contents.text] / $charheight)]

    if {$count == 0} {
	return
    }

    set t $doc.contents
    set insertproc [$t cget -insertproc]
    $t configure -insertproc {}
    set deleteproc [$t cget -deleteproc]
    $t configure -deleteproc {}
    switch $type {
        Disassembly {
	    # a reload, update from the start address
            # break address into component parts
            set offset 0
	    set base $address
            regexp {(.*)([+-][0-9]+) line[s]?} $address all base offset
	    if {$offset} {
	       # disassembly wants to start some lines up or down from the base
	    }
	    if {[catch {DebuggerDbgExpression $base} startpc]} {
	        # can't get PC for some reason
                $t delete 1.0 end
		$t insert insert $startpc
                $t configure -insertproc $insertproc
                $t configure -deleteproc $deleteproc
		return
	    }
	    set pc [format %X $startpc]
            set s [$t search -regexp "^0*${pc}:" 1.0]
	    set src "Can't disassemble."
	    if {!$force && $s == ""} {
		# see if we can scroll
		set end [$t get end-1l end-1c]
		if {$end != {}} {
		    set val {}
		    scan $end %x val
		    set at $startpc
		    if {$val != {} && $val + \
			    ([string length [lindex $end 1]] / 2) == $at} {
		        $t delete 1.0 "1.0 lineend + 1c"
			catch {DebuggerDbg disassemble -count $at 1 $offset} src
                        $t insert end \n$src disassembly
		        $t delete "end -1c"
		        set s "end-1l"
		    }
		}
	    }
            if {!$force && $s != ""} {
	        # found a disassembly line just tag it
	        if {$startpc == $Debugger(PC)} {
	            $t tag add cursrc $s "$s lineend + 1 c"
		}
                $t configure -insertproc $insertproc
                $t configure -deleteproc $deleteproc
		return
	    }
	    # get a screen full
            catch {DebuggerDbg disassemble -count $base $count $offset} src
            $t delete 1.0 end
            $t insert insert $src disassembly
            $t delete "end -1c"
            $t mark set insert 1.0
	    if {$startpc == $Debugger(PC)} {
                set s [$t search -regexp "^0*${pc}:" 1.0]
                if {$s != ""} {
	            $t tag add cursrc $s "$s lineend + 1c"
	        }
	    }
	}
        Memory {
	    set src "Can't get memory."
            catch {DebuggerDbg dump -count $address $count} src
            $t delete 1.0 end
            $t insert insert $src disassembly
            $t delete "end -1c"
            $t mark set insert 1.0
	}
        Trace {
            set offset 0
	    set base 0
            regexp {(.*)([+-][0-9]+) instruction[s]?} $address all foo offset
	    set end [expr $offset + $count - 1]
            catch {DebuggerDbg tracebuffer get $end $base} src
            $t delete 1.0 end
            $t insert insert $src disassembly
            set s [$t search -regexp {^[+]} 1.0]
	    if {$s != {}} {
		$t delete $s "$s + 1 char"
		$t tag add cursrc $s "$s + 1 line"
                $t mark set insert $s
	    } else {
                $t mark set insert "end linestart"
	    }
	    $t configure -insertproc $insertproc
	    $t configure -deleteproc $deleteproc
	    set top [format %u $Debugger(tracesize)].0
	    set first [format %u $offset].0
	    set last [format %u $end].0
	    if {$Debugger(tracesize) != 0} {
	        set startview [expr $first / $top]
	        set sizeview [expr $last / $top]
	    } else {
		set startview 0
		set sizeview 1
	    }
	    $t.scrv set $startview $sizeview
	    return
	}
    }
    $t configure -insertproc $insertproc
    $t configure -deleteproc $deleteproc
    if {[catch {DebuggerDbgExpression $address} address]} {
	return
    }
    set end [$t get "end - 1l linestart" end]
    set endaddress 0
    regexp {([^:]*):.*} 0x$end all endaddress
    set top [Preference Build [Preference Build processor]maxaddress]
    if {$top == {}} {
	set top 0xFFFFFFFF
    }
    set top [format %u $top].0
    set first [format %u $address].0
    set last [format %u $endaddress].0
    set startview [expr $first / $top]
    set sizeview [expr $last / $top]
    $t.scrv set $startview $sizeview
}

# MEMORYstartup - return a command to recreate a memory window
#
proc code::MEMORYstartup {doc} {
    return "MEMORY [list [$doc.contents cget -contents]]"
}

#
# DEBUGscroll - scroll a debugger display window
#
proc code::DEBUGscroll {doc operation number {what {}}} {
    variable Debugger

    if {$number == 0} {
	return 0
    }
    set info [$doc.contents cget -contents]
    set auto [lindex $info 0]
    set address [lindex $info 1]
    # break address into component parts
    set offset 0
    set base $address
    regexp {(.*)([+-][0-9]+)} $address all base offset
    set type [$doc cget -type]
    if {$type != "Trace"} {
	if {[catch {DebuggerDbgExpression $base} start]} {
            # can't get address for some reason
            return 0
	}
    } else {
	set start 0
    }
    set units ""
    switch $type {
	Memory {
	    # scroll a memory window. units are 16 byte lines
            if {$operation == "scroll"} {
                if {$what == "pages"} {
                    # scroll pages
                    set charheight [font metrics [$doc.contents cget -font] -linespace]
                    set count [expr int([winfo height $doc.contents.text] / $charheight) - 3]
                } else {
                    # scroll lines
                    set count 1
                }
	        # calculate new offset
                set offset [expr $offset + ($count * $number * 16)]
            } else {
	        # moveto
                set top [Preference Build [Preference Build processor]maxaddress]
                if {$top == {}} {
	            set top 0xFFFFFFFF
                }
                set top [format %u $top].0
	        set newaddress [expr int($top * $number) & 0xFFFFFFF0]
	        set offset [expr $newaddress - $start]
            }
	}
	Disassembly {
	    # scroll a disassembly window. units are disassembly lines.
	    # if line information available, then backward scrolling
	    # is done in units of source lines.
            if {$operation == "scroll"} {
                if {$what == "pages"} {
                    # scroll pages
                    set charheight [font metrics [$doc.contents cget -font] -linespace]
                    set count [expr int([winfo height $doc.contents.text] / $charheight) - 3]
                } else {
                    # scroll lines
                    set count 1
                }
	        # calculate new offset
                set offset [expr $offset + ($count * $number)]
            } else {
	        # moveto
                set top [Preference Build [Preference Build processor]maxaddress]
                if {$top == {}} {
	            set top 0xFFFFFFFF
                }
                set top [format %u $top].0
	        set newaddress [expr int($top * $number) & 0xFFFFFFF0]
	        set offset [expr $newaddress - $start]
            }
	    if {$offset == 1 || $offset == -1} {
	        set units " line"
	    } else {
	        set units " lines"
	    }
	}
	Trace {
	    # scroll a trace window. units are trace instructions.
            if {$operation == "scroll"} {
                if {$what == "pages"} {
                    # scroll pages
                    set charheight [font metrics [$doc.contents cget -font] -linespace]
                    set count [expr int([winfo height $doc.contents.text] / $charheight) - 3]
                } else {
                    # scroll lines
                    set count 1
                }
	        # calculate new offset
                set offset [expr $offset + ($count * $number)]
            } else {
	        # moveto
                set top $Debugger(tracesize)
                set top [format %u $top].0
	        set newaddress [expr int($top * $number)]
	        set offset [expr $newaddress - $start]
            }
	    if {$offset < 0} {
		set offset 0
	    }
	    if {$offset == 1} {
	        set units " instruction"
	    } else {
	        set units " instructions"
	    }
	}
    }
    if {$offset == 0} {
	set offset ""
	set units ""
    } elseif {$offset > 0} {
	set offset +$offset
    }
    set address $base$offset$units
    $doc.contents configure -contents [list $auto $address]
    DEBUGupdateschedule $doc
    return 1
}

#
# MEMORYnextbyte - go to the next byte in the memory window
#
proc code::MEMORYnextbyte {doc s si} {
    upvar $s start
    upvar $si startinfo

    while {1} {
        # find the next data byte
        set startaddress [DEBUGaddress $doc $start startinfo]
        if {[lindex $startinfo 1] == 0} {
            # get to next byte
	    if {[lindex $startinfo 2]} {
		# at end of this column
		if {[lindex $startinfo 3] == "ASCII"} {
		    # in ASCII column
		    set next [$doc.contents index "$start + 1l linestart"]
		    if {[$doc.contents compare $next == "$start linestart"]} {
			# at end, scroll
			if {![DEBUGscroll $doc scroll 1 units]} {
			    return {}
			}
	                set next [$doc.contents index "end - 1l"]
		    }
		    set line [$doc.contents get $next "$next lineend"]
		    set ascii [string first "   " $line]
		    incr ascii 2
		    set start [$doc.contents index "$next + $ascii c"]
		    continue
		} else {
		    # in HEX column
		    set next [$doc.contents index "$start + 1l linestart"]
		    if {[$doc.contents compare $next == "$start linestart"]} {
			# at end, scroll
			if {![DEBUGscroll $doc scroll 1 units]} {
			    return {}
			}
	                set next [$doc.contents index "end - 1l"]
		    }
		    set start $next
		    continue
		}
	    }
        } elseif {$startinfo != {}} {
            break
        }
        set next [$doc.contents index "$start + 1c"]
        if {[$doc.contents compare $start == $next]} {
            # no change
            return {}
        }
        set start $next
    }
    $doc.contents mark set insert $start
    $doc.contents see insert
    return $startaddress
}

#
# DEBUGzero - zero out memory bytes, and update the memory window
#
proc code::DEBUGzero {doc first last startadr startnibble endadr endnibble} {
    variable Debugger

    set nomore 0
    if {$startnibble == 1} {
	# do the first nibble
	set byte [BYTE $startadr]
	set byte [expr {$byte & 0xF0}]
	BYTE $startadr $byte
	if {$startadr != $endadr} {
	    incr startadr
	} else {
	    set nomore 1
	}
    }
    if {!$nomore && $endnibble == 2} {
	# do the last nibble
	set byte [BYTE $endadr]
	set byte [expr {$byte & 0x0F}]
	BYTE $endadr $byte
	if {$startadr == $endadr} {
	    set nomore 1
	} else {
	    incr endadr -1
	}
    }
    if {!$nomore} {
	# do the even bytes
	DebuggerDbg fill $startadr $endadr 0
    }
    DEBUGupdate $doc 1 $first $last
}

#
# DEBUGdeposit - deposit bytes into memory, and update the window
#
proc code::DEBUGdeposit {doc first last address nibble value} {
    variable Debugger
    set len [string length $value]
    for {set index 0} {$index < $len} {incr index 2} {
        if {$nibble == 1} {
	    # do the first nibble
	    set byte [BYTE $address]
	    set byte [expr ($byte & 0xF0) | 0x0[string index $value 0]]
	    BYTE $address $byte
	    set nibble 0
	    incr index
	    if {$index >= $len} {
		break
	    }
	    incr address
        }
        set next [string range $value $index [expr $index + 1]]
        if {[string length $next] == 2} {
	    BYTE $address 0x$next
	    incr address
	} else {
	    # do the last nibble
	    set byte [BYTE $address]
	    set byte [expr ($byte & 0x0F) | 0x${next}0]
	    BYTE $address $byte
	    break
	}
    }
    DEBUGupdate $doc 1 $first $last
}

#
# MEMORYinsert - insert bytes into the memory window
#
proc code::MEMORYinsert {doc start args} {
    set first {}
    set value ""
    foreach "text tags" $args {
	# get bytes
	set end [string length $text]
	for {set index 0} {$index < $end} {incr index} {
            set startaddress [MEMORYnextbyte $doc start startinfo]
            if {$startaddress == {}} {
                return 0
            }
	    if {$first == {}} {
		set first $start
		set firstinfo $startinfo
		set firstaddress $startaddress
	    }
            # insert the next character at "startaddress"
	    set ch [string index $text $index]
            if {[lindex $startinfo 3] == "ASCII"} {
	        # inserting ASCII
		# convert to hex
                scan $ch %c ch
		set ch [format %02X $ch]
	    } else {
	        set ch [string toupper $ch]
		if {![string is xdigit $ch]} {
		    # only hex digits are valid
		    continue
		}
	    }
	    set last $start
	    append value $ch
            # move the cursor
	    set start [$doc.contents index "$start + 1c"]
	}
    }
    if {$value != ""} {
        DEBUGdeposit $doc $first $last $firstaddress [lindex $firstinfo 1] $value
        MEMORYnextbyte $doc start startinfo
    }
    return 0
}

#
# MEMORYdelete - delete bytes in the memory window
#
proc code::MEMORYdelete {doc start end} {
    while {1} {
	# find the preceeding data byte
        set startaddress [DEBUGaddress $doc $start startinfo]
	set move 0
	if {[lindex $startinfo 1] == 0} {
	    # at start of byte move up
	    set move 1
	} elseif {$startinfo != {}} {
	    break
	}
	set next [$doc.contents index "$start + 1c"]
	if {[$doc.contents compare $next >= $end]} {
	    # no change
	    if {$move} {
                $doc.contents mark set insert $start
                $doc.contents see $start
	    }
	    return 0
	}
	set start $next
    }
    while {1} {
	# find the last data byte
        set endaddress [DEBUGaddress $doc $end endinfo]
	if {[lindex $endinfo 1] == 2 && [lindex $endinfo 3] != "ASCII"} {
	    # and the end of a byte
	} elseif {$endinfo != {}} {
	    break
	}
	set next [$doc.contents index "$end - 1c"]
	if {[$doc.contents compare $next < $start]} {
	    # no change
	    return 0
	}
	set end $next
    }
    set endnibble [lindex $endinfo 1]
    if {[lindex $endinfo 3] != "ASCII"} {
	set endadr [lindex $endinfo 0]
	incr endnibble
    } else {
	set endadr [format 0x%04X [expr [lindex $endinfo 0] - 1]]
	set endnibble 1
    }
    DEBUGzero $doc $start $end [lindex $startinfo 0] [lindex $startinfo 1] \
	$endadr $endnibble
    $doc.contents mark set insert $start
    $doc.contents see $start
    return 0
}

#
# MEMORY - create a memory window
#
proc code::MEMORY {{state {1 0x0000}} args} {
    variable Debugger

    set title "Memory"
    set doc [eval TEXT $args]
    set t $doc.contents
    $doc configure -type Memory -startupproc "code::MEMORYstartup"
    set image [CODEicon efile.icon]
    $doc configure -image $image -icontext $title -title $title
    SearchUpdate $doc $doc.contents $title $title 0
    $t configure -contents $state -positionproc "code::EDITposition $doc" \
	-scrollbar on \
	-insertproc "code::MEMORYinsert $doc" \
	-deleteproc "code::MEMORYdelete $doc"
    $t.scrv configure -command "code::DEBUGscroll $doc"
    $t.text configure -yscroll "code::DEBUGscroll $doc set"
    bind $doc <<State>> "code::DEBUGupdate $doc; break"
    update idletasks
    DEBUGupdate $doc 1
    bind $doc <Configure> "code::DEBUGupdateschedule $doc"
    $doc raise
    return $doc
}

#
# DEBUGupdateschedule - schedule an update of this display window
#
proc code::DEBUGupdateschedule {doc} {
    variable Debugger

    catch {after cancel $Debugger($doc,update)}
    set Debugger($doc,update) [after 200 "code::DEBUGupdate $doc 1"]
}

#
# DebuggerWhere - display program position in a nice form and gather source
# position information
#
proc code::DebuggerWhere {where} {
    variable Debugger

    set pos "No current context"
    if {[lindex $where 0] != {}} {
	set pos "[file tail [lindex $where 0]]: [lindex $where 1]"
    } else {
	set pos "PC: [lindex $where 8]"
    }
    set Debugger(PC) [lindex $where 8]
    set Debugger(where) $pos
    set Debugger(context) $where
}

#
# DebuggerDbgExpression - do a expression command, remove type info
#
proc code::DebuggerDbgExpression {expr} {
    set result [DebuggerDbg expression $expr]
    if {$result != "' '" && [llength $result] > 1} {
	 set result [lrange $result 1 end]
    }
    return $result
}

# the list of local debugger commands
array set code::DebuggerLocal {
 path 1
 files 1
 processor 1
}

#
# DebuggerDbg - do a dbg command, either local or remote
#
proc code::DebuggerDbg {command args} {
    variable Debugger 
    variable DebuggerLocal

    if {[info exists DebuggerLocal($command)] || ![Preference DebuggerInternal serverclient]} {
        set result [eval dbg::$command $args]
        if {[Preference Command debugcommands]} {
	    CommandAdd "dbg::$command $args" $result
	} 
	return $result
    } 

    set fd $Debugger(client)
    puts $fd $args
    flush $fd
    fconfigure $fd -blocking 1
    while {[gets $Debugger(client) code] == -1 || $code == "waiting"} {
	update
    }
    fconfigure $fd -blocking 0
    set result [read $fd]
    if {$code == "ok"} {
	return $result
    } 
    error $result
}

#
# DebuggerRemote - a client has opened a port
#
proc code::DebuggerRemote {fd ip port} {
    fileevent $fd readable "code::DebuggerRemoteCommand $fd"
}

#
# DebuggerRemoteCommand - get a remote command to execute
#
proc code::DebuggerRemoteCommand {fd} {
    variable Debugger

    set line [gets $fd]
    set waiting 0
    switch [lindex $line 0] {
	go -
	step {
	    DebuggerState 1 "Program running"
            set Debugger($fd,after) [after 100 "code::DebuggerWaiting $fd"]
            set Debugger($fd,waiting) 1
	    set waiting 1
	}
    }
    if {[catch {eval dbg::$line} result]} {
	if {$waiting} {
	    set Debugger($fd,waiting) 0
            after cancel $Debugger($fd,after)
	}
	puts $fd error
	set msg $result
	set result {}
    } else {
	if {$waiting} {
	    set Debugger($fd,waiting) 0
            after cancel $Debugger($fd,after)
	}
	puts $fd ok
	set msg {}
    }
    switch [lindex $line 0] {
	go -
	step {
	    DebuggerState 0 $result
	}
    }
    if {[Preference Command debugcommands]} {
	CommandAdd $line $result $msg
    }
    puts $fd $result
    if {[catch {flush $fd}]} {
	close $fd
    }
}

#
# DebuggerWaiting - ping the client periodically
#
proc code::DebuggerWaiting {fd} {
    variable Debugger

    if {!$Debugger($fd,waiting)} {
	return
    }
    puts $fd waiting
    if {[catch {flush $fd}]} {
	close $fd
    }
    set Debugger($fd,after) [after 100 "code::DebuggerWaiting $fd"]
}

#
# DebuggerTerminate - the program termination breakpoint has been reached
#
proc code::DebuggerTerminate {w} {
    variable Debugger

    # stop the processor
    DebuggerStop

    set Debugger(stopmsg) "Program Terminated"
    if {[winfo exists $Debugger(stdio)]} {
        set t $Debugger(stdio).contents
        $t tag remove current 1.0 end
        $t insert insert "\nProgram Terminated\n\n" current
	$t see insert
    }
}

#
# DebuggerIOwrite - handle replacing the I/O function __io_write
#
proc code::DebuggerIOwrite {} {
    variable Debugger

    # the __io_write function has been entered
    # get the character to print (from parameter "ch")
    set ch [DebuggerDbgExpression ch]
    # ch is a character, remove quotes
    if {[string index $ch 0] == "'"} {
	# a character value
        set ch [string trim $ch "'"]
	# change \n, etc. to char
	set ch [subst -nocommands -novariables $ch]
        # this seems to be a tcl bug: space is left as ' '
        if {$ch == "' '"} {
            set ch " "
        }
    } else {
        set ch [format %c $ch]
    }
    # output to the Stdio window
    DebuggerStdioOut $ch
    DebuggerDbg ret 1
    if {$Debugger(stopping)} {
        # the processor has been stopped by the user interface,
	# make sure it stays stopped
        DebuggerStop
    }
}

#
# DebuggerIOread - handle replacing the I/O function __io_read
#
proc code::DebuggerIOread {} {
    variable Debugger

    # the __io_read function has been entered
    # get a character entered at the Stdio window
    set ch {}
    while {$ch == {}} {
	# wait until a character is available
        set ch [DebuggerStdioIn]
	# let the user interface function
	update
        if {$Debugger(stopping)} {
	    # the processor has been stopped by the user interface
            return
        }
    }
    # use character read as return value from function
    scan $ch %c ch
    DebuggerDbg ret $ch
}

#
# DebuggerSetupBreakpoints - set up standard breakpoints, if enabled
#
proc code::DebuggerSetupBreakpoints {w} {
    variable Debugger
    variable DebuggerBreakpoints
    variable DebuggerAtBreakpoints

    # forget old breakpoints
    DebuggerAtBreakpoints remove 0
    DebuggerComplexBreakpoints remove 0

    # set user breakpoints
    array set DebuggerBreakpoints [Preference DebuggerInternal breakpoints]
    foreach id [array names DebuggerBreakpoints] {
	set info $DebuggerBreakpoints($id)
	set type [lindex $info 0]
	set address [lindex $info 1]
	set condition [lindex $info 2]
	set command [lindex $info 3]
	set dbgid [lindex $info 4]
	set enabled [lindex $info 5]
	set stop [lindex $info 6]
        if {$stop} {
	    # remove added stopping code
	    regexp {(.*)\n# CODE.*# CODE\n} $command all command
        }
        # remove error catching code
        regexp {if \{\[catch \{\n(.*)\n\} code::breakpointerror\]\}.*} \
	    $command all command
        
	if {[catch {DebuggerSetBreakpoint $type $address 0 $enabled {} \
	    $condition $command $id $stop 1} msg]} {
	    tk_messageBox \
	        -parent . \
	        -icon warning \
	        -message "Can't set complex breakpoint at $address: $msg" \
	        -type ok
	}
    }

    foreach "break state" [Preference DebuggerInternal atbreakpoints] {
	if {[catch {DebuggerSetBreakpoint at $break 1 $state} msg]} {
	    tk_messageBox \
	        -parent . \
	        -icon warning \
	        -message "Can't set breakpoint at $break: $msg" \
	        -type ok
	}
    }
    if {[Preference DebuggerInternal atmain]} {
        # re-set breakpoint at main
	# but let a user's definition override
	set where [DebuggerDbg where main]
	set file [lindex $where 0]
	set line [lindex $where 1]
	if {![file exists $file]} {
	    catch {DebuggerDbg path -find $file} file
	}
        if {[file dirname $file] == "."} {
            set file [file join [pwd] $file]
        } 
	set address "'$file'@$line"
	if {![info exists DebuggerAtBreakpoints($address)]} {
	    if {[catch {DebuggerSetBreakpoint at main} id]} {
	    tk_messageBox \
	        -parent . \
	        -icon warning \
	        -message "Can't set breakpoint at main(): $id" \
	        -type ok
	    }
	}
    }
    # handle the termination breakpoint

    if {$Debugger(breakexit) != {}} {
        # delete old termination breakpoint(s)
        eval DebuggerDbg breakpoint delete $Debugger(breakexit)
        set Debugger(breakexit) {}
    }

    if {[Preference DebuggerInternal breakexit]} {
        if {[catch {set Debugger(breakexit) [DebuggerDbg breakpoint at __exit \
	    -command "code::DebuggerTerminate $w"]}]} {
	set Debugger(breakexit) {}
	}
    }

    # handle the standard I/O breakpoints
    if {$Debugger(breakio) != {}} {
	# delete old I/O breakpoint(s)
	eval DebuggerDbg breakpoint delete $Debugger(breakio)
	set Debugger(breakio) {}
	# clear the input buffer
	set Debugger(stdioinput) {}
	set Debugger(stdioinputhandler) {}
    }
    if {[Preference DebuggerInternal breakio]} {
	# set up the output breakpoint handler
	if {![catch {DebuggerDbg where &__io_write} where] && \
	    [lindex $where 5] != {}} {
	    # __io_write is defined
            lappend Debugger(breakio) \
	        [DebuggerDbg breakpoint at [lindex $where 5] \
	            -command code::DebuggerIOwrite]
	}
	if {![catch {DebuggerDbg where &__io_read} where] && \
	    [lindex $where 5] != {}} {
	    # set up the input breakpoint handler
            lappend Debugger(breakio) \
	        [DebuggerDbg breakpoint at [lindex $where 5] \
	            -command code::DebuggerIOread]
	}
    }
}

#
# STDIOstartup - return the command to open the stdio window
#
proc code::STDIOstartup {doc} {
    return "STDIO"
}

#
# STDIO - open the command prompt window
#
proc code::STDIO {args} {
    variable Debugger

    set type "Stdio"
    set name "Program Stdio"
    set w [CODEfind $type $name]
    if {$w != {}} {
	# have a command prompt, raise and return
	if {   [$w cget -state] == "minimized" 
	    || [$w cget -state] == "withdrawn"} {
	    $w configure -state normal
	}
	$w raise
	return $w
    }
    # a new stdio window
    set w [eval TEXT $args]
    $w menu entryconfigure Close -command "$w configure -state withdrawn"
    set t $w.contents
    StdioNew $w $t
    $w configure -type $type -startupproc "code::STDIOstartup"
    set image [CODEicon file.icon]
    $w configure -image $image -icontext $type -title $name
    $t configure -contents $name
    bind $t <KeyPress> "code::DebuggerStdioHandler %A; break"
    set Debugger(stdio) $w
    return $w
}

proc code::StdioNew {w t} {
    variable Debugger

    set type "Stdio"
    set name "Program Stdio"
    $w configure -type $type
    $t configure -contents $name

    $t delete 1.0 end
    $t configure -highlightthickness 0
    $t tag configure limit
          
    PreferenceWhenChanged Debugger $w "StdioPreferences $t"
    StdioPreferences $t
}

#
# StdioPreferences - set up Stdio window preferences
#
proc code::StdioPreferences {t} {
    $t configure -foreground [Preference Debugger colornormal]
    $t configure -font [Preference Debugger font]
    $t config -bg [Preference General colorbackground]
    $t tag configure current -foreground [Preference Debugger colorselect]
}

#
# DebuggerStdioHandler - handle a Stdio window keypress
#
proc code::DebuggerStdioHandler {ch} {
    variable Debugger

    if {$Debugger(stdioinputhandler) != {}} {
	# call a character received handler
	if {$ch == {}} {
	    return
	}
	scan $ch %c ch
	eval $Debugger(stdioinputhandler) $ch
	return
    }
    # save the character for later
    append Debugger(stdioinput) $ch
}

#
# DebuggerStdioIn - return the next character from the Stdio window
#
proc code::DebuggerStdioIn {} {
    variable Debugger

    set ch [string index $Debugger(stdioinput) 0]
    set Debugger(stdioinput) [string range $Debugger(stdioinput) 1 end]
    return $ch
}

#
# DebuggerStdioOut - send a character to the stdio window
#
proc code::DebuggerStdioOut {ch} {
    variable Debugger

    STDIO
    DebuggerInputChars $Debugger(stdio) $ch
}

#
# DebuggerInputChars - input characters and do any translations
#
proc code::DebuggerInputChars {doc chars} {
    variable Debugger

    if {![info exists Debugger($doc,inbusy)]} {
	set Debugger($doc,inbusy) 0
	set Debugger($doc,inbuffer) ""
    } elseif {$Debugger($doc,inbusy)} {
	# already inputting, just save
	append Debugger($doc,inbuffer) $chars
	return
    }

    set Debugger($doc,inbusy) 1
    set text $doc.contents
    append Debugger($doc,inbuffer) $chars
    while {[string length $Debugger($doc,inbuffer)] > 0} {
	set ch [string index $Debugger($doc,inbuffer) 0]
	set Debugger($doc,inbuffer) \
	    [string range $Debugger($doc,inbuffer) 1 end]
	switch -- $ch {
	    "\r" {
		# go to the beginning of the line
		$text mark set insert "insert linestart"
	    }
	    "\x0a" -
	    "\n" {
		# go to next line
		$text mark set insert "insert lineend"
		$text insert insert "\n"
	    }
	    "\a" {
		bell
	    }
	    "\b" -
	    "\x7f" {
		$text delete "insert - 1c"
	    }
	    default {
		# just insert the character
                $text insert insert $ch
	    }
	}
        $text see insert
    }
    set Debugger($doc,inbusy) 0
}

#
# TERMINALstartup - return the command to open a terminal  window
#
proc code::TERMINALstartup {doc} {
    return "TERMINAL [list [$doc.contents cget -contents]]"
}

#
# TERMINAL - open a terminal window
#
proc code::TERMINAL {{config {}} args} {
    variable Debugger
    variable TID

    if {![info exists TID]} {
	set TID 1
    }
    set type "Terminal $TID"
    incr TID
    set name ""
    # a new terminal window
    set w [eval TEXT -title \"$type\" $args]
    $w menu entryconfigure Close -command "code::TERMINALclose $w"
    set t $w.contents
    $w configure -type $type -startupproc "code::TERMINALstartup"
    $t configure -contents $config
    set image [CODEicon file.icon]
    $w configure -image $image -icontext $type -title $type
    DebuggerSetupTerminal $w 1
    set name "$type: [lindex [$w.contents cget -contents] 0]"
    $w configure -title $name
    return $w
}

#
# TERMINALclose - close and destroy a terminal window
#
proc code::TERMINALclose {w} {
    variable Debugger

    set name [lindex [$w.contents cget -contents] 0]
    serialPortClose [$w cget -type] $name
    foreach name [array names Debugger *,terminal*] {
	unset Debugger($name)
    }
    destroy $w
}

#
# serialPort - open and configure a serial port
#
proc code::serialPort {owner title f n b p d s {onopen {}} {onclosed {}}} {
    global tcl_platform
    upvar $f fd
    upvar $n name
    upvar $b baud
    upvar $p parity
    upvar $d data
    upvar $s stop
    variable serialStatus
    variable serialState
    variable serialParity
    variable serialPort

    if {![info exists fd]} {
	set serialState closed
    } else {
	set serialState opened
    }
    set origstate $serialState
    set origname $name
    if {$name == {}} {
        if {$tcl_platform(platform) == "windows"} {
            set list {com1: com2: com3: com4:}
	} else {
	    set list {/dev/tty00 /dev/tty01 /dev/tty02 /dev/tty03}
	}
	foreach pname $list {
	    if {![info exists serialPort($pname)]} {
		set name $pname
		break
	    }
	}
    }

    if {$baud == {}} {
	set baud 9600
    }
    set bauds {115200 57600 38400 19200 9600 4800 2400 1800 1200 300 150 110 75 50}
    if {$parity == {}} {
	set parity n
    }
    set parities {n o e m s}
    set paritynames {none odd even mark space}
    if {$data == {}} {
	set data 8
    }
    set datas {8 7 6 5}
    if {$stop == {}} {
	set stop 1
    }
    set stops {1 2}

    set box .serial
    toplevel $box 
    wm transient $box .
    wm title $box "$owner $title"
    wm protocol $box WM_DELETE_WINDOW "set code::serialStatus ok"
    set f [frame $box.b]
    set okBtn [button $f.ok -text Ok -width 6 \
        -command "set code::serialStatus ok"]
    help . $okBtn "Open the port and set or modify the configuration"
    set cancelBtn [button $f.cancel -text Cancel -width 6 \
        -command "set code::serialStatus cancel"]
    help . $cancelBtn "Cancel the serial port configuration"
    grid $okBtn -in $f -row 0 -column 0 
    grid $cancelBtn -in $f -row 0 -column 1 
    grid $f -in $box -row 3 -column 0 -sticky ew -padx 5 -pady 5
    set f [frame $box.s]
    grid $f -in $box -row 2 -column 0 -sticky ew
    set rb [radiobutton $f.on -variable code::serialState -value "opened" -text "Opened" \
	-command "set code::serialStatus open"]
    grid $rb -in $f -row 1 -column 0 -sticky w
    set rb [radiobutton $f.off -variable code::serialState -value "closed" -text "Closed"  \
	-command "set code::serialStatus close"]
    grid $rb -in $f -row 2 -column 0 -sticky w

    set f [frame $box.p]
    grid $f -in $box -row 1 -column 0 -sticky ew

    set l [label $f.nl -text "Serial Port:"]
    grid $l -in $f -row 1 -column 1 -sticky e
    set e [entry $f.n -textvariable $n]
    grid $e -in $f -row 1 -column 2 -sticky w

    set l [label $f.bl -text "Baud Rate:"]
    grid $l -in $f -row 2 -column 1 -sticky e
    set c [combobox $f.b \
    -maxheight 0 \
    -textvariable $b \
    -width 8 \
    -editable false]
    grid $c -in $f -row 2 -column 2 -sticky w
    foreach number $bauds {
        $c list insert end $number
    }
    set l [label $f.pl -text "Parity:"]
    grid $l -in $f -row 3 -column 1 -sticky e
    set i [lsearch $parities $parity]
    set serialParity [lindex $paritynames $i]
    set pcombo [combobox $f.p \
    -maxheight 0 \
    -textvariable code::serialParity \
    -width 8 \
    -editable false]
    grid $pcombo -in $f -row 3 -column 2 -sticky w
    foreach number $paritynames {
        $pcombo list insert end $number
    }

    set l [label $f.dl -text "Data Bits:"]
    grid $l -in $f -row 4 -column 1 -sticky e
    set c [combobox $f.d \
    -maxheight 0 \
    -textvariable $d \
    -width 8 \
    -editable false]
    grid $c -in $f -row 4 -column 2 -sticky w
    foreach number $datas {
        $c list insert end $number
    }

    set l [label $f.sl -text "Stop Bits:"]
    grid $l -in $f -row 5 -column 1 -sticky e
    set c [combobox $f.s \
    -maxheight 0 \
    -textvariable $s \
    -width 8 \
    -editable false]
    grid $c -in $f -row 5 -column 2 -sticky w
    foreach number $stops {
        $c list insert end $number
    }

    placewindow $box widget .

    set oldFocus [focus]
    set oldGrab [grab current $box]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $box
    focus $cancelBtn

    set done 0
    while {!$done} {
        tkwait variable code::serialStatus
	if {$serialStatus == "cancel"} {
	    break
	}
        set par [$pcombo get]
        set i [lsearch $paritynames $par]
        set parity [lindex $parities $i]
	if {$serialStatus == "close"} {
	    # close old port, if open
	    serialPortClose $owner $name
	    set serialState closed
	    catch {unset fd}
	    continue
	}
	if {   $origstate == "opened"
	    && $serialState == "closed" && $serialStatus == "ok"} {
	    # hit ok after closing
	    # close old port, if open
	    serialPortClose $owner $name
	    catch {unset fd}
	} elseif {   $origstate == "closed"
	         && $serialState == "closed" && $serialStatus == "ok"} {
	    # hit ok with no change
	} elseif {$serialStatus == "open" || ![info exists fd] || $origname != $name} {
	    # close old port, if open
	    serialPortClose $owner $name
	    set serialState closed
	    if {$name == {}} {
		# no name given, just close port
		break
	    }
	    # open port and configure for communications
            if {[catch {open $name RDWR} fd]} {
		if {[info exists serialPort($name)]} {
                    tk_messageBox \
		        -parent . \
               	        -icon error \
                        -message "$name is already in use by $serialPort($name,owner)" \
                        -type ok
		} else {
                    tk_messageBox \
		        -parent . \
               	        -icon error \
                        -message "$fd" \
                        -type ok
		}
	        catch {unset fd}
	    } else {
		set serialState opened
                fconfigure $fd -buffering none -blocking 0 -translation binary
		set origname $name
		set origstate "opened"
		set serialPort($name) $fd
		set serialPort($name,onopen) $onopen
		set serialPort($name,onclosed) $onclosed
		set serialPort($name,owner) $owner
		if {$onopen != {}} {
		    if {[catch {uplevel #0 $onopen} msg]} {
                        tk_messageBox \
	                    -parent . \
                            -icon error \
                            -message "Error opening $name: $msg" \
                            -type ok
		    }
		}
	    }
        } elseif {[info exists fd]} {
            fconfigure $fd -mode $baud,$parity,$data,$stop 
	    set serialPort($name) $fd
	    set serialPort($name,onopen) $onopen
	    set serialPort($name,onclosed) $onclosed
	    set serialPort($name,owner) $owner
	    if {$onopen != {}} {
	        if {[catch {uplevel #0 $onopen} msg]} {
                    tk_messageBox \
	                -parent . \
                        -icon error \
                        -message "Error opening $name: $msg" \
                        -type ok
		}
	    }
        }
	if {$serialStatus == "ok"} {
	    set done 1
	}
    }
    catch {focus $oldFocus}
    grab release $box
    destroy $box
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
}

#
# serialPortClose - close an open serial port
#
proc code::serialPortClose {owner name} {
    variable serialPort

    if {![info exists serialPort($name)]} {
	# not open
	return
    }
    if {$serialPort($name,owner) != $owner} {
	# not the owner of the port
	return
    }
    set fd $serialPort($name)
    close $fd
    set onclosed $serialPort($name,onclosed)
    if {$onclosed != {}} {
	if {[catch {uplevel #0 $onclosed} msg]} {
            tk_messageBox \
	        -parent . \
                -icon error \
                -message "Error closing $name: $msg" \
                -type ok
	}
    }
    unset serialPort($name)
    unset serialPort($name,onopen)
    unset serialPort($name,onclosed)
    unset serialPort($name,owner)
}

#
# DebuggerSetupTerminal - set up the terminal widget
#
proc code::DebuggerSetupTerminal {w {force 0}} {
    variable Debugger

    set text $w.contents
    bind $text <KeyPress> "code::DebuggerTerminalOut $w %A; break"
    $text configure -foreground [Preference Debugger colornormal]
    $text configure -font [Preference Debugger font]
    if {$force || ![info exists Debugger($w,terminalfd)]} {
        # ask for the serial port
        TERMINALsetup $w
    }
}

#
# TERMINALsetup - set up a debugger terminal port
#
proc code::TERMINALsetup {{w {}}} {
    variable Debugger
    variable CODEdocument

    if {$w == {}} {
	set w $CODEdocument
    }

    set list [$w.contents cget -contents]
    set Debugger($w,terminal) [lindex $list 0]
    set Debugger($w,terminalbaud) [lindex $list 1]
    set Debugger($w,terminalparity) [lindex $list 2]
    set Debugger($w,terminaldata) [lindex $list 3]
    set Debugger($w,terminalstop) [lindex $list 4]
    serialPort [$w cget -type] "Serial Port Setup" code::Debugger($w,terminalfd) \
      code::Debugger($w,terminal) \
      code::Debugger($w,terminalbaud) code::Debugger($w,terminalparity) \
      code::Debugger($w,terminaldata) code::Debugger($w,terminalstop) \
      "fileevent \$code::Debugger($w,terminalfd) readable \"code::DebuggerTerminalIn $w\""
    $w.contents configure -contents \
	[list $Debugger($w,terminal) $Debugger($w,terminalbaud) \
	      $Debugger($w,terminalparity) $Debugger($w,terminaldata) \
	      $Debugger($w,terminalstop)]
}

#
# DebuggerTerminalIn - put input characters in the terminal window
#
proc code::DebuggerTerminalIn {w} {
    variable Debugger

    set chars [read $Debugger($w,terminalfd)]
    DebuggerInputChars $w $chars
}

#
# DebuggerTerminalOut - send a character to the Terminal window
#
proc code::DebuggerTerminalOut {w ch} {
    variable Debugger

    if {![info exists Debugger($w,terminalfd)]} {
        # no port open, ask for one
        DebuggerSetupTerminal $w
    }
    if {[info exists Debugger($w,terminalfd)]} {
	puts -nonewline $Debugger($w,terminalfd) $ch
    } 
}

#
# DebuggerGetRegisters - get the current register values
#
proc code::DebuggerGetRegisters {} {
    variable Debugger
    variable DebuggerRegisters

    if {[catch {DebuggerDbg registers} regs]} {
	DebuggerCommError $regs
	return
    }
    foreach list $regs {
	set name [lindex $list 0]
	if {$name == {}} {
	    continue
	}
	set DebuggerRegisters($name) 0x[lindex $list 1]
    }
}

#
# DebuggerRegisterCheck - make sure contents of register variables are valid
#
proc code::DebuggerRegisterCheck {reg} {
    variable Debugger
    variable DebuggerRegisters

    set value $DebuggerRegisters($reg)
    if {[catch {DebuggerDbgExpression "%$reg=$value"} msg]} {
        tk_messageBox \
	    -parent . \
            -icon error \
            -message "The specified value for register $reg is invalid: $msg" \
            -type ok
    }
    DebuggerState
}

#
# DebuggerCommError - some kind of communication error has occured
#
proc code::DebuggerCommError {msg} {
    tk_messageBox \
	-parent . \
        -icon error \
        -message "Target error: $msg" \
        -type ok
}

#
# Module handling
#

#
# DebuggerModuleSelect - select debugger modules
#
proc code::DebuggerModuleSelect {} {
    variable INTROL
    variable Debugger
    variable DebuggerModules

    set top .modules
    set geometry [Preference Geometry modules]
    if {[winfo exists $top]} {
        if {$geometry != {}} {
	    # the window has had its geometry change saved
	    setGeometry $top $geometry
	}
        wm deiconify $top
	raise $top
        wm transient $top .
        # this code seems to get around a bug in Tk (wm geometry for
        # a hidden window)
        global tcl_platform
        if {$tcl_platform(platform) == "windows" && $geometry != {}} {
            setGeometry $top $geometry
        }
	return
    }
    set top [toplevel $top]
    wm transient $top .
    wm title $top "CODE Debugger modules"
    wm protocol $top WM_DELETE_WINDOW "wm withdraw $top"
    set n $top.nb

    # make a list of module tabs
    if {[Preference Build processor] != {}} {
        set list "
	    [DebuggerDbg processor]
	    Common
	    Project
        "
        # a list of tab descriptions
        set helps {
	    "Modules specific to the current processor"
	    "Modules that can be used for any processor"
	    "Modules specific to this project"
        }
        # a list of module directories
        set dirs "
	    [list [file join $INTROL Modules [DebuggerDbg processor]]]
	    [list [file join $INTROL Modules]]
	    {}
        "
    } else {
        set list "
	    Common
	    Project
        "
        # a list of tab descriptions
        set helps {
	    "Modules that can be used for any processor"
	    "Modules specific to this project"
        }
        # a list of module directories
        set dirs "
	    [list [file join $INTROL Modules]]
	    {}
        "
    }

    Notebook:create $n -pages $list -pad 20 -status code::status($top)
    grid $n -in $top -row 0 -column 0 
    label $top.l -text "" -textvariable code::status($top) \
        -relief sunken -bd 2 \
	-bg [Preference General colorstatusbackground] \
        -anchor w -font [Preference General fontstatus]
    PreferenceWhenChanged General $top "$top.l config \
	-bg \[Preference General colorstatusbackground] \
        -font \[Preference General fontstatus]"
    grid $top.l -in $top -row 2 -column 0 -sticky ew
    grid columnconfigure $top.l 0 -weight 1

    # initialize each tab
    foreach tab $list help $helps dir $dirs {
        set nb [Notebook:frame $n $tab]
        Notebook:pageconfig $n $tab -status $help
	help $top $nb "$help"
	set files [lsort [glob -nocomplain [file join $dir *.mod]]]
	set row 0
	set column 0
	foreach file $files {
            set module [file rootname [file tail $file]]
	    if {[lsearch $Debugger(modules) $module] != -1} {
		set DebuggerModules($module) 1
	    } else {
		set DebuggerModules($module) 0
	    }
	    # get the module info
	    set f [open $file]
	    set info "No information available about this module"
	    while {[gets $f line] != -1} {
	        if {[regexp {([ \t]*INFO[ \t]*)(.*)} $line foo foo line]} {
		    set info $line
		    break
		}
	    }
	    close $f
	    set c [checkbutton $nb.rb$module -text $module \
		-variable code::DebuggerModules($module) \
		-width 15 \
		-anchor w \
		-command "code::DebuggerModuleToggle $top $module" \
		]

	    help $top $c "$info"
	    bind $c <Button-3> "code::EDIT $file" 
	    grid $c -in $nb -row $row -column $column -sticky w
	    incr column
	    if {$column > 2} {
		set column 0
		incr row
	    }
	}
	incr row
	grid rowconfigure $nb $row -weight 1
	incr row
        label $nb.ll -text "Single click on a module to enable or disable"
        grid $nb.ll -in $nb -row $row -column 0 -columnspan 3 -sticky w
	incr row
        label $nb.l -text "Right click on a module to edit"
        grid $nb.l -in $nb -row $row -column 0 -columnspan 3 -sticky w
	grid columnconfigure $nb 3 -weight 1
    }
    if {$geometry != {}} {
        # the window has had its geometry change saved
        setGeometry $top $geometry
    }
    bind $top <Configure> "code::PreferenceMove $top modules"
    bind $top <F1> "code::OpenHelp Applications Debugger Modules.html"
}

#
# DebuggerModule - add a module to the debugging environment
#
proc code::DebuggerModule {args} {
    variable Debugger

    if {$args == {}} {
	# just return modules
	return $Debugger(modules)
    }
    foreach module $args {
	set file [DebuggerFindModule $module 1]
	if {$file == {}} {
	    # already loaded
	    continue
	}
    
        if {[catch {namespace eval :: source \"$file\"} msg]} {
	    tk_messageBox -icon error \
	        -parent . \
	        -message "Error reading module $file.\n$msg" \
	        -type ok
	    return
        }
	# initialize and start the module
	if {[info commands ${module}start] != {}} {
	    if {[catch {${module}start} msg]} {
	        tk_messageBox -icon error \
	            -parent . \
	            -message "Error starting module $module: $msg" \
	            -type ok
	        return
	    }
	}
	if {[info commands ${module}reset] != {}} {
	    if {[catch {${module}reset} msg]} {
	        tk_messageBox -icon error \
	            -parent . \
	            -message "Error reseting module $module: $msg" \
	            -type ok
	        return
	    }
	}
        lappend Debugger(modules) $module
    }
}

#
# DebuggerFindModule - find a debugger script module
#
proc code::DebuggerFindModule {module {error 0} {extension mod}} {
    variable INTROL
    variable Debugger

    set file [set module].$extension
    if {[lsearch $Debugger(modules) $module] != -1} {
        # module already loaded
        return {}
    }
    if {![file exists $file]} {
        # check the common module directory
         set file [file join $INTROL Modules $module.$extension]
        if {![file exists $file]} {
            # try the processor specific module directory
            set file [file join $INTROL Modules [DebuggerDbg processor] $module.$extension]
            if {![file exists $file]} {
	    # can't find this module
		if {$error} {
	            tk_messageBox -icon error \
	                -parent . \
	                -message "Cannot find module $module." \
	                -type ok
		}
    	        return {}
   	    }
	}
    }
    return $file
}

#
# DebuggerModuleDeleteAll - delete all debugger modules
#
proc code::DebuggerModuleDeleteAll {} {
    variable Debugger

    foreach module $Debugger(modules) {
	DebuggerModuleDelete $module
    }
}

#
# DebuggerModuleDelete - delete a debugger module
#
proc code::DebuggerModuleDelete {module} {
    variable Debugger
    variable MODULEwindows

    # issue a delete to all modules that need it
    if {[info commands [set module]delete] != {}} {
        [set module]delete
    }
    # destroy any open windows
    DESTROY $module
    set Debugger(modules) [lremove $Debugger(modules) $module]
}

#
# DebuggerModuleToggle - enable or disable a module
#
proc code::DebuggerModuleToggle {w module} {
    variable Debugger
    variable DebuggerModules

    set modules [Preference DebuggerInternal modules]
    if {$DebuggerModules($module)} {
	DebuggerModule $module
	if {[lsearch $modules $module] == -1} {
	   lappend modules $module
	}
    } else {
        DebuggerModuleDelete $module
	set modules [lremove $modules $module]
    }
    Preference DebuggerInternal modules $modules 1
}

#
# define common commands used in modules
#

#
# HELP - define balloon help for a widget
#
proc code::HELP {wid string} {
    help . $wid $string
}

#
# BYTE - read/write a byte from memory
#
proc code::BYTE {address {value {}}} {
    if {$value == {}} {
        return [DebuggerDbgExpression \$getub($address)]
    }
    DebuggerDbgExpression \$putub($address,$value)
}

#
# WORD - read/write a word from memory
#
proc code::WORD {address {value {}}} {
    if {$value == {}} {
        return [DebuggerDbgExpression \$getuw($address)]
    }
    DebuggerDbgExpression \$putuw($address,$value)
}

#
# LONG - read/write a long from memory
#
proc code::LONG {address {value {}}} {
    if {$value == {}} {
        return [DebuggerDbgExpression \$getul($address)]
    }
    DebuggerDbgExpression \$putul($address,$value)
}

#
# BR - set a processor breakpoint
#
proc code::BR {args} {
    return [eval DebuggerDbg breakpoint $args]
}

#
# REG - get or set a processor register
#
proc code::REG {name {value {}}} {
    if {$value == {}} {
	return [DebuggerDbgExpression %$name]
    }
    DebuggerDbgExpression "%$name=$value"
}

#
# VAR - get or set a debugger variable
#
proc code::VAR {name {value {}}} {
    if {$value == {}} {
	return [DebuggerDbgExpression \$$name]
    }
    DebuggerDbgExpression "\$$name=$value"
}

#
# VADR - get the address of a variable in an object file
#
proc code::VADR {name} {
    return [DebuggerDbgExpression "&$name"]
}

#
# PROCESSOR - return the current processor
#
proc code::PROCESSOR {} {
    return [DebuggerDbg processor]
}

#
# VARIANT - return the current processor variant
#
proc code::VARIANT {} {
    variable Configure
    return $Configure(Build,variant)
}

#
# PREFERENCE - return the current processor variant
#
proc code::PREFERENCE {module name {value {}} {force 0}} {
    if {$value != {} || $force} {
        return [PreferenceSetIfChanged $module $name $value]
    }
    return [Preference $module $name]
}

#
# PORT - set the target parallel i/o port
#
proc code::PORT {address} {

    DebuggerDbg port -address $address
}

#
# SERIAL - set the target serial i/o port
#
proc code::SERIAL {} {
    variable Debugger

    # get a serial port
    serialPort [DebuggerDbg target -id] "Serial Port Setup" \
        code::Debugger(serialfd) \
        code::Debugger(serial) \
        code::Debugger(serialbaud) code::Debugger(serialparity) \
        code::Debugger(serialdata) code::Debugger(serialstop) \
        {code::DebuggerDbg port -speed $code::Debugger(serialbaud) $code::Debugger(serialfd)} \
        {code::DebuggerDbg port {}}
}

#
# WHERE - return the program position for an address
#
proc code::WHERE {where} {
    if {[catch {DebuggerDbg where &$where} where]} {
        set pos "No current context"
    } elseif {[lindex $where 0] != {}} {
	set pos "[file tail [lindex $where 0]]: [lindex $where 1]"
    } else {
	set pos "PC: [lindex $where 8]"
    }
    return $pos
}

#
# TK - load Tk into the slave interpreter
#
proc code::TK {} {
    return
}

#
# WINDOW - create a window for a debugger module
#
proc code::WINDOW {module title} {
    variable Debugger
    variable MODULEcount
    variable MODULEwindows

    if {![info exists MODULEcount]} {
	set MODULEcount 0
    }
    incr MODULEcount
    set type "Module $MODULEcount"
    set w [document .work.mod$MODULEcount -type $type -raiseproc code::DOCraise]
    bind $w <<State>> break
    $w menu entryconfigure Close -command "DESTROY $module $w.mod.f"

    # set up bindings
    set c [scrollcanvas $w.mod -scrollbar auto]
    $w pack $c -fill both -expand 1 -side right
    set f [eval frame $c.f]
    bind $f <1> "$w raise"
    $c configure -background [$f cget -background]
    $c create window 0 0 -window $f -anchor nw
    if {![info exists MODULEwindows($module)]} {
        set MODULEwindows($module) $f
    } else {
        lappend MODULEwindows($module) $f
    }
    $c configure -scrollbar auto
    set image [CODEicon efile.icon]
    $w configure -image $image -icontext $title -title $title
    return $f
}

#
# TITLE - set a module window title
#
proc code::TITLE {window title} {
    [winfo parent [winfo parent $window]] configure -icontext $title -title $title
}

#
# WINDOWS - return the list of windows for this module
#
proc code::WINDOWS {module} {
    variable MODULEwindows

    if {![info exists MODULEwindows($module)]} {
	return {}
    } 

    return $MODULEwindows($module)
}

#
# DESTROY - the module's destroy command
#
proc code::DESTROY {module args} {
    variable Debugger
    variable MODULEwindows

    if {![info exists MODULEwindows($module)]} {
	# no windows exist
	return
    }
    if {$args == {}} {
	set args $MODULEwindows($module)
    }
    foreach w $args {
        destroy [winfo parent [winfo parent $w]]
        set MODULEwindows($module) [lremove $MODULEwindows($module) $w]
    }
    if {$MODULEwindows($module) == {}} {
        unset MODULEwindows($module)
    }
}

#
# UI - load a user-interface into a module
#
proc code::UI {name w} {
    variable Debugger

    # find the user-interface file
    set file [DebuggerFindModule $name.ui 1 tcl]
    if {$file == {}} {
        # can't find it
        return
    }
    # read the user interface file
    namespace eval :: source \"$file\"
    # build the user interface
    ${name}_ui $w
}

#
# BUTTON - create a button for a module
#
proc code::BUTTON {name action {help {}}} {
    variable Debugger

    set tbar .moduletools
    if {![winfo exists $tbar]} {
	# create the module toolbar
        toolbar $tbar -side bottom
        tooltip .mtip
    }

    # remove illegal characters from widget name
    regsub -all {[\. ]} $name {} wname
    if {[catch {$tbar add button but$wname -text $name \
	-highlightbackground gray -command $action} widget]} {
	# button already exists
	$tbar itemconf but$wname -command $action
    } else {
        .mtip add $widget $help
    }
    incr Debugger(moduletools)
}

#
# DELETEBUTTON - delete button(s) for a module
#
proc code::DELETEBUTTON {args} {
    variable Debugger

    foreach name $args {
        regsub -all {[\. ]} $name {} wname
        if {![catch {.moduletools delete but$wname}]} {
            incr code::Debugger(moduletools) -1
	}
    }
    if {$code::Debugger(moduletools) == 0} {
	 destroy .moduletools
	 destroy .mtip
    }
}

#
# STDIN - set up a stdin handler
#
proc code::STDIN {handler} {
    variable Debugger

    set Debugger(stdioinputhandler) $handler
}

#
# INFO - dummy command for module info
#
proc code::INFO {args} {
}

#
# TARGET - return the current debugger target
#
proc code::TARGET {} {
    return [DebuggerDbg target -id]
}

#
# STDOUT - show stdout characters
#
proc code::STDOUT {ch} {
    DebuggerStdioOut $ch
}

#
# UPDATE - update the debugger's state
#
proc code::UPDATE {} {
    DebuggerState
}

#
# MODULE - load debugger modules
#
proc code::MODULE {args} {
    return [eval DebuggerModule $args]
}

#
# RESET - reset the target processor
#
proc code::RESET {} {
    return [DebuggerDoReset]
}

#
# EXPR - evaluate an expression in the target environment
#
proc code::EXPR {args} {
    return [DebuggerDbgExpression $args]
}
