#
#	Directory - manage a directory window
#

array set code::DirectoryInit {
    select {}
    selectindex {}
    selectrect {}
    selectinput 0
    tree {}
}

set code::DIRECTORYcount 0

#
# Check for windows system, handle drive letters if so.
#
if {$tcl_platform(platform) == "windows"} {
    set code::DirectoryDos 1
} else {
    set code::DirectoryDos 0
}

#
# DIRECTORYstartup - return the command to open the directory window
#
proc code::DIRECTORYstartup {doc} {
    variable Directory

    return "DIRECTORY [list [$doc.dir cget -contents]]"
}

#
# DIRECTORY - open a directory window
#
proc code::DIRECTORY {{file {}} args} {
    global tk_library
    variable window
    variable Directory
    variable DirectoryInit
    variable DirectoryDos
    variable FileIcons
    variable Build
    variable DIRECTORYcount

    if {$file != {} && ![file isdirectory $file]} {
        # this should only occur in startup code
        error "$file is not a directory"
    }
    set w [eval document .work.dir$DIRECTORYcount -type Folder \
	-raiseproc code::DIRECTORYraise \
	-width 550 \
	-startupproc code::DIRECTORYstartup $args]
    bind $w <<State>> break
    trace variable code::Directory($w,select) w \
	"if \{\$code::CODEdocument == \"$w\"\} \{ set code::CODEcurrentfiles \$code::Directory($w,select) \}; #"
    incr DIRECTORYcount

    # set configuration options
    foreach name [array names DirectoryInit] {
        set Directory($w,$name) $DirectoryInit($name)
    }

    # set up bindings
    set c [scrollcanvas $w.dir -scrollbar auto]
    if {$file == {}} {
        # get initial directory
        $w.dir configure -contents [DirectoryPwd]
    } else {
        $w.dir configure -contents $file
    }

    $w pack $c -fill both -expand 1 -side right
    $c bind entry <Enter> \
        "code::DirectoryEntryBinding $w enter %W %x %y %X %Y"
    $c bind entry <Leave> \
        "code::DirectoryEntryBinding $w leave %W %x %y %X %Y"
    $c bind entry <Button-1> \
	"code::DirectoryEntryBinding $w btn1 %W %x %y %X %Y"
    $c bind entry <B1-Motion> \
	"code::DirectoryEntryBinding $w btn1m %W %x %y %X %Y"
    $c bind entry <Shift-Button-1> \
	"code::DirectoryEntryBinding $w shift1 %W %x %y %X %Y"
    $c bind entry <Shift-B1-Motion> \
	"code::DirectoryEntryBinding $w shift1 %W %x %y %X %Y"
    $c bind entry <Double-Button-1> \
	"code::DirectoryEntryBinding $w dbl1 %W %x %y %X %Y"
    $c bind entry <Button-3> \
	"code::DirectoryEntryBinding $w btn3 %W %x %y %X %Y"
    bind $c <Control-a> "code::Directory $w selectall"
    bind $c <Control-s> "code::Directory $w selectfiles"

    # we want to handle button 1 everywhere
    bind $c <Button-1> \
	"code::DirectoryEntryBinding $w btn1 %W %x %y %X %Y"
    bind $c <Shift-Button-1> { # nothing }

    # build the tree widgit
    set tf [frame $w.tf -bd 2 -relief sunken]
    $w pack $tf -fill both -expand 0 -side left
    grid rowconfigure $tf 0 -weight 1
    set t $tf.t
    set Directory($w,tree) $t
    set sb [scrollbar $tf.sb -orient vertical -command "$t yview"]
    Tree:create $t -bg [Preference General colorbackground] \
	-width 0 -height 0 -yscrollcommand "$sb set"
    grid $t -in $tf -row 0 -column 0 -sticky ns
    grid columnconfigure $tf 0 -weight 1
    grid $sb -in $tf -row 0 -column 1 -sticky ns
    set dir [file split [$w.dir cget -contents]]
    set path [lindex $dir 0]
    set dir [lrange $dir 1 end]
    if {$DirectoryDos} {
	# deal with windows drives

	set v [file volume]
	foreach vol $v {
	    regexp {([^:]*:)/} $vol all vol
	    set vol [string toupper $vol]
	    # XXX should be a drive icon
	    Tree:newitem $t /$vol -image $FileIcons(directory.icon)
	}
    }
    
    $t bind x <1> "code::DirectoryTreeSingle %W %x %y $w"
    $t bind x <Double-1> "code::DirectoryTreeDouble %W %x %y"

    # get root: path is either "/" or e.g. "C:/" 
    DirectoryFillTree $t $path
    if {$DirectoryDos} {
	Tree:open $t /$path
    }
    foreach element $dir {
	set path $path$element
	DirectoryFillTree $t $path

	if {$DirectoryDos} {
	    Tree:open $t /$path
	} else {
	    Tree:open $t $path
	}
	append path /
    }
    
    if {$DirectoryDos} {
	set pwd /[$w.dir cget -contents]
    } else {
	set pwd [$w.dir cget -contents]
    }
    Tree:setselection $t $pwd
    Tree:show $t $pwd
    Tree:open $t $pwd

    PreferenceWhenChanged Directory $w "DirectoryFill $w"
    PreferenceWhenChanged Build $w "DirectoryFill $w"
    PreferenceWhenChanged General $w "DirectoryFill $w"
    after idle code::DIRECTORYraise $w
    after idle code::DirectoryFill $w
    return $w
}

#
# DIRECTORYraise - a directory window has been raised
#
proc code::DIRECTORYraise { doc } {
    variable Directory
    variable CODEcurrentfiles

    DOCraise $doc
    # catch the following because the window may not be initialized yet
    catch {focus $doc.dir}
    catch {set CODEcurrentfiles $Directory($doc,select)}
    ToolGenerate toolbar
}

#
# DirectoryGotoProject - go to the project directory
#
proc code::DirectoryGotoProject {} {
    variable Build

    if {$Build(PROJECT) == {}} {
        tk_messageBox -icon error \
            -parent . \
	    -message "No project file is open" -type ok
	return
    }
    DirectoryOpen $Build(PROJECT)
}

#
# DirectoryGotoOutput - go to the project output directory
#
proc code::DirectoryGotoOutput {} {
    variable Build

    if {$Build(OBJECT) == {}} {
        return
    }
    DirectoryOpen [BuildSubstitute $Build(OBJECT)]
}
#
# DirectorySetOutput - set the project output directory
#
proc code::DirectorySetOutput {doc} {
    variable Build
    variable Directory

    set Build(OBJECT) [BuildFileName [$doc.dir cget -contents]]
    Preference Build OBJECT $Build(OBJECT)
}

#
# DirectoryNewFolder - create a new folder
#
proc code::DirectoryNewFolder {} {
    variable Directory
    variable DirectoryDos
    variable CODEdocument
    variable FileIcons

    set w $CODEdocument

    # need a unique name
    set pwd [$w.dir cget -contents]
    if {$DirectoryDos && [file split $pwd] == $pwd} {
        set file [file join $pwd/ "New Folder"]
    } else {
        set file [file join $pwd "New Folder"]
    }
    if {[file exists $file]} {
        set index 1
        while {[file exists "$file ($index)"]} {
   	    incr index
	}
        set file "$file ($index)"
    }
    if {[catch {file mkdir $file} msg]} {
        tk_messageBox -icon error \
	    -parent $w \
	    -message "$msg" -type ok
	return
    }

    # refresh the directory tree
    set dir [$w.dir cget -contents]
    if {$DirectoryDos} {
        # add a leading /
        set dir /$dir
    } 
    set t $Directory($w,tree)
    Tree:delitem $t $dir
    Tree:newitem $t $dir -image $FileIcons(directory.icon)
    DirectoryFillTree $t [$w.dir cget -contents]
    Tree:open $t $dir
    Tree:setselection $t $dir

    DirectoryFill $w
    update idletasks

    # display the new entry
    set c $w.dir
    set ids [$c find withtag name]
    foreach id $ids {
        set tags [$c gettags $id]
        set name [DirectoryFindTag $tags file]
        if {$name == $file} {
   	    set y [lindex [$c coords $id] 1]
	    set bbox [$c bbox all]
	    set all [expr {[lindex $bbox 3] - [lindex $bbox 1]}]
	    set y [expr {$y - [lindex $bbox 1]}]
	    $c yview moveto [expr {$y / $all}]
	}
    }
    DirectoryRename $file
}

#
# DirectoryRename - rename a file
#
proc code::DirectoryRename {file} {
    variable Directory
    variable CODEdocument

    set w $CODEdocument

    set c $w.dir
    # get all the entries
    set ids [$c find withtag name]
    foreach id $ids {
        set tags [$c gettags $id]
	set name [DirectoryFindTag $tags file]
	if {$name == $file} {
	    DirectoryClearSelect $w $c
	    # select the item
            set sc [Preference General colorselection]
	    set rect [$c create rect 0 0 0 0 -fill $sc -outline black]
	    $c icursor $id end
	    focus $c.canvas
	    $c focus $id
	    $c bind $id <Key> "code::DirectoryEdit $w $c $id %K %A"
	    $c bind $id <1> "code::DirectoryEditPos $w $c $id %x %y"
	    $c lower $rect
	    set bbox [$c bbox $id]
	    eval $c coords $rect $bbox
	    set Directory($w,select) $file
	    set Directory($w,selectindex) $id
	    set Directory($w,selectrect) $rect
	    set Directory($w,selectinput) 1
	    set Directory($w,selectorig) [$c itemcget $id -text]
	    break
        }
    }
}

#
# DirectoryDelete - delete file(s)
#
proc code::DirectoryDelete {files} {
    variable Directory
    variable DirectoryDos
    variable CODEdocument

    set w $CODEdocument

    if {[llength $files] == 1} {
	# deleting a single file
	set result [tk_messageBox \
	    -parent $w \
	    -icon question \
	    -message "Delete [file nativename $files]?" \
	    -default no \
	    -type yesno]
	if {$result == "no"} {
	    return
	}
    } else {
        # deleting multiple files
	set result [tk_messageBox \
	    -parent $w \
	    -icon question \
	    -message "Delete all the selected files?" \
	    -default no \
	    -type yesno]
	if {$result == "no"} {
	    return
	}
    }
    foreach arg $files {
	# are we deleting a directory?
        if {[file isdirectory $arg]} {
	    set directory 1
	} else {
	    set directory 0
	}

	if {[catch {file delete $arg} msg]} {
	    regexp {.*"[^"]*": (.*)} $msg all base
	    if {$base == "directory not empty"} {
	        set result [tk_messageBox \
		    -parent $w \
	            -icon question \
	            -message "Folder [file nativename $arg] is not empty. Delete?" \
	            -default yes \
	            -type yesnocancel]
	        if {$result == "cancel"} {
		    break
	        } elseif {$result == "yes"} {
	            if {![catch {file delete -force $arg} msg]} {
                        if {$directory} {
                            # refresh the directory tree
	                    set W $Directory($w,tree)
		            # get the parent
		            set arg [file dirname $arg]
		            if {$DirectoryDos} {
	                        if {[file split $arg] == $arg} {
		                    # remove the trailing /
                                    regexp {(.*:)/} $arg all arg
		                }
		                set lbl /$arg
		            } else {
		                set lbl $arg
		            }
                            Tree:delitem $W $lbl
	                    Tree:newitem $W $lbl -image $FileIcons(directory.icon)
	                    DirectoryFillTree $W $arg
		            Tree:open $W $lbl
                        }
			continue
		    }
	        } else {
		    continue
	        }
	    }
            tk_messageBox -icon error \
	        -parent $w \
	        -message "$msg" -type ok
	}
        if {$directory} {
            # refresh the directory tree
	    set W $Directory($w,tree)
	    # get the parent
	    set arg [file dirname $arg]
	    if {$DirectoryDos} {
	        if {[file split $arg] == $arg} {
	  	    # remove the trailing /
                    regexp {(.*:)/} $arg all arg
		}
		set lbl /$arg
	    } else {
	        set lbl $arg
	    }
            Tree:delitem $W $lbl
	    Tree:newitem $W $lbl -image $FileIcons(directory.icon)
	    DirectoryFillTree $W $arg
	    Tree:open $W $lbl
        }
    }
    DirectoryRefresh
}

#
# DirectoryOpen - open a directory
#
proc code::DirectoryOpen {file} {
    variable CODEdocument
    variable Directory
    variable DirectoryDos
    variable FileIcons

    set w $CODEdocument
    if {$w == {} || [$w cget -type] != "Folder"} {
	DIRECTORY $file
    } else {
	if {[file pathtype $file] == "relative"} {
	    set file [file join [$w.dir cget -contents] $file]
	}
	# do the current directory
	if {$DirectoryDos} {
	    set lbl /$file
	} else {
	    set lbl $file
	}
	set W $Directory($w,tree)
        if {[file isdirectory $file]} {
            # refresh the directory
            Tree:delitem $W $lbl
	    Tree:newitem $W $lbl -image $FileIcons(directory.icon)
	    DirectoryFillTree $W $file
        }
        Tree:setselection $W $lbl
        Tree:open $W $lbl
	Tree:show $W $lbl
	DirectoryChange $w $file
    }
}

#
# DirectorySelectAll - select all files in a directory
#
proc code::DirectorySelectAll {} {
    variable CODEdocument

    DirectorySelectFiles $CODEdocument *
}

#
# DirectorySelectPattern - select all files in a directory that match a pattern
#
proc code::DirectorySelectPattern {} {
    variable CODEdocument

    DirectorySelectFiles $CODEdocument [DirectoryGetPattern $CODEdocument]
}

#
# DirectoryRefresh - refresh all or selected directory windows
#
proc code::DirectoryRefresh {{thisdoc {}}} {
    variable Directory

    foreach doc [allDocs] {
	if {[$doc cget -type] == "Folder"} {
	    if {$thisdoc == {} || $doc == $thisdoc} {
	        DirectoryFill $doc
	    }
	}
    }
}

#
# DirectorySelectFiles - select matching files
#
proc code::DirectorySelectFiles {w pattern} {
    variable Directory

    set c $w.dir
    set ids [$c find withtag name]
    foreach id $ids {
        set tags [$c gettags $id]
        set name [DirectoryFindTag $tags file]
	if {![string match $pattern $name]} {
	    continue
	}
	if {[lsearch $Directory($w,selectindex) $id] != -1} {
   	    # already selected
	    continue
	}
	# add to a selection
        set sc [Preference General colorselection]
	set rect [$c create rect 0 0 0 0 -fill $sc -outline $sc]
	$c lower $rect
	set bbox [$c bbox $id]
	eval $c coords $rect $bbox
	lappend Directory($w,select) [DirectoryFindTag $tags file]
	lappend Directory($w,selectindex) $id
	lappend Directory($w,selectrect) $rect
    }
    $w raise
}

#
# DirectoryGetPattern - get a pattern for file matching
#
proc code::DirectoryGetPattern {w} {
    variable Directory

    if {$w == "."} {
        set box .pattern
    } else {
	set box $w.pattern
    }

    toplevel $box 
    wm transient $box $w
    wm protocol $box WM_DELETE_WINDOW "set code::Directory($w,pattern) {}"
    set f [frame $box.buttons]
    set okBtn [button $f.ok -text Ok -width 6 \
        -command "set code::Directory($w,pattern) \$code::Directory($w,patternentry)"]
    set cancelBtn [button $f.cancel -text Cancel -width 6 \
        -command "set code::Directory($w,pattern) {}"]
    grid $okBtn -in $f -row 0 -column 0 
    grid $cancelBtn -in $f -row 0 -column 1 
    grid $f -in $box -row 2 -column 0 -sticky ew -padx 5 -pady 5
    set e [entry $box.entry -textvariable code::Directory($w,patternentry) -width 30]
    bind $e <Key-Return> "set code::Directory($w,pattern) \$code::Directory($w,patternentry)"
    grid $e -in $box -row 1 -column 0 -sticky ew -padx 5 -pady 5
    set l [label $box.l -text "Enter a file pattern to match"]
    grid $l -in $box -row 0 -column 0 -sticky ew -padx 5 -pady 5

    placewindow $box widget $w
    wm title $box "File Matching Pattern"
    # bind $box <F1> "code::OpenHelp Applications Directory Goto.html"

    set oldFocus [focus]
    set oldGrab [grab current $box]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $box
    focus $e

    tkwait variable code::Directory($w,pattern)
    catch {focus $oldFocus}
    grab release $box
    destroy $box
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    return $Directory($w,pattern)
}

#
# DirectoryMenuEdit - generate the Edit menu
#
proc code::DirectoryMenuEdit {w menu} {

    MenuCommand $w Edit "Select All" -cmd selectall -help "Select all files in this directory"
    MenuBind $w <Control-A> Edit "Select All"
    MenuCommand $w Edit "Select Files..." -cmd selectfiles -help "Select matching files in this directory"
    MenuBind $w <Control-S> Edit "Select Files..."
}

#
# DirectoryPwd - like pwd but removes the / from a DOS drive letter
#
proc code::DirectoryPwd {} {
    variable DirectoryDos

    set pwd [pwd]
    if {!$DirectoryDos} {
	return $pwd
    }

    if {[file split $pwd] != $pwd} {
	# not a drive
	return $pwd
    }
    regexp {(.*:)/} $pwd all drive
    return $drive
}

#
# DirectoryEdit - edit a file name in a directory entry
#
proc code::DirectoryEdit {w c id K A} {
    variable Directory

    switch -exact $K {
	Escape {
	    # restore original value
	    $c itemconfig $id -text $Directory($w,selectorig) 
            DirectoryClearSelect $w $c
	    return
	}
	Left {
	    # one character left
	    set index [$c index $id insert]
	    incr index -1
	    if {$index >= 0} {
		$c icursor $id $index
	    }
	    return
	}
	Right {
	    # one character right
	    set index [$c index $id insert]
	    incr index
	    if {$index <= [$c index $id end]} {
		$c icursor $id $index
	    }
	    return
	}
	BackSpace {
	    # remove last character
	    set index [$c index $id insert]
	    incr index -1
	    if {$index >= 0} {
	        $c dchars $id $index
	    }
	    return
	}
	Delete {
	    # remove next character
	    $c dchars $id insert
	    return
	}
	Return {
	    # done editing
            DirectoryClearSelect $w $c
	    return
	}
    }

    if {$A == {} || $A == "/" || $A == "\\"} {
	# ignore empty characters and path separators
	return
    }

    # insert this character
    $c insert $id insert $A

    # expand box if it gets bigger
    set bbox [$c bbox $id]
    set rect $Directory($w,selectrect)
    set sbbox [$c bbox $Directory($w,selectrect)]
    set blen [expr {[lindex $bbox 2] - [lindex $bbox 0]}]
    set sblen [expr {[lindex $sbbox 2] - [lindex $sbbox 0]}]
    if {$blen > $sblen} {
        eval $c coords $rect $bbox
	# raise the rectangle and id so they always are on top
	$c raise $rect
	$c raise $id
    }
}

#
# DirectoryEditPos - position the insert cursor in an editable filename
#
proc code::DirectoryEditPos {w c id x y} {
    set index [$c index $id @$x,$y]
    $c icursor $id $index
}

#
# DirectoryTreeSingle - a single click on a directory tree
#
proc code::DirectoryTreeSingle {W x y w} {
    variable FileIcons
    variable DirectoryDos

    set lbl [Tree:labelat $W $x $y]

    Tree:setselection $W $lbl
    if {$DirectoryDos} {
	# remove the leading /
	regexp {/(.*)} $lbl all file
    } else {
	set file $lbl
    }
    if {[file isdirectory $file]} {
        # refresh the directory
        Tree:delitem $W $lbl
	Tree:newitem $W $lbl -image $FileIcons(directory.icon)
	DirectoryFillTree $W $file
        Tree:open $W $lbl
    }
    DirectoryChange $w $lbl/
}

#
# DirectoryTreeDouble - a double click in the directory tree
#
proc code::DirectoryTreeDouble {W x y} {
    set f [Tree:labelat $W $x $y]
    Tree:open $W $f
}

#
# DirectoryFillTree - fill the tree widgit with directories
#
proc code::DirectoryFillTree {w dir} {
    variable FileIcons

    if {[string match */ $dir]} {
        set list [lsort -dictionary [glob -nocomplain $dir*]]
    } else {
        set list [lsort -dictionary [glob -nocomplain $dir/*]]
    }
    foreach z $list {
        if {[file isdirectory $z]} {
	    if {![string match /* $z]} {
		# fix a windows drive
		set z /$z
	    }
            Tree:newitem $w $z -image $FileIcons(directory.icon)
        }
    }
}

#
# DirectoryChange - change the current directory
#
proc code::DirectoryChange {w to} {
    variable Directory
    variable DirectoryDos
    variable PreferenceFile

    if {$DirectoryDos} {
	if {"/[file tail $to]" == $to} {
	    # this is a drive
	    set to [file tail $to]
	} else {
	    # this is a directory
	    regexp {/([^:]*:.*)} $to all to
	}
    }

    # clear a possible input selection
    set c $w.dir
    DirectoryClearSelect $w $c

    if {$PreferenceFile != {}} {
       # preserve project directory
       set pwd [pwd]
    } 
    if {[catch {cd $to} msg]} {
        tk_messageBox -icon error \
            -parent $w \
	    -message $msg -type ok
    } else {
	# get new directory
        $w.dir configure -contents [DirectoryPwd]

	if {$PreferenceFile != {}} {
	    # change back to default (project) directory
            cd $pwd
        }  else {
            # remember the last directory we visit
	    Preference Directory lastdir [pwd]
        }
        DirectoryFill $w
    }
}

#
# DirectoryFill - fill the directory window when idle
#
proc code::DirectoryFill w {
    variable Directory

    if {![info exists Directory($w,buildpending)]} {
        set Directory($w,buildpending) 1
        after idle "catch \{code::DirectoryFillNow $w\}"
    }
}

#
# DirectoryFillNow - display the current directory
#
proc code::DirectoryFillNow {w} {
    variable Directory
    variable FileIcons
    variable Build

    catch {unset Directory($w,buildpending)}
    set current [pwd]
    cd [$w.dir cget -contents]
    set icons 1
    set files [lsort -dictionary [glob -nocomplain *]]
    set directories {}
    set regular {}
    set projects {}
    set count 0
    set longest {0 0 0 0}
    set c $w.dir
    set color [Preference General colorbackground]
    $w config -bg $color
    $c config -bg $color
    $Directory($w,tree) config -bg $color
    DirectoryClearSelect $w $c
    $c delete all
    # set entry font

    # get the list of project files
    set projectfiles {}
    set pwd [$w.dir cget -contents]
    set project $Build(PROJECT)
    foreach file [Preference Build files] {
	set file [BuildSubstitute [lindex $file 0]]
	set dir [file dirname $file]
	set tail [file tail $file]
	if {$pwd != $project && $dir == "."} {
	    # these are in the project directory
	    lappend projectfiles [file join $project $tail]
	} elseif {$pwd == $dir} {
	    # these are in the current directory
	    lappend projectfiles $tail
	} else {
	    # somewhere else
	    lappend projectfiles $file
	}
    }

    # get the subprojects
    set subprojects {}
    foreach file [Preference Build subprojects] {
	set file [BuildSubstitute [lindex $file 0]]
	set dir [file dirname $file]
	set tail [file tail $file]
	if {$pwd != $project && $dir == "."} {
	    # these are in the project directory
	    lappend subprojects [file join $project $tail]
	} elseif {$pwd == $dir} {
	    # these are in the current directory
	    lappend subprojects $tail
	} else {
	    # somewhere else
	    lappend subprojects $file
	}
    }
    # sort out directories, count entries build name identifiers
    foreach file $files {
	# create the name holder in a out-of-the-way place
	if {[lsearch $subprojects $file] != -1} {
            if {[Preference Directory showproject]} {
		# show subprojects later
		continue
	    }
	    set color [Preference Directory colorproject]
	    set projectfiles [lremove $projectfiles $file]
	    set project 1
	    
	} elseif {[lsearch $projectfiles $file] == -1} {
	    # not in project
	    set color [Preference Directory colorentry]
	    set project 0
            if {![Preference Directory showregular]} {
		continue
	    }
	} else {
	    set color [Preference Directory colorproject]
	    set projectfiles [lremove $projectfiles $file]
	    set project 1
	}
        set id [$c create text 1000 1000 -text $file \
	    -anchor nw -justify left -width 0 -tags {entry name} \
	    -font [Preference Directory fontdirectory] \
	    -fill $color \
	    ]

	if {[catch {file stat $file stat}]} {
	    # file is inaccessable for some reason
	    set stat(type) file
	    if {$project} {
		# not found in project
		set project 2
	    }
	}
        if {$stat(type) == "directory"} {
	    lappend directories "[list $file] directory $id $project"
        } else {
	    set ext [file extension $file]
	    switch $ext {
		.code -
		.cod {
	            lappend projects "[list $file] $stat(type) $id $project"
	        }
                default {
	            lappend regular "[list $file] $stat(type) $id $project"
		}
	    }
        }
	# find the longest filename
        set bbox [$c bbox $id]
        if {([lindex $bbox 2] - [lindex $bbox 0]) > \
	    ([lindex $longest 2] - [lindex $longest 0])} {
	    set longest $bbox
	}
        incr count
    }
    set files [concat $projects $directories $regular]

    set info [grid info $c]
    set grid [lindex $info [expr {[lsearch $info -in] + 1}]]
    set row [lindex $info [expr {[lsearch $info -row] + 1}]]
    set column [lindex $info [expr {[lsearch $info -column] + 1}]]
    set height [winfo height $c]
    set width [winfo width $c]
    # set up the longest entry
    set deltax [expr {[lindex $longest 2] - [lindex $longest 0]}]
    set deltay [expr {[lindex $longest 3] - [lindex $longest 1]}]
    if {$icons} {
        if {$FileIcons(iconheight) > $deltay} {
	    set deltay $FileIcons(iconheight)
        }
        incr deltax $FileIcons(iconwidth)
    }
    set columns [expr {int($width / $deltax)}]
    if {$columns < 1} {
	set columns 1
    }
    set maxx [expr {$columns * $deltax}]
    set maxy [DirectorySendFiles $w $c $icons $deltax $deltay $files $count $columns $maxx]
    if {[Preference Directory showproject]} {
	# show all project files in directory window
        set projectfiles [lsort -dictionary $projectfiles]
	set regular {}
	set directories {}
	set count 0
        foreach file $projectfiles {
	    set file [BuildRealFileName $file]
	    if {[catch {file stat $file stat}]} {
		# file is inaccessable for some reason
	        set stat(type) file
		set color [Preference Directory colormissing]
		set inproject 2
	    } else {
	        set color [Preference Directory colorproject]
		set inproject 1
	    }
	    # create the name holder in a out-of-the-way place
            set id [$c create text 1000 1000 -text $file \
	        -anchor nw -justify left -width 0 -tags {entry name} \
	        -font [Preference Directory fontdirectory] \
	        -fill $color \
	        ]
            if {$stat(type) == "directory"} {
	        lappend directories "[list $file] directory $id $inproject"
            } else {
	        lappend regular "[list $file] $stat(type) $id $inproject"
            }
	    # find the longest filename
            set bbox [$c bbox $id]
            if {([lindex $bbox 2] - [lindex $bbox 0]) > \
	        ([lindex $longest 2] - [lindex $longest 0])} {
	        set longest $bbox
	    }
            incr count
        }
	if {$count} {
            # set up the longest entry
            set deltax [expr {[lindex $longest 2] - [lindex $longest 0]}]
            set deltay [expr {[lindex $longest 3] - [lindex $longest 1]}]
            if {$icons} {
                if {$FileIcons(iconheight) > $deltay} {
	            set deltay $FileIcons(iconheight)
                }
                incr deltax $FileIcons(iconwidth)
            }
            set columns [expr {int($width / $deltax)}]
            if {$columns < 1} {
	        set columns 1
            }
            set maxx [expr {$columns * $deltax}]
            set files [concat $directories $regular]
            set maxy [DirectorySendFiles $w $c $icons $deltax $deltay $files \
	        $count $columns $maxx [expr {$maxy + 20}]]
	}

	# show any sub-projects
        set projectfiles {}
        foreach file $subprojects {
	    lappend projectfiles [BuildSubstitute $file]
        }
	set regular {}
	set directories {}
	set count 0
        foreach file $projectfiles {
	    # create the name holder in a out-of-the-way place
	    if {[catch {file stat $file stat}]} {
	        continue
	    } else {
	        set color [Preference Directory colorsubproject]
	        set inproject 1
	    }
            set id [$c create text 1000 1000 -text $file \
	        -anchor nw -justify left -width 0 -tags {entry name} \
	        -font [Preference Directory fontdirectory] \
	        -fill $color \
	        ]
            if {$stat(type) == "directory"} {
	            lappend directories "[list $file] directory $id $inproject"
            } else {
	        lappend regular "[list $file] $stat(type) $id $inproject"
            }
	    # find the longest filename
            set bbox [$c bbox $id]
            if {([lindex $bbox 2] - [lindex $bbox 0]) > \
	        ([lindex $longest 2] - [lindex $longest 0])} {
	        set longest $bbox
	    }
            incr count
        }
	if {$count} {
            # set up the longest entry
            set deltax [expr {[lindex $longest 2] - [lindex $longest 0]}]
            set deltay [expr {[lindex $longest 3] - [lindex $longest 1]}]
            if {$icons} {
                if {$FileIcons(iconheight) > $deltay} {
	            set deltay $FileIcons(iconheight)
                }
                incr deltax $FileIcons(iconwidth)
            }
            set columns [expr {int($width / $deltax)}]
            if {$columns < 1} {
	        set columns 1
            }
            set maxx [expr {$columns * $deltax}]
            set files [concat $directories $regular]
            set maxy [DirectorySendFiles $w $c $icons $deltax $deltay $files \
	        $count $columns $maxx [expr {$maxy + 20}]]
        }

	if {[Preference Build target] == "program"} {
	    # make a list of real library names
            set projectfiles {}
            foreach file [Preference Build libraries] {
	        lappend projectfiles [BuildSubstitute [lindex $file 0]]
            }
	    set regular {}
	    set directories {}
	    set count 0
            foreach file $projectfiles {
	        # create the name holder in a out-of-the-way place
	        if {[catch {file stat $file stat}]} {
		    continue
	        } else {
	            set color [Preference Directory colorlibrary]
		    set inproject 1
	        }
                set id [$c create text 1000 1000 -text $file \
	            -anchor nw -justify left -width 0 -tags {entry name} \
	            -font [Preference Directory fontdirectory] \
	            -fill $color \
	            ]
                if {$stat(type) == "directory"} {
	            lappend directories "[list $file] directory $id $inproject"
                } else {
	            lappend regular "[list $file] $stat(type) $id $inproject"
                }
	        # find the longest filename
                set bbox [$c bbox $id]
                if {([lindex $bbox 2] - [lindex $bbox 0]) > \
	            ([lindex $longest 2] - [lindex $longest 0])} {
	            set longest $bbox
	        }
                incr count
            }
	    if {$count} {
                # set up the longest entry
                set deltax [expr {[lindex $longest 2] - [lindex $longest 0]}]
                set deltay [expr {[lindex $longest 3] - [lindex $longest 1]}]
                if {$icons} {
                    if {$FileIcons(iconheight) > $deltay} {
	                set deltay $FileIcons(iconheight)
                    }
                    incr deltax $FileIcons(iconwidth)
                }
                set columns [expr {int($width / $deltax)}]
                if {$columns < 1} {
	            set columns 1
                }
	        set columns 1
                set maxx [expr {$columns * $deltax}]
                set files [concat $directories $regular]
                set maxy [DirectorySendFiles $w $c $icons $deltax $deltay $files \
	            $count $columns $maxx [expr {$maxy + 20}]]
	    }
        }
    }
    bind $c <Configure> {}
    $c config -scrollregion "0 0 [expr {$deltax > $width ? $deltax : 0}] $maxy"
    $c xview moveto 0
    $c yview moveto 0
    bind $c <Configure> "code::DirectoryFill $w"
    cd $current
    set file [file nativename [$w.dir cget -contents]]
    set image $FileIcons(directory.icon)
    $w configure -image $image -icontext [file tail $file] -title "Folder: $file"
}

#
# DirectorySendFiles - place files and icons on the directory canvas
#
proc code::DirectorySendFiles {w c icons deltax deltay files count columns maxx {starty 0}} {
    variable Directory
    variable FileIcons
    variable PreferenceFile
    variable Build

    incr deltax 2
    incr deltay 2
    set percolumn [expr {ceil(double($count) / $columns)}]
    set maxy $starty
    set x 0
    set y $starty
    set count 0
    set pwd [$w.dir cget -contents]
    if {$Build(PROJECT) == $pwd} {
	# in the project directory
	set project 1
    } else {
	set project 0
    }
    foreach info $files {
	set file [lindex $info 0]
	set type [lindex $info 1]
	set id [lindex $info 2]
	set inproject [lindex $info 3]
        set bbox [$c bbox $id]
	set extension [string tolower [file extension $file]]
	set Type [Preference FileTypes $type$extension]
	if {$Type == {}} {
	    set Type [Preference FileTypes $type]
	    if {$Type == {}} {
	        error "File type botch."
	    }
	}
	set desc [lindex $Type 1]
	set commands [lindex $Type 2]
	set class [lindex $Type 3]
	if {   $class == "project" \
	    && $project
	    && $PreferenceFile == $file} {
	    set inproject 3
	    $c itemconfig $id -fill [Preference Directory colorproject]
	}
	if {$icons} {
	    set image $FileIcons([lindex $Type 0])
	    set img [$c create image $x $y -anchor nw -image $image -tag entry]
	    $c coords $id [expr {$x + $FileIcons(iconwidth)}] $y
	} else {
	    $c coords $id $x $y
	}
        variable DirectoryDos
	if {$DirectoryDos && [file split $pwd] == $pwd} {
	    # fix DOS drive
	    set file file:[file join $pwd/ $file]
	} else {
	    set file file:[file join $pwd $file]
	}
	$c addtag "desc:$desc" withtag $id
	$c addtag "text:$id" withtag $id
	$c addtag "img:$img" withtag $id
	$c addtag $file withtag $id
	$c addtag "cmds:$commands" withtag $id
	$c addtag "type:$type" withtag $id
	$c addtag "project:$inproject" withtag $id
	$c addtag "desc:$desc" withtag $img
	$c addtag "text:$id" withtag $img
	$c addtag $file withtag $img
	$c addtag "cmds:$commands" withtag $img
	$c addtag "type:$type" withtag $img
	$c addtag "project:$inproject" withtag $img
        incr y $deltay
	incr count
	if {$count >= $percolumn} {
	    # go to the next column
	    set count 0
	    set x [expr {$x + $deltax}]
	    if {$y > $maxy} {
		set maxy $y
	    }
	    set y $starty
	}
    }
    return $maxy
}

#
# DirectoryFindTag - find a labeled tag in a tag list
#
proc code::DirectoryFindTag {tags label} {
    set label $label:
    foreach tag $tags {
	if {[string match $label* $tag]} {
	    return [string range $tag [string length $label] end]
	}
    }
    return ""
}

#
# DirectoryEntryBinding - handle bindings on directory entries
#
proc code::DirectoryEntryBinding {w cmd c x y X Y} {
    variable Directory

    set id [$c find closest [$c canvasx $x] [$c canvasy $y] 0]
    set tags [$c gettags $id]
    switch -exact $cmd {
	enter {
	    set words [DirectoryFindTag $tags desc]
	    set project [DirectoryFindTag $tags project]
	    if {$words == "folder"} {
	        set desc "A folder"
	    } elseif {$words != ""} {
	        set char [string tolower [string index $words 0]]
	        if {[string first $char "aeiouh"] != -1} {
		    set first "An"
	        } else {
	            set first "A"
	        }
	        set desc "$first $words file"
	    } else {
	        set desc "A file"
	    }
	    switch $project {
		1 { append desc ", in project" }
		2 { append desc ", in project but does not exist" }
		3 { append desc ": currently open" }
	    }
	    status . $desc
	}
	leave {
	    status . ""
	}
	btn1m -
	btn1 {
	    if {$cmd != "btn1m" && $tags == ""} {
	        DirectoryClearSelect $w $c
	        $w raise
		return
	    }
	    set index [DirectoryFindTag $tags text]
	    if {$index == ""} {
	        $w raise
		return
	    }
	    if {$Directory($w,selectindex) == $index} {
		# already selected
                if {$Directory($w,selectinput)} {
		    return
		} else {
		    DirectoryUnSelect $w $c $index
		}
	        $w raise
		return
	    }
	    # deselect old items
	    DirectoryClearSelect $w $c
	    # select an item
            set sc [Preference General colorselection]
	    set rect [$c create rect 0 0 0 0 -fill $sc -outline $sc]
	    $c lower $rect
	    set bbox [$c bbox $index]
	    eval $c coords $rect $bbox
	    set Directory($w,select) [list [DirectoryFindTag $tags file]]
	    set Directory($w,selectindex) $index
	    set Directory($w,selectrect) $rect
	    set Directory($w,selectinput) 0
	    $w raise
	}
	shift1 {
	    set index [DirectoryFindTag $tags text]
	    if {$index == ""} {
		return
	    }
	    if {[lsearch $Directory($w,selectindex) $index] != -1} {
		# already selected
                if {$Directory($w,selectinput)} {
	            DirectoryClearSelect $w $c
		} else {
		    DirectoryUnSelect $w $c $index
		}
	        $w raise
		return
	    }
	    # add to a selection
            set sc [Preference General colorselection]
	    set rect [$c create rect 0 0 0 0 -fill $sc -outline $sc]
	    $c lower $rect
	    set bbox [$c bbox $index]
	    eval $c coords $rect $bbox
	    lappend Directory($w,select) [DirectoryFindTag $tags file]
	    lappend Directory($w,selectindex) $index
	    lappend Directory($w,selectrect) $rect
	    $w raise
	}
	dbl1 {
	    # try to open this file
	    OpenFile [list [DirectoryFindTag $tags file]]
	}
	btn3 {
	    set index [DirectoryFindTag $tags text]
	    if {$index == ""} {
		return
	    }
	    if {[lsearch $Directory($w,selectindex) $index] == -1} {
		# not selected, select
	        DirectoryEntryBinding $w btn1 $c $x $y $X $Y
	    }
	    $w raise
	    ToolGenerate popup [DirectoryFindTag $tags cmds] $w . $x $y $X $Y
	}
    }
}

#
# DirectoryClearSelect - clear any old selections
#
proc code::DirectoryClearSelect {w c} {
    variable Directory
    variable DirectoryDos
    variable FileIcons

    foreach rect $Directory($w,selectrect) {
	$c delete $rect
    }
    if {$Directory($w,selectinput)} {
	set Directory($w,selectinput) 0
	# was an input selection
	$c focus {}
	set id $Directory($w,selectindex)
	set orig $Directory($w,select)
	set new [$c itemcget $id -text]
	if {[file dirname $new] == "."} {
	    set pwd [$w.dir cget -contents]
	    if {$DirectoryDos && [file split $pwd] == $pwd} {
	        # fix DOS drive
	        set new [file join $pwd/ $new]
	    } else {
	        set new [file join $pwd $new]
	    }
	}
	if {$orig != $new} {
	    # rename this file
	    if {[file exists $new]} {
		# file already exists
	        $c itemconfig $id -text $Directory($w,selectorig) 
                tk_messageBox -icon error \
		    -parent $w \
	            -message "\"$new\" already exists." -type ok
	    } elseif {[catch {file rename $orig $new} msg]} {
		# error renaming
	        $c itemconfig $id -text $Directory($w,selectorig) 
                tk_messageBox -icon error \
		    -parent $w \
	            -message "$msg" -type ok
	    } else {
		# renamed: need to fix the file:* tag
                set tags [$c gettags $id]
	        set name [DirectoryFindTag $tags file]
		# delete the old tag
		$c dtag $id file:$name
		$c addtag file:$new withtag $id

		# change the name in the icon, also
		set img [DirectoryFindTag $tags img]
		$c dtag $img file:$name
		$c addtag file:$new withtag $img

		# if this is in the current project, rename it
		ConfigureRename $orig $new

		# check to see if the editor has this file open
		EDITrename $orig $new

                # refresh the directory tree
	        set dir [$w.dir cget -contents]
                if {$DirectoryDos} {
	            # add a leading /
	            set dir /$dir
                } 
	        set t $Directory($w,tree)
                Tree:delitem $t $dir
	        Tree:newitem $t $dir -image $FileIcons(directory.icon)
	        DirectoryFillTree $t [$w.dir cget -contents]
                Tree:setselection $t $dir
                Tree:open $t $dir
	        Tree:show $t $dir

		# refresh the directory
		DirectoryFill $w
	    }
	}
    }
    set Directory($w,select) {}
    set Directory($w,selectindex) {}
    set Directory($w,selectrect) {}
}

#
# DirectoryUnSelect - clear an selection
#
proc code::DirectoryUnSelect {w c this} {
    variable Directory
    variable DirectoryDos
    variable FileIcons

    set index 0
    set found 0
    foreach id $Directory($w,selectindex) {
	if {$this == $id} {
	    set found 1
            $c delete [lindex $Directory($w,selectrect) $index]
	    break
	}
	incr index
    }
    if {!$found} {
	return
    }
    set Directory($w,select) [lreplace $Directory($w,select) $index $index]
    set Directory($w,selectindex) [lreplace $Directory($w,selectindex) $index $index]
    set Directory($w,selectrect) [lreplace $Directory($w,selectrect) $index $index]
}

#
# Copyright (C) 1997,1998 D. Richard Hipp
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
# 
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA  02111-1307, USA.
#
# Author contact information:
#   drh@acm.org
#   http://www.hwaci.com/drh/
#
# $Revision: 1.139 $
#

#
# Create a new tree widget.  $args become the configuration arguments to
# the canvas widget from which the tree is constructed.
#
proc code::Tree:create {w args} {
  variable Tree
  eval canvas $w $args
  bind $w <Destroy> "code::Tree:delitem $w /"
  Tree:dfltconfig $w /
  Tree:buildwhenidle $w
  set Tree($w:selection) {}
  set Tree($w:selidx) {}
  set Tree($w:font) {}
}

# Initialize a element of the tree.
# Internal use only
#
proc code::Tree:dfltconfig {w v} {
  variable Tree
  set Tree($w:$v:children) {}
  set Tree($w:$v:open) 0
  set Tree($w:$v:icon) {}
  set Tree($w:$v:y) 0
}

#
# Pass configuration options to the tree widget
#
proc code::Tree:config {w args} {
  eval $w config $args
}

#
# Insert a new element $v into the tree $w.
#
proc code::Tree:newitem {w v args} {
  variable Tree
  variable DirectoryDos

    if {$DirectoryDos && "/[file tail $v]" != $v} {
        # is not a drive
        regexp {(.*)/([^/]*)} $v all dir n
    } else {
        set dir [file dirname $v]
        set n [file tail $v]
    }
    if {![info exists Tree($w:$dir:open)]} {
	Tree:newitem $w $dir
    }

  set i [lsearch -exact $Tree($w:$dir:children) $n]
  if {$i>=0} {
    error "item \"$v\" already exists"
  }
  lappend Tree($w:$dir:children) $n
  set Tree($w:$dir:children) [lsort -dictionary $Tree($w:$dir:children)]
  Tree:dfltconfig $w $v
  foreach {op arg} $args {
    switch -exact -- $op {
      -image {set Tree($w:$v:icon) $arg}
    }
  }
  Tree:buildwhenidle $w
}

#
# Delete element $v from the tree $w.  If $v is /, then the widget is
# deleted.
#
proc code::Tree:delitem {w v} {
  variable Tree
  variable DirectoryDos

  if {![info exists Tree($w:$v:open)]} return
  if {[string compare $v /]==0} {
    # delete the whole widget
    catch {destroy $w}
    foreach t [array names Tree $w:*] {
      unset Tree($t)
    }
    return
  }
  foreach c $Tree($w:$v:children) {
    Tree:delitem $w $v/$c
  }
  unset Tree($w:$v:open)
  unset Tree($w:$v:children)
  unset Tree($w:$v:icon)
  unset Tree($w:$v:y)
    if {$DirectoryDos && "/[file tail $v]" != $v} {
        # is not a drive
        regexp {(.*)/([^/]*)} $v all dir n
    } else {
        set dir [file dirname $v]
        set n [file tail $v]
    }
  set i [lsearch -exact $Tree($w:$dir:children) $n]
  if {$i>=0} {
    set Tree($w:$dir:children) [lreplace $Tree($w:$dir:children) $i $i]
  }
  Tree:buildwhenidle $w
}

#
# Change the selection to the indicated item
#
proc code::Tree:setselection {w v} {
  variable Tree

  set Tree($w:selection) $v
  Tree:drawselection $w
}

# 
# Retrieve the current selection
#
proc code::Tree:getselection w {
  variable Tree

  return $Tree($w:selection)
}

#
# Bitmaps used to show which parts of the tree can be opened.
#
set maskdata "#define solid_width 9\n#define solid_height 9"
append maskdata {
  static unsigned char solid_bits[] = {
   0xff, 0x01, 0xff, 0x01, 0xff, 0x01, 0xff, 0x01, 0xff, 0x01, 0xff, 0x01,
   0xff, 0x01, 0xff, 0x01, 0xff, 0x01
  };
}
set data "#define open_width 9\n#define open_height 9"
append data {
  static unsigned char open_bits[] = {
   0xff, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x7d, 0x01, 0x01, 0x01,
   0x01, 0x01, 0x01, 0x01, 0xff, 0x01
  };
}
image create bitmap Tree:openbm -data $data -maskdata $maskdata \
  -foreground black -background white
set data "#define closed_width 9\n#define closed_height 9"
append data {
  static unsigned char closed_bits[] = {
   0xff, 0x01, 0x01, 0x01, 0x11, 0x01, 0x11, 0x01, 0x7d, 0x01, 0x11, 0x01,
   0x11, 0x01, 0x01, 0x01, 0xff, 0x01
  };
}
image create bitmap Tree:closedbm -data $data -maskdata $maskdata \
  -foreground black -background white

# Internal use only.
# Draw the tree on the canvas
proc code::Tree:build w {
  variable Tree

  $w delete all
  catch {unset Tree($w:buildpending)}
  set Tree($w:y) 30
  Tree:buildlayer $w / 10
  $w config -scrollregion [$w bbox all]
  Tree:drawselection $w
  set all [$w bbox all]
  set width [expr {[lindex $all 2] - [lindex $all 0] + 5}]
  # set min, max width
  set max [expr {[winfo width [winfo toplevel $w]] / 2}]
  if {$width < 100} {
      set width 100
  } elseif {$width > $max} {
      set width $max
  }
  $w config -width $width
}

# Internal use only.
# Build a single layer of the tree on the canvas.  Indent by $in pixels
proc code::Tree:buildlayer {w v in} {
  variable Tree

  if {$v=="/"} {
    set vx {}
  } else {
    set vx $v
  }
  # remember this placement
  set Tree($w:$v:y) [expr {$Tree($w:y) - 50}]
  set start [expr {$Tree($w:y)-10}]
  foreach c $Tree($w:$v:children) {
    set y $Tree($w:y)
    incr Tree($w:y) 17
    $w create line $in $y [expr {$in+10}] $y -fill gray50 
    set icon $Tree($w:$vx/$c:icon)
    set x [expr {$in+12}]
    if {[string length $icon]>0} {
      set k [$w create image $x $y -image $icon -anchor w -tags x]
      incr x 20
      set Tree($w:tag:$k) $vx/$c
    }
    set Tree($w:$vx/$c:y) [expr {$y - 50}]
    set j [$w create text $x $y -text $c \
	-font [Preference Directory fontdirectory] -anchor w -tags x]
    set Tree($w:tag:$j) $vx/$c
    set Tree($w:$vx/$c:tag) $j
    if {[string length $Tree($w:$vx/$c:children)]} {
      if {$Tree($w:$vx/$c:open)} {
         set j [$w create image $in $y -image Tree:openbm]
         $w bind $j <1> "set \"code::Tree($w:$vx/$c:open)\" 0; code::Tree:build $w"
         Tree:buildlayer $w $vx/$c [expr {$in+18}]
      } else {
         set j [$w create image $in $y -image Tree:closedbm]
         $w bind $j <1> "set \"code::Tree($w:$vx/$c:open)\" 1; code::Tree:build $w"
      }
    }
  }
  if {[info exists y]} {
      set j [$w create line $in $start $in [expr {$y+2}] -fill gray50 ]
      $w lower $j
  }
}

# Open a branch of a tree
#
proc code::Tree:open {w v} {
  variable Tree

  if {[info exists Tree($w:$v:open)] && $Tree($w:$v:open)==0
      && [info exists Tree($w:$v:children)] 
      && [string length $Tree($w:$v:children)]>0} {
    set Tree($w:$v:open) 1
    Tree:build $w
  }
}

# Show a branch of a tree
#
proc code::Tree:show {w v} {
  variable Tree

  update idletasks
  set all [$w bbox all]
  set yview [expr $Tree($w:$v:y).0 / ([lindex $all 3].0 - [lindex $all 1].0)]
  $w yview moveto $yview
}

proc code::Tree:close {w v} {
  variable Tree

  if {[info exists Tree($w:$v:open)] && $Tree($w:$v:open)==1} {
    set Tree($w:$v:open) 0
    Tree:build $w
  }
}

# Internal use only.
# Draw the selection highlight
proc code::Tree:drawselection w {
  variable Tree

  if {[string length $Tree($w:selidx)]} {
    $w delete $Tree($w:selidx)
  }
  set v $Tree($w:selection)
  if {[string length $v]==0} return
  if {![info exists Tree($w:$v:tag)]} return
  set bbox [$w bbox $Tree($w:$v:tag)]
  if {[llength $bbox]==4} {
    set sc [Preference General colorselection]
    set i [eval $w create rectangle $bbox -fill $sc -outline {{}}]
    set Tree($w:selidx) $i
    $w lower $i
  } else {
    set Tree($w:selidx) {}
  }
}

# Internal use only
# Call Tree:build then next time we're idle
proc code::Tree:buildwhenidle w {
  variable Tree

  if {![info exists Tree($w:buildpending)]} {
    set Tree($w:buildpending) 1
    after idle "code::Tree:build $w"
  }
}

#
# Return the full pathname of the label for widget $w that is located
# at real coordinates $x, $y
#
proc code::Tree:labelat {w x y} {
  variable Tree

  set x [$w canvasx $x]
  set y [$w canvasy $y]
  foreach m [$w find overlapping $x $y $x $y] {
    if {[info exists Tree($w:tag:$m)]} {
      return $Tree($w:tag:$m)
    }
  }
  return ""
}
