#
# Editor - the CODE editor
#

#
# EDITstartup - return a command to recreate an editor window
#
proc code::EDITstartup {doc} {
    set name [BuildFileName  [$doc.contents cget -contents]]
    return "EDIT [list $name] [$doc.contents index insert]"
}

#
# EDIT - start an edit window
#
proc code::EDIT {{file {}} {pos 1.0} args} {
    variable Build
    variable CODEevents
    variable CODEID
    variable TEXTusebindings
    variable EDITppcontext
    variable CODEbindkeys

    regexp {([0-9]+)\.([0-9]+)} $pos all line column
    set file [BuildSubstitute $file]
    if {$file != {} && [file pathtype $file] == "relative"} {
        set file [file join [pwd] $file]
    }
    if {$file != {} && ![file exists $file] && [CODEfind Edit* $file] == {}} {
	# this should only happen at startup
	error "$file does not exist"
    }

    set edit [Preference General EDIT]
    if {$edit != {}} {
        # use an external editor

        set Build(FILE) [file nativename $file]
        set Build(LINE) $line
        set Build(COLUMN) $column
        set edit [BuildSubstitute $edit]
        eval exec $edit &
        return $file
    }
    # use the CODE editor
    set doc .work.doc$CODEID
    incr CODEID
    
    document $doc
    bind $doc <<State>> "code::EDITstate $doc; break"
    set new 0
    if {$file == {} || ![file exists $file]} {
	set new 1
    }
    $doc configure -type Editor
    editor $doc.contents -scrollbar auto -wrap none \
	-positionproc "code::EDITposition $doc" \
	-line $line -column $column -new $new -highlightthickness 0
    DOCraise $doc
    foreach "bind key" $TEXTusebindings {
	bind $doc.contents <<$bind>> "[bind Text <$key>]; break"
    }
    foreach bind [array names CODEevents] {
	set command [lindex $CODEevents($bind) 6]
	if {$command == {}} {
	    # this uses a previous binding
	    continue
	}
	bind $doc.contents <<$bind>> $command
	if {[info exists CODEbindkeys($bind)]} {
	    if {$CODEbindkeys($bind) != {}} {
	        # bind keys to text widget
	        bind $doc.contents.text.t <$CODEbindkeys($bind)> \
		    "event generate $doc.contents <<$bind>>; break"
	    }
	}
    }
    $doc pack $doc.contents -fill both -expand 1
    $doc menu entryconf Close -command "$doc.contents destroy $doc"
    # check to see if this file is a source file
    set extension [string tolower [file extension $file]]
    set type [Preference FileTypes file$extension]
    set type [lindex $type 3]
    if {$type == "c" || $type == "asm"} {
	# add the info widget
	$doc.contents configure -infowidth 2
	bind [$doc.contents infotext] <1> "code::EDITinfoclick $doc %x %y; $doc raise"
	bind [$doc.contents infotext] <Motion> "code::EDITinfomove $doc %x %y"
	bind [$doc.contents infotext] <Leave> "code::status . {}"
	$doc.contents infotag configure breakpoint -justify center
	$doc.contents infotag configure inactivebreakpoint -justify center
    } 
    $doc.contents open $file
    if {[Preference Editor showcolors] && ![info exists EDITppcontext($file)]} {
	# color the syntax elements
        set EDITppcontext($file) [colorText $doc.contents $file]
    }
    EDITmarkdocerrors $doc $file
    bind $doc.contents <1> "[bind Editor <1>]; $doc raise" 
    bind $doc.contents <3> "$doc raise; code::ToolGenerate popup editorpopup $doc . %x %y %X %Y"
    TEXTpreferences $doc
    PreferenceWhenChanged Editor $doc "code::TEXTpreferences $doc"
    PreferenceWhenChanged General $doc "code::TEXTpreferences $doc"
    PreferenceWhenChanged Control $doc "code::TEXTpreferences $doc"
    PreferenceWhenChanged Debugger $doc "code::TEXTpreferences $doc"
    $doc.contents configure -changeproc "code::EDITchanged $doc"
    $doc.contents configure -filewriteproc "code::DirectoryRefresh"
    $doc configure -startupproc "code::EDITstartup"
    EDITtitle $doc
    if {$args != {}} {
	# use previous values
        eval $doc configure $args
    } else {
        set x [$doc cget -x]
        set y [$doc cget -y]
        set width [expr {[winfo width .work] - 10 - $x}]
        set height [expr {[winfo height .work] - 10 - $y}]
        $doc config -width $width -height $height
    }
    $doc configure -raiseproc code::DOCraise
    set diff [expr [$doc.contents cget -height] / 2]
    after idle "$doc.contents see \"$pos - ${diff}l\""
    return $doc
}

#
# EDITrename - see if this file is being edited and rename it
#
proc code::EDITrename {orig new} {
    variable EDITppcontext

    foreach doc [allDocs] {
	if {   [$doc cget -type] == "Editor"
	    && [$doc.contents cget -contents] == $orig} {
	    $doc.contents configure -contents $new
            if {[info exists EDITppcontext($orig)]} {
		set EDITppcontext($new) EDITppcontext($orig)
		unset EDITppcontext($orig)
	    }
	    break
	}
    }
}

#
# EDITinfois - what is at this place in an info area?
#
proc code::EDITinfois {i x y} {
    set index [$i index @$x,$y]
    set value [$i get "$index linestart" "$index lineend"]
    if {$value == {}} {
        set value [$i tag names $index]
    }
    return $value
}

#
# EDITinfoclick - the info section of an editor has been hit with Mouse-1
#
proc code::EDITinfoclick {doc x y} {
    variable Debugger

    set i [$doc.contents infotext]
    switch -- [EDITinfois $i $x $y] {
	"" {
            set end [$doc.contents infoindex "end - 1c"]
            if {$Debugger(opened) != {} && $end == 1.0} {
	        EDITsetupinfo $doc
	    }
	}
	"breakpoint" -
	"inactivebreakpoint" -
	"--" {
            DebuggerSetBreakpoint line [list $doc.contents [$i index @$x,$y]]
	}
	default {
	    # nothing
	}
    }
}

#
# EDITinfomove - the info section of an editor is being passed over by the mouse
#
proc code::EDITinfomove {doc x y} {
    variable Debugger

    set i [$doc.contents infotext]
    switch -- [EDITinfois $i $x $y] {
	"" {
            set end [$doc.contents infoindex "end - 1c"]
            if {$Debugger(opened) != {} && $end == 1.0} {
	        status . "Click to get line information for this file"
	    } else {
	        status . ""
	    }
	}
	"breakpoint" {
	    status . "Click to inactivate the breakpoint on this line"
	}
	"inactivebreakpoint" {
	    status . "Click to remove the breakpoint on this line"
	}
	"--" {
	    status . "Click to set a breakpoint on this line"
	}
	default {
	    # nothing
	}
    }
}

#
# EDITsetupinfo - set up info for an editor
#
proc code::EDITsetupinfo {doc} {
    variable Debugger
    variable DebuggerAtBreakpoints

    set file [$doc.contents cget -contents]
    set modified [$doc.contents ismodified]
    if {$modified} {
        # Warning, Will Robinson! The editor has been modified.
        return 0
    } elseif {[$doc.contents cget -infowidth]} {
        # have an info widget
        set end [$doc.contents infoindex "end - 1c"]
        if {($Debugger(newfile) || $end == 1.0) && $Debugger(opened) != {}} {
	    # no info yet
	    set file [$doc.contents cget -contents]
	    set end [$doc.contents index end]
	    regexp {([0-9]+)\.[0-9]+} $end all end
	    incr end -1
	    $doc.contents infodelete 1.0 end
	    status . "Setting up [file nativename $file]"
	    # mark executable lines
	    $doc.contents infoinsert 1.0 [DebuggerDbg breakpoint getlines $file $end]
	    # show current breakpoints
	    foreach where [array names DebuggerAtBreakpoints '$file'@*] {
	        EDITshowbreak $where $doc
	    }
	    status . ""
	    .status configure -progress 0.0
        }
    }
    return 1
}

#
# EDITstate - called when the debugger state changes
#
proc code::EDITstate {doc} {
    variable Debugger

    if {![$doc.contents isfirst]} {
	# let the first window to a file get the changes.
	# they will propogate to all the rest
	return
    }
    if {!$Debugger(newfile) && !$Debugger(pcchanged)} {
	# no change in the PC or object file
	return
    }
    # remove any current source tag
    $doc.contents tag remove $Debugger(tag) 1.0 end
    if {!$Debugger(running)} {
	# the program has stopped
	set file $Debugger(source)
        # delete disassembly text
        foreach "from to" [$doc.contents tag ranges disassembly] {
	    $doc.contents quietdelete $from $to
        }
        if {[$doc.contents cget -infowidth]} {
            foreach "from to" [$doc.contents infotag ranges disassembly] {
                $doc.contents infodelete $from $to
            }
        }
	if {   $file != {}
	    && $Debugger(tagstart) != {}
	    && [string equal $file [$doc.contents cget -contents]]} {
	    # file in this editor, set execution marker

	    EDITsetupinfo $doc
            $doc.contents tag add $Debugger(tag) $Debugger(tagstart) $Debugger(tagend)
	    # make sure sel stays on top
	    $doc.contents tag raise sel
	    set modified [$doc.contents ismodified]
	    if {   !$modified
	        && $Debugger(mixed) && $Debugger(disassembly) != {}} {
		# display disassembly of this line
                $doc.contents quietinsert $Debugger(tagend) $Debugger(disassembly) disassembly
		if {[$doc.contents cget -infowidth]} {
		    # clear the info for disassembly lines
		    foreach "start end" [$doc.contents tag ranges disassembly] {
		        if {[$doc.contents compare $start == $Debugger(tagend)]} {
			    set first $start
			    set info ""
			    set current [$doc.contents index insert]
			    while {[$doc.contents compare $start < $end]} {
				set address 0x[$doc.contents get $start "$start lineend"]
				regexp {([^:]*):.*} $address all address
				if {$address == $Debugger(PC)} {
				    # the current line. make sure we can see it
				    set current $start
			            append info "->\n"
				} else {
			            append info "  \n"
				}
			        set start [expr $start + 1.0]
			    }
			    $doc.contents infoinsert $Debugger(tagend) $info disassembly
			    $doc.contents see $current
			    if {$current != $first} {
				# give disassembly address
	                        EDITposition $doc $current
			    } else {
	                        EDITposition $doc [$doc.contents index insert]
			    }
		        }
		    }
		}
	    } else {
	        EDITposition $doc [$doc.contents index insert]
	    }
	}
    }
}

# the image used for a set breakpoint
image create bitmap code::breakimage -foreground red -data  {
#define circle_width 8
#define circle_height 8
static unsigned char circle_bits[] = {
   0x00, 0x3c, 0x7e, 0x7e, 0x7e, 0x7e, 0x3c, 0x00
   }
}

# the image used for a disabled breakpoint
image create bitmap code::nobreakimage -foreground white -data  {
#define circle_width 8
#define circle_height 8
static unsigned char circle_bits[] = {
   0x00, 0x3c, 0x7e, 0x7e, 0x7e, 0x7e, 0x3c, 0x00
   }
}

#
# EDITshowbreak - show a breakpoint in a info widget
#
proc code::EDITshowbreak {where {docs {}}} {
    variable Debugger
    variable DebuggerAtBreakpoints

    if {![regexp {'(.*)'@([0-9]+)} $where all file line]} {
	return
    }
    if {$docs == {}} {
	set docs [allDocs]
    }
    foreach doc $docs {
	if {[catch {$doc.contents cget -contents} thisfile]} {
	    # not an editor
	    continue
	}
	if {$file != $thisfile || ![$doc.contents cget -infowidth]} {
	    # not this file
	    continue
	}
        set end [$doc.contents infoindex "end - 1c"]
        if {$Debugger(opened) != {} && $end == 1.0 && ![EDITsetupinfo $doc]} {
	    # file was modified
	    continue
	}
	# translate for possible disassembly lines
	set line [EDITfakeline $doc.contents $line]
	$doc.contents infodelete $line.0 $line.end
        if {[info exists DebuggerAtBreakpoints($where)]} {
            if {$DebuggerAtBreakpoints($where) == {}} {
		$doc.contents infoimage create $line.0 -image code::nobreakimage -align center
		$doc.contents infotag add inactivebreakpoint $line.0 $line.end
	    } else {
		$doc.contents infoimage create $line.0 -image code::breakimage -align center
		$doc.contents infotag add breakpoint $line.0 $line.end
            }
        } else {
	     $doc.contents infoinsert $line.0 "--"
        }
	return
    }
}

#
# EDITnexterror - go to the next build error
#
proc code::EDITnexterror {} {
    variable EDITerrors
    variable EDITerrororder
    variable EDITcurrenterror
    variable EDITerrorindex

    set index $EDITerrorindex($EDITcurrenterror)
    incr index
    set fileindex [lsearch $EDITerrororder $EDITcurrenterror]
    if {$index >= [llength $EDITerrors($EDITcurrenterror)]} {
	# go to next file
	set index 0
	incr fileindex
	if {$fileindex <= 0 || $fileindex >= [llength $EDITerrororder]} {
	    set fileindex 0
	}
    }
    set file [lindex $EDITerrororder $fileindex]
    set errors $EDITerrors($file)
    set current [lindex $errors $index]
    set line [lindex $current 0]
    regexp {([0-9]+)\.([0-9]+)} $line all line column
    if {![info exists column]} {
        set column 0
    }
    set pos $line.$column
    EDITexisting $file $pos
    set EDITcurrenterror $file
    set EDITerrorindex($EDITcurrenterror) $index
    set warn [lindex $current 1]
    if {$warn != {}} {
        set warn "$warn - "
    } 
    status . $warn[lindex $current 2]
}

#
# EDITlasterror - go to the last build error
#
proc code::EDITlasterror {} {
    variable EDITerrors
    variable EDITerrororder
    variable EDITcurrenterror
    variable EDITerrorindex

    set index $EDITerrorindex($EDITcurrenterror)
    incr index -1
    set fileindex [lsearch $EDITerrororder $EDITcurrenterror]
    if {$index < 0} {
	# go to last file
	incr fileindex -1
	if {$fileindex < 0} {
	    set fileindex [llength $EDITerrororder]
	    incr fileindex -1
	}
        set file [lindex $EDITerrororder $fileindex]
	set index [llength $EDITerrors($file)]
	incr index -1
    }
    set file [lindex $EDITerrororder $fileindex]
    set errors $EDITerrors($file)
    set current [lindex $errors $index]
    set line [lindex $current 0]
    regexp {([0-9]+)\.([0-9]+)} $line all line column
    if {![info exists column]} {
        set column 0
    }
    set pos $line.$column
    EDITexisting $file $pos
    set EDITcurrenterror $file
    set EDITerrorindex($EDITcurrenterror) $index
    set warn [lindex $current 1]
    if {$warn != {}} {
        set warn "$warn - "
    } 
    status . $warn[lindex $current 2]
}

#
# EDITmarkdocerrors - mark errors in an edit document
#
proc code::EDITmarkdocerrors {doc file} {
    variable EDITerrors
    variable EDITcurrenterror
    variable EDITerrorindex
 
    if {![info exists EDITerrors($file)]} {
        # no errors here
        return
    }
    set index 0

    set errors $EDITerrors($file)
    set t $doc.contents
    set index 0
    foreach error $errors {
        foreach "line warn text" $error {
            if {$warn != {}} {
  	        set tag warning
	        set warn "$warn - "
	    } else {
	        set tag error
	    }
  	    regexp {([0-9]+)\.([0-9]+)} $line all line column
	    if {![info exists column]} {
	        set column 0
	    }
	    set pos $line.$column
            if {$EDITerrorindex($file) == -1} {
		# set the initial position
		set EDITerrorindex($file) 0
		set EDITcurrenterror $file
		$t mark set insert $pos
		$t see insert
            }
            $t tag add $tag "$pos linestart" "$pos lineend + 1c"
            $t tag add error$index "$pos linestart" "$pos lineend + 1c"
	    # make sure sel stays on top
	    $t tag raise sel
            $t tag bind error$index <Enter> "code::status . \{$warn$text\}"
            $t tag bind error$index <Leave> "code::status . {}"
            incr index
        }
    }
}

#
# EDITmarkfileerrors - mark errors for a file
#
proc code::EDITmarkfileerrors {file} {
    foreach doc [allDocs] {
	if {   [$doc cget -type] == "Editor"
	    && [$doc.contents cget -contents] == $file} {
	    # mark errors in this editor
	    EDITmarkdocerrors $doc $file
	}
    }
}

#
# EDITmarkerrors - mark errors in any existing editors
#
proc code::EDITmarkerrors {} {
   variable EDITerrororder
   
   foreach file $EDITerrororder {
       EDITmarkfileerrors $file
   }
}

#
# EDITerrors - handle build errors for specific files
#
proc code::EDITerrors {errors} {
   variable EDITerrororder
   variable EDITerrors
   variable EDITerrorindex

   if {[Preference Editor showerrors]} {
	# show an error list
	set doc [EDITview "Build Errors" {} 1 {}]
	EDITtitle $doc
	set t $doc.contents
	$t configure -state normal -positionproc {}
	set EDITerrororder {}
        foreach one $errors {
            foreach "name errors" $one {
                if {[file pathtype $name] == "relative"} {
                    set fullname [file join [pwd] $name]
                } else {
	            set fullname $name
	        }
		lappend EDITerrororder $fullname
		set EDITerrors($fullname) $errors
                set EDITerrorindex($fullname) 0
	        set index 0
	        set name [file nativename $name]
	        foreach error $errors {
	            foreach "line warn text" $error {
	                if {$warn != {}} {
			    set tag warning
		            set warn "$warn - "
	                } else {
			    set tag error
		        }
			regexp {([0-9]+)\.([0-9]+)} $line all line column
			if {![info exists column]} {
			    set column 0
			}
			set pos $line.$column
	                $t insert insert "$name, line $line: $warn$text\n" "error$index finger"
		        $t tag bind error$index <1> "$t tag remove error 1.0 end; $t tag remove warning 1.0 end; $t tag add $tag \"error$index.first linestart\" \"error$index.first lineend + 1c\"; set code::EDITcurrenterror $fullname; code::EDITexisting $fullname $pos; $t tag raise sel; break"
		        incr index
	            }
	        }
            }
	}
	set EDITcurrenterror [lindex $EDITerrororder 0]
	set oldcursor [$t cget -cursor]
        $t tag bind finger <Enter> "%W config -cursor hand2"
        $t tag bind finger <Leave> "%W config -cursor $oldcursor"
	$t configure -state disabled
    } else {
	# build the error list
	set EDITerrororder {}
	set firstfile {}
        foreach one $errors {
            foreach "name errors" $one {
                if {[file pathtype $name] == "relative"} {
                    set fullname [file join [pwd] $name]
                } else {
	            set fullname $name
	        }
		lappend EDITerrororder $fullname
		set EDITerrors($fullname) $errors
                set EDITerrorindex($fullname) 0
		if {$firstfile == {}} {
                    set EDITerrorindex($fullname) -1
		    EDITexisting $fullname
		    set firstfile $name
		}
            }
	}
	EDITmarkerrors
    }
}

#
# EDITexisting - edit an existing file, if it exists, or open a new one
#
proc code::EDITexisting {file {pos 1.0} {tag {}} {msg {}}} {
    variable EDITlast

    if {[file pathtype $file] == "relative"} {
        set file [file join [pwd] $file]
    }
    set doc [CODEfind Editor $file]
    if {$doc != {}} {
	DOCstate $doc
	$doc raise
	$doc.contents mark set insert $pos
	$doc.contents see $pos
    } else {
        # open a new editor
	if {[info exists EDITlast] && [winfo exists $EDITlast]} {
	    # set size and position from the last window
	    set doc [EDIT $file $pos \
		-x [$EDITlast cget -x] -y [$EDITlast cget -y] \
		-height [$EDITlast cget -height] -width [$EDITlast cget -width]]
	} else {
            set doc [EDIT $file $pos]
	}
    }
    set EDITlast $doc
    # set a tag, if wanted
    if {$tag != {}} {
        $doc.contents tag add $tag "$pos linestart" "$pos lineend + 1c"
	# make sure sel stays on top
	$doc.contents tag raise sel
    }
    status . $msg
    return $doc
}

#
# EDITsaveall - save all modified editors
#
proc code::EDITsaveall {} {
    foreach doc [allDocs] {
	if {[string match Edit* [$doc cget -type]]} {
	    $doc.contents save
	}
    }
}

#
# EDITsavefile - save a specific file, if modified
#
proc code::EDITsavefile {file} {
    if {[file pathtype $file] == "relative"} {
        set file [file join [pwd] $file]
    }
    foreach doc [allDocs] {
	if {   [string match Editor [$doc cget -type]]
	    && [$doc.contents cget -contents] == $file} {
	    $doc.contents save
	    break
	}
    }
}

#
# EDITprebuild - initialize for a build
#
proc code::EDITprebuild {} {
    variable EDITerrors
    variable EDITerrororder
    variable EDITcurrenterror
    variable EDITerrorindex

    foreach doc [allDocs] {
	if {[string match "Edit Value" [$doc cget -type]]} {
            # save all variables
	    $doc.contents save
	} 
	if {   [info exists EDITerrororder]
	    && [string match Editor [$doc cget -type]]
	    && [lsearch $EDITerrororder [$doc.contents cget -contents]] != -1} {
	    # clear error tags for open editors
	    $doc.contents tag remove error 1.0 end
	    $doc.contents tag remove warning 1.0 end
	    set file [$doc.contents cget -contents]
	    for {set index 0} {$index < [llength $EDITerrors($file)]} {incr index} {
		$doc.contents tag remove error$index 1.0 end
	    }
	}
    }
    catch {unset EDITerrors}
    catch {unset EDITerrororder}
    catch {unset EDITcurrenterror}
    catch {unset EDITerrorindex}
}

#
# EDITtitle - set the title of a document window
#
proc code::EDITtitle {doc} {
    set file [$doc.contents cget -contents]
    set image [CODEicon $file]
    set type [$doc cget -type]
    if {[string match Edit* $type]} {
        if {[$doc.contents ismodified]} {
	    set modified { [Modified]}
        } else {
	    set modified {}
        }
        $doc configure -image $image -icontext [file tail $file] \
	    -title  "$type: [file nativename $file]$modified"
    } else {
        $doc configure -image $image -icontext [file tail $file] \
	    -title  "$file"
    }
    SearchUpdate $doc $doc.contents $file $file [string match Edit* $type] 
}

#
# EDITmacroEnter - A macro has been entered
#
proc code::EDITmacroEnter { w x y {cpp {}}} {
    if {$cpp != {}} {
	set pos [$w index @$x,$y]
	set macro [eval $w get [$w tag prevrange macro "$pos + 1c" 1.0]]
        regexp {([0-9]+)\.([0-9]+)} $pos all line column
        set line [EDITrealline $w $line]
        set realindex $line.$column
        status . "$macro = [cppexpansion $cpp $realindex]"
    }
}

#
# EDITmacroLeave - A macro has been left
#
proc code::EDITmacroLeave {} {
    status . ""
}

#
# EDITexpandLine - expand all the macros in a source line
#
proc code::EDITexpandLine {srcfile cpp t start expansions} {
    set expansion {}
    set start [$t index "$start linestart"]
    regexp {([0-9]+)\.([0-9]+)} $start all line column
    set line [EDITfakeline $t $line]
    set fakestart $line.$column
    while {$expansions != {}} {
	set range [lindex $expansions 0]
	set first [lindex $range 0]
	set last [lindex $range 1]
        regexp {([0-9]+)\.([0-9]+)} $first all line column
        set line [EDITfakeline $t $line]
        set fakefirst $line.$column
        regexp {([0-9]+)\.([0-9]+)} $last all line column
        set line [EDITfakeline $t $line]
        set fakelast $line.$column

	if {[$t compare $fakestart < $fakefirst]} {
	    # need preceeding characters
	    append expansion [$t get $fakestart $fakefirst]
	}
	# add this expansion
	append expansion [cppexpansion $cpp $first]
	# check for a need new line
	if {[$t compare "$fakefirst linestart" < "$fakelast linestart"]} {
	    # this macro spans lines
	    set expansions [cppexpansion -all $cpp $last]
	} 

	# remove the current expansion
	set expansions [lrange $expansions 1 end]
	# set next start
	set start $last
        regexp {([0-9]+)\.([0-9]+)} $start all line column
        set line [EDITfakeline $t $line]
        set fakestart $line.$column
    }
    # add end of line
    append expansion [$t get $fakestart "$fakestart lineend"]
    EDITexpandMacro $srcfile {} $expansion
}

#
# EDITexpandMacro - display this macro
#
proc code::EDITexpandMacro {srcfile macro expansion} {
    set t [EDITmacro].contents
    $t configure -state normal
    set macros [$t tag ranges macro]
    $t mark set insert end
    set begin [$t index insert]
    $t insert insert $expansion\n

    # color the syntax elements
    set cpp [colorText $t $srcfile]
    cppclose $cpp
    foreach "start end" $macros {
	$t tag add macro $start $end
    }
    if {$macro != {}} {
        $t insert $begin "$macro = " macro
    }
    $t see $begin
    $t configure -state disabled 
}

#
# EDITmacrostartup - return a command to recreate a macro expansion window
#
proc code::EDITmacrostartup {doc} {
    return "EDITmacro"
}

#
# EDITmacro - create the macro expansion windo
#
proc code::EDITmacro {args} {
    set title "Macro Expansions"
    set doc [CODEfind "Show" $title]
    if {$doc != {}} {
	DOCstate $doc
	$doc raise
	return $doc
    }
    set doc [eval TEXT $args]
    set t $doc.contents
    $doc configure -type Show -startupproc "code::EDITmacrostartup"
    set image [CODEicon cfile.icon]
    $doc configure -image $image -icontext Expansions -title $title
    SearchUpdate $doc $doc.contents $title $title 0
    $t configure -contents $title
    $t configure -state disabled 
    return $doc
}

#
# EDITindent - auto indent in an editor window
#
proc code::EDITindent {wid} {
    # get the last line
    set line [$wid index "insert linestart"]
    set last [$wid get $line "$line lineend"]
    set indent {}
    # find whitespace
    regexp {^([ 	]*).*} $last all indent
    $wid insert insert "\n$indent"
}

#
# EDITrealline - return the real source line of a text position
#
proc code::EDITrealline {t line} {
    set newline $line
    foreach "start end" [$t tag ranges disassembly] {
        regexp {([^.]*)\.(.*)} $start all sline
        regexp {([^.]*)\.(.*)} $end all eline
	if {$sline < $line} {
	    # disassembly preceeds line
	    set newline [expr $newline - ($eline - $sline)]
	}
	if {$line >= $sline && $line < $eline} {
	    # a disassembly line
	    return {}
	}
    }
    return $newline
}

#
# EDITfakeline - return the text position of a source line
#
proc code::EDITfakeline {t line} {
    foreach "start end" [$t tag ranges disassembly] {
        regexp {([^.]*)\.(.*)} $start all sline
        regexp {([^.]*)\.(.*)} $end all eline
	if {$sline < $line} {
	    # disassembly preceeds line
	    set line [expr $line + ($eline - $sline)]
	}
    }
    return $line
}

#
# EDITposition - called when an editor insert position is changed
#
proc code::EDITposition {doc where} {
    variable CODEdocument

    if {$CODEdocument == {} || $doc != $CODEdocument} {
	return
    }
    regexp {([^.]*)\.(.*)} $where all line column
    # get the line, count tabs
    if {[catch {$doc.contents get $line.0 $line.$column} text]} {
	# not an editor
	return
    }
    set origline $line
    set line [EDITrealline $doc.contents $line]
    if {   $line == {}
	&& (   [lsearch [$doc.contents tag names $where] "disassembly"] != -1
	    || [$doc cget -type] == "Memory")} {
	# in display area
	set pos [DEBUGaddress $doc $where]
	if {$pos != {}} {
	    set pos "At: $pos"
	}
        .status itemconf Position -text $pos
	return
    }
    set count 0
    set index 0
    set tabs [Preference Editor tabs]
    while {$index < $column} {
	if {[string index $text $index] == "\t"} {
	    # expand tabs
	    incr count [expr $tabs - ($count % $tabs)]
	} else {
	    incr count
	}
	incr index
    }
    .status itemconf Position -text "Line: $line Col: $count"
}

#
# EDITchanged - called when an editor's contents have been changed
#
proc code::EDITchanged {doc change args} {
    variable EDITppcontext
    variable CODEdocument

    set result 1; # should insert or delete operation continue?
    set file [$doc.contents cget -contents]
    if {[info exists EDITppcontext($file)]} {
	set cpp $EDITppcontext($file)
    } else {
	set cpp {}
    }

    if {   (   [string equal $change "insert"] \
            || [string equal $change "delete"]) \
	&& [Preference Editor showcolors]} {
	# color the syntax elements
	set start [lindex $args 0]
	set end [lindex $args 1]
        set EDITppcontext($file) [colorText $doc.contents \
	    [$doc.contents cget -contents] \
	    $cpp $start $end \
	    [string equal $change "delete"]]
	return 1
    } elseif {[string equal $change "renamed"]} {
	# file was renamed: recolor
	if {$cpp != {}} {
	    cppclose $cpp
	    unset EDITppcontext($file)
	}
        # check to see if this file is a source file
        set extension [string tolower [file extension $file]]
        set type [Preference FileTypes file$extension]
        set type [lindex $type 3]
        if {$type == "c" || $type == "asm"} {
	    # add the info widget
	    $doc.contents configure -infowidth 2
	    bind [$doc.contents infotext] <1> "code::EDITinfoclick $doc %x %y; $doc raise"
	    bind [$doc.contents infotext] <Motion> "code::EDITinfomove $doc %x %y"
	    bind [$doc.contents infotext] <Leave> "code::status . {}"
	    $doc.contents infotag configure breakpoint -justify center
	    $doc.contents infotag configure inactivebreakpoint -justify center
        } else {
	    $doc.contents configure -infowidth 0
	}
        set EDITppcontext($file) [colorText $doc.contents \
	    [$doc.contents cget -contents] $cpp 1.0 end 0]
        if {$doc == $CODEdocument} {
	    # make sure a change is reflected in CODEcurrentfiles
	    $doc raise
        }
    } elseif {[string equal $change "closing"]} {
	# the last view of a file is closing
	if {$cpp != {}} {
	    # close the preprocessing context
	    cppclose $cpp
	    unset EDITppcontext($file)
	}
    } elseif {   [string equal $change "modified"]
	      && [lindex $args 0]
	      && [$doc.contents cget -infowidth]} {
	# text will be modified, info and disassmbly no longer valid
        variable Debugger

        # remove any current source tag
        $doc.contents tag remove $Debugger(tag) 1.0 end
	$doc.contents infodelete 1.0 end

        # delete any disassembly text
        foreach "from to" [$doc.contents tag ranges disassembly] {
	    set result 0; # have some disassembly, defer change
	    $doc.contents quietdelete $from $to
        }
	if {[$doc.contents cget -infowidth]} {
            foreach "from to" [$doc.contents infotag ranges disassembly] {
	        $doc.contents infodelete $from $to
            }
	}
    }
    EDITtitle $doc
    wm title . "[$doc cget -title]"
    ToolGenerate toolbar
    return $result
}

#
# EDITcolor - color the text of a document
#
proc code::EDITcolor {doc} {
    variable EDITppcontext

    set file [$doc.contents cget -contents]
    if {[info exists EDITppcontext($file)]} {
	cppclose $EDITppcontext($file)
    }

    set EDITppcontext($file) [colorText $doc [$doc cget -contents]] 
}

#
# EDITsearch - search a text widget
#
proc code::EDITsearch {kind} {
    variable CODEdocument

    if {$CODEdocument == {}} {
	# no document active
	return
    }
    # get the document type
    set type [$CODEdocument cget -type]
    if {   ![string match Edit* $type]
	&& [lsearch {Show Command Disassembly Memory Manual} $type] == -1} {
	# must be an editor, display, or command window
	return
    }
    if {$kind == "Replace" && [lsearch {Show Command Disassembly Memory Manual} $type] != -1} {
	# can't replace in a display window
	return
    }
    set t $CODEdocument.contents
    Search $CODEdocument $t $kind [$t cget -contents] 1
}

#
# EDITmark - put a bookmark in a text widget
#
proc code::EDITmark {} {
    variable CODEdocument

    if {$CODEdocument == {}} {
	# no document active
	return
    }
    # get the document type
    set type [$CODEdocument cget -type]
    if {   ![string match Edit* $type]
	&& [lsearch {Show Command Disassembly Memory Manual} $type] == -1} {
	# must be an editor, display, or command window
	return
    }
    set t $CODEdocument.contents
    set file [$t cget -contents]
    SearchMark $CODEdocument $t $file
}

#
# EDITshow - show the manual page for a word, or the expansion or definition of a macro
#
proc code::EDITshow {t {type {}} {index {}}} {
    variable EDITppcontext
    variable CODEdocument

    set doc $CODEdocument
    if {   ![catch {$doc.contents cget -contents} file] 
	&& [info exists EDITppcontext($file)]} {
	set cpp $EDITppcontext($file)
    } else {
	set cpp {}
    }

    if {$index == {}} {
	set index [$t index insert]
    } else {
	set index [$t index $index]
    }
    regexp {([0-9]+)\.([0-9]+)} $index all line column
    set line [EDITrealline $t $line]
    set realindex $line.$column
    set tags [$t tag names $index]
    set helpfile {}
    set foundid 0
    foreach tag $tags {
	switch $tag {
	    identifier {
		# get the word we are pointing to
		set word [eval $t get [$t tag prevrange identifier "$index + 1c"]]
		set helpfile [HelpLookupC $word]
	        if {![catch {DebuggerDbg where $word} def]} {
		    if {[lindex $def 0] != {} && [lindex $def 1] != 0} {
			set foundid 1
		    }
	        }
	    }
	    opcode {
		# get the word we are pointing to
		set word [eval $t get [$t tag prevrange opcode "$index + 1c"]]
		set helpfile [HelpLookupAsm $word]
	    }
	    macro {
		# show macro expansion
		if {$cpp != {}} {
		    set expansion [cppexpansion $cpp $realindex]
		    set word \
			    [eval $t get [$t tag prevrange macro "$index + 1c"]]
		    set srcfile [$doc.contents cget -contents]
		    switch $type {
			line {
                            set expansions [cppexpansion -all $cpp $realindex]
	                    EDITexpandLine $srcfile $cpp $t $realindex $expansions
			}
			mgoto {
		            set macrofile [cppexpansion -define $cpp $realindex]
	                    EDIT [lindex $macrofile 0] [lindex $macrofile 1].0
			}
			macro {
		            EDITexpandMacro $srcfile $word $expansion
			}
		    }
		}
	    }
	}
    }
    if {$type != "igoto" && $helpfile != {}} {
	MANUAL $helpfile
    } elseif {$type == "igoto" && $foundid} {
	set file [lindex $def 0]
        if {![catch {DebuggerDbg path -find $file} file]} {
	    EDITexisting $file [lindex $def 1].0
	}
    }
}

#
# EDITviewstartup - return the command to open a view
#
proc code::EDITviewstartup {name value readonly openproc doc} {
    set name [BuildFileName $name]
    return "EDITview \{$name\} \{$value\} $readonly \{$openproc\}"
}

#
# EDITview - open an editor window with arbitrary text
#
proc code::EDITview {name value readonly openproc args} {
    if {$readonly} {
	set type Show
    } else {
	set type "Edit View"
    }

    set box [CODEfind $type $name]
    if {$box == {}} {
	# a new document
        set box [eval EDIT \{\} 1.0 $args]
        set t $box.contents
	$box configure -type $type
        $t configure -contents $name
        EDITtitle $box
    } else {
	# have a document, delete old contents
	DOCstate $box
	$box raise

        set t $box.contents
	$t noundo 1
        $t configure -state normal 
	$t delete 1.0 end
    }
    # turn off undo for the initial load
    $t noundo 1
    # insert the initial value
    $t insert insert $value
    if {$openproc != {}} {
	$t configure -contents $name
	$t insert insert [eval $openproc]
        $box configure -startupproc "code::EDITviewstartup \{$name\} \{$value\} $readonly \{$openproc\}"
    }
    if {$readonly} {
        $t configure -state disabled 
    }
    # treat contents as unmodified
    $t modified 0
    $t noundo 0
    return $box
}

proc code::EDITgetvariable {category element text} {
    global $category

    if {$element == {}} {
	# editing a tcl variable
        $text insert 1.0 [set $category]
    } else {
	# editing a CODE preference
        $text insert 1.0 [Preference $category $element]
    }
}

proc code::EDITsetvariable {category element title text} {
    global $category

    if {$element == {}} {
	# editing a tcl variable
	set $category [$text get 1.0 end-1c]
    } else {
	# editing a CODE preference
        PreferenceSetIfChanged $category $element [$text get 1.0 end-1c]
    }
    return $title
}

#
# EDITvariablestartup - return the command to open a variable
#
proc code::EDITvariablestartup {category element title doc} {
    return "EDITvariable $category \{$element\} \{$title\}"
}

#
# EDITvariable - open an editor window with a CODE variable or preference variable
#
proc code::EDITvariable {category element title args} {
    set box [CODEfind "Edit Value" $title]
    if {$box != {}} {
	DOCstate $box
	$box raise
	return $box
    }
    # a new document
    set box [eval EDIT \{\} 1.0 $args]
    set t $box.contents
    $box configure -type "Edit Value"
    if {$element != {}} {
	# only preferences can have persistance
        $box configure -startupproc "code::EDITvariablestartup $category $element \{$title\}"
    }
    $t configure -new 0 -contents $title \
    	-openproc "code::EDITgetvariable $category \{$element\}" \
	-saveproc "code::EDITsetvariable $category \{$element\}"
    $t open $title
    EDITtitle $box
    return $box
}

