#
#	Manual - manage an on-line manual window
#

package require http 2.0

#
# MANUALevents - define manual window events
#
# All CODE operations go through events
#
# For each event (lindex value):
#	0 - Menu name (used in menus)
#	1 - Help string (shown in tool tips and status bar)
#	2 - Conditions to enable (used to set state)
#		debug - the debugger is active
#		running - the debugged program is running
#		stopped - the debugged program is stoped
#		nocomplex - the complex breakpoint editor is not opened
#		hascontext - a debugger context is available
#		notdisassembly - not in a disassembly area
#		isdisplay - in a debugger display window
#		iswatch - in a debugger watch window
#		isterminal - in a terminal window
#		foundid - an identifier has been found in the debug context
#		hasfile - a file name is associated with this event
#		onefile - a single file name has been given
#		issource - the file(s) are program source
#		isobject - the file(s) are object files
#		canmakeobject - the file(s) are or can be object files
#		isddf - the file(s) are device definitions files
#		hasredo - redo is available
#		hasundo - undo is available
#		istext - document is a text widget
#		hasid - line has an identifier
#		editable - document is editable text
#		hasselection - the current editor has a selection
#		hasclipboard - the current clipboard has contents
#		haserrors - document has errors
#		ismodified - document is modified
#		hasmacro - line has a macro
#		hasmacrofile - definition of macro is known
#		hasexpansions - line contains macro expansions
#		isinproject - file(s) in project
#		hasdepends - the file is in a project and has dependencies
#		notinproject - file(s) not in project
#		canaddtoproject - file is appropriate for project inclusion
#		options - file(s) can have project options set
#		isdirectory - current window is a directory window
#	3 - noshow flag (popup menu only, if not {}, don't show if disabled)
#	4 - {} for normal button {check <variable>} for checkbutton, 
#	    cascade <list> to make a cascade of events,
#	    or cascade command (command to make a cascaded menu)
#	5 - Tool button name (name used in a toolbar)
#	6 - Code to perform operation
#	7 - non-{} if this should be broadcast. "all": broadcast to all windows
#
set code::MANUALevents {
About
    { "About"
      "About CODE"
      {}
      {}
      {}
      "About"
      { code::signon 1; break }
    }
OpenFile
    { "Open..."
      "Open a file"
      {}
      {}
      {}
      "Open"
      { code::ManualOpen {}; break }
    }
Preferences
    { "Preferences..."
      "Change CODE preferences"
      {}
      {}
      {}
      "Preferences"
      { code::PreferenceEdit .; break }
    }
SavePreferences
    { "Save Preferences"
      "Save your current preferences for next time"
      {}
      {}
      {}
      "Save Preferences"
      { code::PreferenceSaveUser; break }
    }
Exit
    { "Exit"
      "Exit the manual"
      {}
      {}
      {}
      "Exit"
      { code::MANUALexit; break }
    }
Copy
    { "Copy"
      "Copy to buffer"
      { hasselection }
      {}
      {}
      "Copy"
    }
Find
    { "Find..."
      "Search for a string or pattern"
       istext
       {}
       {}
       "Find"
       { code::EDITsearch Find; break }
    }
Goto
    { "Goto..."
      "Goto a specified place in the file"
       istext
       {}
       {}
       "Goto"
       { code::EDITsearch Goto; break }
    }
Mark
    { "Mark..."
      "Place a bookmark at the insertion point"
      istext
      {}
      {}
      "Mark"
      { code::EDITmark; break }
    }
Reload
    { "Reload"
      "Reload the current page"
      istext
      {}
      {}
      "Reload"
      { code::ManualOpenURL $code::CODEdocument $code::Manual($code::CODEdocument,entrytext) 0 1; break }
    }
DocumentSource
    { "Document Source"
      "View the source of the current page"
      istext
      {}
      {}
      "Document Source"
      { code::ManualSource $code::CODEdocument; break }
    }
Home
    { "Table of Contents"
      "Go to the table of contents"
      {}
      {}
      {}
      "Home"
      { code::ManualHistoryHome $code::CODEdocument; break }
    }
Back
    { "Back"
      "Go to the previous page in the history list"
      {istext code::Manual(historyback)}
      {}
      {}
      "Back"
      { code::ManualHistoryBack $code::CODEdocument; break }
    }
Forward
    { "Forward"
      "Go to the next page in the history list"
      {istext code::Manual(historyforward)}
      {}
      {}
      "Forward"
      { code::ManualHistoryForward $code::CODEdocument; break }
    }
Index
    { "Index"
      "Show/hide the index"
      {}
      noshow 
      { check code::Manual(indexshown) }
      "Index"
      { code::Preference Manual indexshown $code::Manual(indexshown); break }
    }
TOC
    { "TOC"
      "Show/hide the table of contents"
      {}
      noshow 
      { check code::Manual(tocshown) }
      "TOC"
      { code::Preference Manual tocshown $code::Manual(tocshown); break }
    }
AddIndex
    { "Add Index..."
      "Add an index entry"
      istext
      {}
      {}
      "Add Index"
      { code::ManualAddIndex $code::CODEdocument %x %y"; break }
    }
SaveIndex
    { "Save Index As..."
      "Save the index entries"
      {}
      {}
      {}
      "Save Index"
      { code::ManualSaveIndexFile .; break }
    }
PageIndex
    { "Page Index"
      "Show/hide the index for this page only"
      istext
      {} 
      { check code::Manual(pageindex) }
      "Page Index"
      { code::ManualIndexBuild 1; break }
    }
}

#
# OpenHelp - open a help window
#
proc code::OpenHelp {w args} {
    set file [eval FindManual $args]
    if {$file == ""} {
	return
    }
    ManualOpen {} $file
}

proc code::MANUAL {{file {}}} {
    global auto_path
    variable MANUALinterp
    variable INTROL

    if {![info exists MANUALinterp]} {
	# create a new interpreter for the manual
	set MANUALinterp [interp create]
        # set up the auto_path
        interp eval $MANUALinterp set auto_path [list $auto_path]
        # load required packages
        load {} Tk $MANUALinterp
        load {} Code $MANUALinterp
        load {} Html $MANUALinterp
        interp eval $MANUALinterp package require mkWidgets
        interp eval $MANUALinterp set code::INTROL \"$INTROL\"

	# set up shared commands
	foreach command [info commands ::code::Preference*] {
            $MANUALinterp alias $command $command
	}
        $MANUALinterp alias code::PreferenceWhenChanged code::ManualWhenChanged
	foreach command [info commands ::code::Build*] {
            $MANUALinterp alias $command $command
	}
	# create the manual window
        interp eval $MANUALinterp code::MANUALnew
    }
    interp eval $MANUALinterp code::CODEstartup
    interp eval $MANUALinterp wm deiconify .
    interp eval $MANUALinterp raise .
    if {[interp eval $MANUALinterp allDocs] > 0 && $file == {}} {
        # there is a startup configuration and no page was specified
        return
    }
    # show the requested manual page
    interp eval $MANUALinterp code::MANUALshow $file
}

#
# MANUALshow - show a manual page
#
proc code::MANUALshow {{file {}}} {
    if {[llength [allDocs]] == 0} {
        ManualOpen {} [ManualPrefix [FindManual contents.html]]
    }
    if {$file != {}} {
	set file [ManualPrefix $file]
        ManualOpen {} $file
    }
}

#
# ManualWhenChanged - PreferenceWhenChanged for the manual
#
proc code::ManualWhenChanged {category tag cmd} {
    PreferenceWhenChanged $category $tag "$code::MANUALinterp eval $cmd"
}

#
# MANUALexit - leave the manual window
#
proc code::MANUALexit {} {
    # save the current manual window configuration
    if {![catch {wm geometry .} geometry]} {
        Preference Geometry manual $geometry
    }
    # save manual state
    CODEclose
    CODEupdatetoolbars
    wm withdraw .
}

#
# MANUALnew - set up the initial manual window
#
proc code::MANUALnew {} {
    variable CODEdocument
    variable CODEevents
    variable CODEalt
    variable MANUALevents
    variable CODEbuttons
    variable Manual

    array set CODEevents $MANUALevents
    # build the button array
    foreach event [array names code::CODEevents] {
        set code::CODEbuttons([lindex $code::CODEevents($event) 5]) $event
    }

    frame .work -bg gray50
    pack .work -fill both -expand 1
    wm protocol . WM_DELETE_WINDOW "code::MANUALexit"
    statusbar .status -ticks 10
    .status add Position -width 200
    set list {
	Go "Go to a page in the history list" 0 "code::ManualMenuGo"
        Window "Manipulate and display CODE windows"  0 "code::MainMenuWindow"
        Help "Browse the CODE Manual" 0 "code::ManualMenuHelp"
    }

    set CODEalt manual
    CODEpreferences
    CODEcontrol manual $list

    set geometry [Preference Geometry manual]
    if {$geometry != {}} {
	# the window has had its geometry change saved
	setGeometry . $geometry
    }

    set toc [Preference Manual toc]
    while {1} {
        set help [ManualPrefix [FindManual $toc]]
        if {$help == {}} {
            set query [tk_messageBox -icon error \
            -parent . \
            -message "Online manual not available. Insert CODE CD-ROM?" \
	    -type retrycancel]
	if {$query == "cancel"} {
	    return
	}
        } else {
	    break
        }
    }   

    if {![info exists Manual(init)]} {
	# initial history
	set Manual(history) ""
	set Manual(historyIndex) 0
	set Manual(historyback) 0
	set Manual(historyforward) 0
        set Manual(source) ""

	# build the tree widgit
	set tf [frame .tf -bd 2 -relief sunken]
        pack $tf -fill y -side left -before .work
	grid rowconfigure $tf 0 -weight 1
	set t $tf.t
	set Manual(tree) $t
	set Manual(treeframe) $tf
	set sb [scrollbar $tf.sb -orient vertical -command "$t yview"]
	ManualTree:create $t -bg [Preference General colorbackground] \
	    -width 250 -height 0 -yscrollcommand "$sb set"
	grid $t -in $tf -row 0 -column 0 -sticky ns
	grid $sb -in $tf -row 0 -column 1 -sticky ns
	set Manual(tocshown) 1
        wm deiconify .
        tkwait visibility .
	ManualTree:show $t /
        update idletasks

	$t bind x <1> "code::ManualTreeSingle %W %x %y"
	$t bind x <Double-1> "code::ManualTreeDouble %W %x %y"

	# fill the Manual tree
        ManualParseURL $help protocol help query target
	set s [ManualGetURL $protocol $help err]
	if {$err} {
	    return
	}
	foreach line [split $s \n] {
	    if {$line == {}} {
	        continue
	    }
	    set tree [lindex $line 0]
	    set treefile [lindex $line 1]
	    set title [lindex $line 2]
            ManualTree:newitem $t $tree -text $title -file $treefile
	}

	set tf [frame .if -bd 2 -relief sunken]
	set Manual(pageindex) 0
        pack $tf -fill y -side right -before .work
	set Manual(indexshown) 1
	grid rowconfigure $tf 0 -weight 1
	set t $tf.t
	set Manual(index) $t
	set Manual(indexframe) $tf
        ManualIndexOpen

	set Manual(init) 1
	ManualPreferences
	MANUALraise {}
    } else {
        wm deiconify .
        tkwait visibility .
    }

    PreferenceWhenChanged Manual . "code::ManualPreferences"
    PreferenceWhenChanged General . "code::ManualPreferences"
    PreferenceWhenChanged General . "code::CODEpreferences"
    PreferenceWhenChanged Control . "code::CODEcontrol manual \{$list\}"
    MANUALraise {}
    update idletasks

    # this code seems to get around a bug in Tk (wm geometry for
    # a hidden window)
    global tcl_platform
    if {$tcl_platform(platform) == "windows" && $geometry != {}} {
        setGeometry . $geometry
    }

    # remember the main window's geometry
    bind . <Configure> "code::PreferenceMove . manual"

    bind .work <Configure> "arrangeDocs icons"
}  

#
# MANUALraise - raise a manual window
#
proc code::MANUALraise {doc} {
    variable CODEdocument
    variable CODEcurrentfiles

    set CODEdocument $doc
    set CODEcurrentfiles {}

    if {$doc != {}} {
        .status itemconf Position -text [$doc.contents cget -contents] -anchor e
    } else {
        .status itemconf Position -text "" -anchor e
    }
    ToolGenerate toolbar
}

#
# ManualMenuHelp - display the Help menu
#
proc code::ManualMenuHelp {menu} {
    variable menustatus
    variable CODEdocument

    set list {
       About Home
    }
    $menu delete 0 end
    ToolGenerate mainmenu $menu $list
    # XXX put the getting started entry up
    # set file [FindManual Applications About.html]
    # if {$file != ""} {
        # MenuCommand $w Help "Getting Started" \
	    # -help "Getting started with CODE" \
	    # -command "code::OpenHelp $file"
    # }

}

#
# ManualSource - show the source of a window
#
proc code::ManualSource {w} {
    variable Manual

    set text $w.contents
    $text delete 1.0 end
    $text insert insert $Manual($w,source)
    $text see 1.0
}

#
# ManualPreferences - Preferences have changed
#
proc code::ManualPreferences {} {
    variable Manual

    set color [Preference General colorbackground]
    $Manual(index) config -bg $color
    $Manual(tree) config -bg $color

    if {[Preference Manual indexshown]} {
        pack $Manual(indexframe) -fill y -side right -before .work
    } else {
        pack forget $Manual(indexframe)
    }
    if {[Preference Manual tocshown]} {
        pack $Manual(treeframe) -fill y -side left -before .work
    } else {
        pack forget $Manual(treeframe)
    }
}

#
# ManualTreeSingle - a single click on a directory tree
#
proc code::ManualTreeSingle {W x y} {
    variable ManualTree
    variable CODEdocument

    set w $CODEdocument
    set lbl [ManualTree:labelat $W $x $y]
    ManualTree:setselection $W $lbl
    if {![info exists ManualTree($W:$lbl:file)] || \
	  $ManualTree($W:$lbl:file) == {}} {
        ManualTree:open $W $lbl
	return
    }
    set target {}
    regexp {([^#]*)(#.*)?} $ManualTree($W:$lbl:file) all file target
    if {$w == {}} {
	set w [HTML [ManualPrefix [FindManual $file]$target]]
    }
    ManualOpenURL $w [ManualPrefix [FindManual $file]$target]
}

#
# ManualTreeDouble - a double click in the directory tree
#
proc code::ManualTreeDouble {W x y} {
    set f [ManualTree:labelat $W $x $y]
    ManualTree:open $W $f
}

#
# ManualTreeMouse3 - a right click on an index tree
#
proc code::ManualTreeMouse3 {W x y} {
    variable ManualTree
    variable CODEdocument

    if {$CODEdocument == {}} {
	return
    }
    set w $CODEdocument

    set lbl [ManualTree:labelat $W $x $y]
    ManualTree:setselection $W $lbl
    if {[info exists ManualTree($W:$lbl:file)]} {
	set file $ManualTree($W:$lbl:file)
	# remove the target
        regexp {([^#]*)(#.*)?} $file all file target
    } else {
	set file {}
    }
    if {![regexp {/[^/]/(.*)} $lbl all lbl]} {
	# a letter was selected
	return
    }
    ManualIndexEdit $w $lbl 0 $file
}

#
# ManualAddIndex - add an entry to the index
#
proc code::ManualAddIndex {w t x y} {

    set index [$t index @$x,$y]
    if {   [selection own -displayof $t] == $t && 
	![catch {selection get -displayof $t} selval]} {
	set sel [$t tag ranges sel]
	set index [lindex $sel 0]
    } else {
        set selval {}
    }

    # set the insertion point, in case there was an old selection
    $t mark set insert $index
    $t see insert

    ManualIndexEdit $w $selval 1
}

#
# ManualIndexEdit - edit an index entry
#
proc code::ManualIndexEdit {w entry {new 0} {originalfile {}}} {
    variable Manual
    variable ManualIndex
    variable manual

    set box .index
    # get the base file name
    set file $Manual($w,file)
    if {[string match http:* $manual]} {
        set count [regsub ^$manual/ $file {} file]
    } else {
        set count [regsub ^file:$manual/ $file {} file]
    }
    if {$count == 0} {
	# not a manual page
        tk_messageBox \
    	    -parent $w \
	    -icon error \
	    -message "$file is not a manual page" \
	    -type ok
	return
    }

    # remove a possible target
    regexp {([^#]*)(#.*)?} $file all file target
    set thismark {}
    if {[info exists ManualIndex($entry)]} {
	# there is already an entry for this
	set info $ManualIndex($entry)
        set italics [lindex $info 0]
	set files [lrange $info 1 end]
        set marks {}
	foreach f $files {
	    if {[lindex $f 0] != $file} {
		continue
	    }
	    # get current marks for this file
	    set marks [concat $marks [lrange $f 2 end]]
	    set thismark [lindex $f 2]
	}
    } else {
	# no entry yet
	set italics 0
	set files {}
	set marks {}
    }

    set t $w.contents
    set tmarks [$t mark names]
    foreach mark $tmarks {
	# gather all IDX marks
	if {[string match N:IDX* $mark]} {
	    regexp N:(.*) $mark all mark
	    if {[lsearch $marks $mark] == -1} {
                lappend marks $mark
	    }
	}
    }

    if {$new || $thismark == {}} {
	# gerate a new mark name

	set thismark IDX[$t index insert]
	if {[lsearch $marks $thismark] != -1} {
	    set count 1
	    while {1} {
		set thismark IDX[$t index insert]:$count
	        if {[lsearch $marks $thismark] == -1} {
		    break
		}
	    }
	}
    }
	     
    set Manual(newindexentry) $entry
    toplevel $box 
    wm transient $box $w
    wm protocol $box WM_DELETE_WINDOW "set code::Manual(newindex) {}"
    set f [frame $box.buttons]
    set okBtn [button $f.ok -text Ok -width 6 \
        -command "set code::Manual(newindex) \$code::Manual(newindexentry)"]
    set cancelBtn [button $f.cancel -text Cancel -width 6 \
        -command "set code::Manual(newindex) {}"]
    grid $okBtn -in $f -row 0 -column 0 
    grid $cancelBtn -in $f -row 0 -column 1 
    set Manual(deleteindex) 0
    if {!$new} {
        set deleteBtn [button $f.delete -text Delete -width 6 \
            -command "set code::Manual(deleteindex) 1; \
		set code::Manual(newindex)  \$code::Manual(newindexentry)"]
        grid $deleteBtn -in $f -row 0 -column 3 
    }
    grid columnconfigure $f 2 -weight 1
    grid $f -in $box -row 5 -column 0 -sticky ew -padx 5 -pady 5
    set e [entry $box.entry -textvariable code::Manual(newindexentry) -width 70]
    bind $e <Key-Return> "set code::Manual(newindex) \$code::Manual(newindexentry)"
    grid $e -in $box -row 1 -column 0 -sticky ew -padx 5 -pady 5
    set l [label $box.l -text "The index entry name (with a slash separating subphrases)"]
    grid $l -in $box -row 0 -column 0 -sticky ew -padx 5 -pady 5


    set code::Manual(italics) $italics
    set c [checkbutton $box.it -text "Use italics" -variable code::Manual(italics)]
    grid $c -in $box -row 2 -column 0 -sticky w -padx 5 -pady 5
    set l [label $box.t -text "HTML Target: $thismark"]
    grid $l -in $box -row 3 -column 0 -sticky w -padx 5 -pady 5
    placewindow $box widget $w
    if {$new} {
	set type Add
    } else {
	set type Edit
    }
    wm title $box "$type an Index Entry"

    set oldFocus [focus]
    set oldGrab [grab current $box]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $box
    focus $e

    tkwait variable code::Manual(newindex)
    catch {focus $oldFocus}
    grab release $box
    destroy $box
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }

    # newindex contains the new index value
    if {$Manual(newindex) == {}} {
	return
    }
    set newentry $Manual(newindex)
    set delete $Manual(deleteindex)
    set changed 0
    set title [$w cget -title]
    if {!$new && $newentry != $entry} {
	if {$delete} {
	    # delete works like cancel if the entry has changed
            return
	}
	# the entry has changed from the original
        if {![info exists ManualIndex($newentry)]} {
	    set new 1
        }

    }

    if {$new} {
	# make a new index entry
	set ManualIndex($newentry) [list $Manual(italics) \
	    [list $file $title $thismark]]
        ManualSaveIndexFile $w $thismark $file
	set changed 1
    } else {
	# find the current entry
	set info $ManualIndex($newentry)
        set italics [lindex $info 0]
	set files [lrange $info 1 end]
	set index 0
	if {$delete} {
	    set file $originalfile
	}
	foreach f $files {
	    if {[lindex $f 0] != $file} {
		incr index
		continue
	    }
	    break
	}
        if {$delete} {
	    # delete an existing entry

	    set ManualIndex($newentry) [list $italics \
	        [lreplace $files $index $index]]
	    if {[llength [lindex $ManualIndex($newentry) 1]] == 0} {
		# completely empty
		unset ManualIndex($newentry)
	    }
        } else {
	    # replace an existing entry
	    set ManualIndex($newentry) [list $italics \
	        [lreplace $files $index $index [list $file $title $thismark]]]
            ManualSaveIndexFile $w $thismark $file
        }
	set changed 1
    }
    if {$changed} {
	ManualIndexBuild 1
    }
}

#
# ManualIndexOpen - open the manual index
#
proc code::ManualIndexOpen {} {
    variable Manual
    variable ManualIndex

    set index [Preference Manual index]
    if {[file exists $index]} {
	# use the index in the current directory
	set file [file join [pwd] $index]
	set indexfile file:$file
    } else {
	# find the global index
	set file [FindManual $index]
        set indexfile [ManualPrefix $file]
    }
    set Manual(indexoff) 1
    if {[file writable $file]} {
	# can write the index
	set Manual(indexfile) $file
    } else {
	catch {unset Manual(indexfile)}
    }
    if {$indexfile == {}} {
	# no index present
	return
    }
    ManualParseURL $indexfile protocol indexfile query target
    set s [ManualGetURL $protocol $indexfile err]
    if {$err} {
	return
    }
    if {[catch {array set ManualIndex $s}]} {
	# probably a not found url
	return
    }
    set Manual(indexoff) 0
    ManualIndexBuild 1
}

#
# ManualIndexBuild - (re)build the manual index
#
proc code::ManualIndexBuild {{force 0}} {
    variable Manual
    variable ManualIndex
    variable manual
    variable CODEdocument

    if {$Manual(indexoff)} {
	if {!$force} {
	    return
	}
        # index is now on
        set Manual(indexoff) 0
    }

    if {!$force && !$Manual(pageindex)} {
	# index already built
	return
    }
    if {$Manual(pageindex) && [info exists CODEdocument] && $CODEdocument != {}} {
        # get the base file name
        set pagefile [$CODEdocument.contents cget -contents]
        if {[string match http:* $manual]} {
            set count [regsub ^$manual/ $pagefile {} pagefile]
        } else {
            set count [regsub ^file:$manual/ $pagefile {} pagefile]
        }
    } else {
	set pagefile {}
    }
    set last {}
    set root {}
    set t $Manual(index)
    set tf $Manual(indexframe)
    if {![info exists Manual(indexsb)]} {
        set sb [scrollbar $tf.sb -orient vertical -command "$t yview"]
        set Manual(indexsb) $sb
	grid $sb -in $tf -row 0 -column 1 -sticky ns
    } else {
	set sb $Manual(indexsb)
    }
    ManualTree:delitem $t /
    ManualTree:create $t -bg [Preference General colorbackground] \
	-width 250 -height 0 -yscrollcommand "$sb set"
    grid $t -in $tf -row 0 -column 0 -sticky ns
    $t bind x <1> "code::ManualTreeSingle %W %x %y"
    $t bind x <3> "code::ManualTreeMouse3 %W %x %y"
    $t bind x <Double-1> "code::ManualTreeDouble %W %x %y"


    set needletter 0
    foreach entry [lsort -dictionary [array names ManualIndex]] {
	set this [string toupper [string index $entry 0]]
	if {$this != $last} {
	    set last $this
	    set needletter 1
	}
	set value $ManualIndex($entry)
	set italic [lindex $value 0]
	set files [lrange $value 1 end]
	set desc [lindex $files 0]
	set file [lindex $desc 0]
	set marks [lrange $desc 2 end]
	if {[llength $files] == 1 && [llength $marks] == 1} {
	    # a single file with a single mark
	    if {$pagefile != {} && $pagefile != $file} {
		continue
	    }
	    if {$needletter} {
	        set needletter 0
                ManualTree:newitem $t $root/$last -text $last
	    }
            ManualTree:newitem $t $root/$last/$entry -text [file tail $entry] \
		-italic $italic -file $file#$marks
	} else {
	    # one or multiple files, one or more marks
	    set hasheading 0
	    foreach desc $files {
	        set file [lindex $desc 0]
		set title [lindex $desc 1]
	        set marks [lrange $desc 2 end]
	        if {$pagefile != {} && $pagefile != $file} {
		    set hasheading 1
		    continue
	        }
		if {!$hasheading} {
	            set hasheading 1
	            if {$needletter} {
	                set needletter 0
                        ManualTree:newitem $t $root/$last -text $last
	            }
                    ManualTree:newitem $t $root/$last/$entry -text [file tail $entry] \
	                -italic $italic
		}
	        if {[llength $marks] == 1} {
		    # a file with a single mark
                    ManualTree:newitem $t $root/$last/$entry/$title \
			-text $title \
		        -italic $italic -file $file#$marks
		} else {
		    # a file with multiple marks
                    ManualTree:newitem $t $root/$last/$entry/$title \
			-text $title \
		        -italic $italic -file $file
		    set count 1
		    foreach mark $marks {
                        ManualTree:newitem $t $root/$last/$entry/$title/$count \
			    -text $count \
		            -italic $italic -file $file#$mark
			incr count
		    }
		}
	    }
	    if {!$hasheading} {
		# for file-less entries
	        set hasheading 1
	        if {$needletter} {
	            set needletter 0
                    ManualTree:newitem $t $root/$last -text $last
	        }
                ManualTree:newitem $t $root/$last/$entry -text [file tail $entry] \
	            -italic $italic
	    }
	}
    }
}

#
# ManualSaveIndexFile - save the index file
#
proc code::ManualSaveIndexFile {w {newmark {}} {file {}} {oldbind 0}} {
    variable ManualIndex
    variable Manual

    set t $w.contents
    if {$oldbind} {
	# called from ambiguous placement click
	# HTML source is in text widget

	$t insert insert "<A NAME=\"$newmark\">"
	$t insert "insert + 1 char" "</A>"
	# restore binding
	bind $t <Button-1> $Manual(oldbind)
	# try to write the HTML file
	set file [FindManual $file]
	set Manual($w,source) [$t get 1.0 end]
	if {[catch {open $file w} fd]} {
	    # error opening for write

            tk_messageBox \
    	        -parent $w \
	        -icon error \
	        -message "$fd" \
	        -type ok
            return
	}
        # we want to preserve newlines, even on DOS
        fconfigure $fd -translation binary
        puts -nonewline $fd $Manual($w,source)
        close $fd
    } elseif {$newmark != {}} {
	# a new mark has been added, add to HTML file
	# all this hackery is required to try to find the biggest
	# string that matches, trying to add a name anchor automatically
	set s $Manual($w,source)
	# try to find the mark point in s
	# find the longest string
	set start [$t index insert]
	set end [$t index  "$start lineend - 1 char"]
        set next [$t mark next "$start + 1 char"]
        if {$next != {}} {
	    set next [$t index $next]
	    if {[$t compare $next < $end]} {
	        # there is a mark before the end of the line
	        set end $next
	    }
	}
	# look for a change in tags
        set tags [$t tag names $start]
	set current $start
	while {[$t compare $current <= $end] && [$t tag names $current] == $tags} {
	    set current [$t index "$current + 1 char"]
	}
        if {[$t compare $current < $end]} {
	    # there is a new tag before the end
	    set end $current
	}
        # have the end, try to find the earliest non-tagged, marked text
	set current $start
	set startline [$t index "insert linestart"]
	while {[$t compare $startline <= $current] && [$t tag names $current] == $tags} {
	    set current [$t index "$current - 1 char"]
	}
        if {[$t compare $current >= $startline] && $current < $start} {
	    # there is a new tag before the beginning of the line
	    set start [$t index "$current + 1 char"]
	}
        set last [$t mark previous "insert - 1 char"]
        if {$last != {}} {
	    set last [$t index $last]
	    if {[$t compare $last > $start]} {
	        # there is a mark before the beginning of the line or previous tag
	        set start $last
	    }
	}

        set before [$t get $start insert]
	set current [$t get insert]
        set after [$t get "insert + 1 char" $end]
	set orig "$before$current$after"
	set replace "$before<A NAME=\"$newmark\">$current</A>$after"
	# change newlines into spaces
	# this hack is required to make a newline
	regsub -all {
} $s { } s
	# end hack
	set first [string first $orig $s]
	set match 0
	if {$first != -1} {
	    set last [string last $orig $s]
	    if {$first == $last} {
	        # get original source
	        set s $Manual($w,source)
	        set before [string range $s 0 [expr {$first - 1}]]
	        set after [string range $s [expr {$first + [string length $orig]}] end]
		set s $before$replace$after
		# whew! an exact match
		set match 1
		set $Manual($w,source) $s
	    }
	}
	if {$match} {
	    # try to write the HTML file
	    set file [FindManual $file]
	    if {[catch {open $file w} fd]} {
		# error opening for write

                tk_messageBox \
    	            -parent $w \
	            -icon error \
	            -message "$fd" \
	            -type ok
                return
	    }
	    # we want to preserve newlines, even on DOS
	    fconfigure $fd -translation binary
	    puts -nonewline $fd $s
	    close $fd
	} else {
	    # ambiguous, need help
	    set Manual(oldbind) [bind $t <Button-1>]
	    ManualSource $w
            tk_messageBox \
    	        -parent $w \
	        -icon question \
	        -message \
"Oops! Unambiguous placement of index anchor can't be determined.\
Please click on the HTML source where you want the index mark $newmark placed\
and press Reload." \
	        -type ok
	    bind $t <Button-1> "
		$Manual(oldbind)
	        code::ManualSaveIndexFile $w $newmark $file 1"
	    return
	}
    }

    if {$newmark == {} || ![info exists Manual(indexfile)]} {
	# didn't have write permission, or it didn't exist
        set file [tk_getSaveFile -initialdir [pwd] \
	    -initialfile [Preference Manual index]]
        if {$file == ""} {
            return
        }
        set Manual(indexfile) $file
    } else {
	set file $Manual(indexfile)
    }

    if {[catch {open $file w} fd]} {
        # error opening file
        tk_messageBox \
    	    -parent $w \
	    -icon error \
	    -message "$fd" \
	    -type ok
        set Manual(indexfile) {}
        return
    }

    foreach name [lsort -dictionary [array names ManualIndex]] {
	puts $fd "[list $name] [list $ManualIndex($name)]"
    }
    close $fd
}

proc size {t g r c {my 1} {mx 1}} {
    $t config -wrap none
    set sx $mx
    set sy $my
    set index 1
    set ly 0
    while {1} {
        set le [$t dlineinfo $index.0]
	if {$le == ""} {
	    break
	}
        set lx [lindex $le 2]
	incr lx 10
        if {$lx > $sx} {
	    set sx $lx
        }
        incr ly [lindex $le 4]
	incr index
    }
    if {$ly > $sy} {
        set sy $ly
    }
    return "$sx $sy"
}

# dynamic subwindow commands

#
# generate the go menu
#
proc code::ManualMenuGo {menu} {
    variable Manual
    variable CODEdocument

    set list {
       Home Back Forward {}
    }
    $menu delete 0 end
    ToolGenerate mainmenu $menu $list
    set count 0
    set Manual(true) 1
    foreach page $Manual(history) {
        set title [lindex $page 0]
        set url [lindex $page 1]
        if {$count == $Manual(historyIndex)} {
	    $menu add checkbutton -label $title \
               -command "code::ManualSetHistoryIndex \{$CODEdocument\} $count; \
		   code::ManualOpen \{$CODEdocument\} $url 0" -variable code::Manual(true)
        } else {
            $menu add command -label $title \
                -command "code::ManualSetHistoryIndex \{$CODEdocument\} $count;\
		    code::ManualOpen \{$CODEdocument\} $url 0"
        }
        incr count
    }
    return $menu
}
        
# history handling

#
# insert a page at the start of the history list
#
proc code::ManualHistoryInsert {w title url} {
    variable Manual
    
    if {$Manual(historyIndex) > 0} {
	# remove preceeding items
	incr Manual(historyIndex) -1
        set Manual(history) \
	    [lreplace $Manual(history) 0 $Manual(historyIndex)]
    }
    set Manual(history) [linsert $Manual(history) 0 [list $title $url]]
    ManualSetHistoryIndex $w 0
}

#
# set the history index
#
proc code::ManualSetHistoryIndex {w index} {
    variable Manual

    set len [llength $Manual(history)]    
    if {$index < 0 || $index >= $len} {
        return 0
    }
    
    set Manual(historyIndex) $index
    if {$index > 0} {
        set Manual(historyforward) 1
    } else {
        set forwardstate disabled
        set Manual(historyforward) 0
    }
    if {$index < [expr {$len - 1}]} {
        set backstate normal
        set Manual(historyback) 1
    } else {
        set Manual(historyback) 0
    }

    return 1 
    }
    
#
# go back up the history list
#
proc code::ManualHistoryBack {w} {
    variable Manual
    
    if ![ManualSetHistoryIndex $w [expr {$Manual(historyIndex) + 1}]] {
	windowBack $w
        return
    }
    ManualOpen $w [lindex [lindex $Manual(history) \
	$Manual(historyIndex)] 1] 0
}
       
#
# go forward down the history list
#
proc code::ManualHistoryForward {w} {
    global code::Manual

    if ![ManualSetHistoryIndex $w [expr {$Manual(historyIndex) - 1}]] {
        return
    }
    ManualOpen $w [lindex [lindex $Manual(history) \
	$Manual(historyIndex)] 1] 0
}
       
#
# go to the home entry
#
proc code::ManualHistoryHome {w} {
    global code::Manual

    set len [llength $Manual(history)]
    if {$len == 0} {
        return
    }
    incr len -1
    if ![ManualSetHistoryIndex $w $len] {
        return
    }
    ManualOpen $w [lindex [lindex $Manual(history) $len] 1] 0
}

#
# set the history list
#
proc code::ManualHistorySet {w new} {
    global code::Manual
    
    set Manual(history) $new
    ManualSetHistoryIndex $w 0
}
       
#
# ManualParseURL - parse a URL into it's component parts
#
proc code::ManualParseURL {name pv fv qv tv} {
    upvar $pv protocol
    upvar $fv file
    upvar $qv query
    upvar $tv target
    set protocol ""
    set file ""
    set query ""
    set target ""
    regexp {^([^:]+):([^?#]*)(\?([^#]*))?(#(.*))?} \
	$name junk protocol file junk query junk target
    }

#
# ManualBuildURL - if a URL is incomplete, use components of current
#
proc code::ManualBuildURL {w name} {
    variable Manual

    ManualParseURL $name protocol file query target
    if {$protocol == ""} {
	# no protocol found, try to build from previous url
	ManualParseURL $Manual($w,entrytext) protocol file query target
	if {![string match {[#\?]*} $name]} {
	    set dir [file dirname $file]
	    set namelist [file split $name]
	    set namestart 0
	    # remove leading ..'s
	    foreach elt $namelist {
		 if {$elt != ".."} {
		     break
		 }
	         set dir [file dirname $dir]
		 incr namestart
	    }
	    set namelist [lrange $namelist $namestart end]
	    set name [eval file join $namelist]
	    # XXX this assumes unix style pathnames
	    set name $protocol:$dir/$name
	    if {[string match http:/* $name] && ![string match http://* $name]} {
                regsub / $name // name
	    }
	} else {
	    set name $protocol:$file$name
	}
    }
    return $name
}

#
# handle callbacks from html::insert
#
set code::labnum 1
proc code::ManualHandler {win type args} {
    variable ManualImages
    variable labnum
    switch -exact $type {
	tag {
	    # have a unknown tag

	    switch -exact [lindex $args 0] {
		img {
		    # this is an image tag
		    set arg_border 1
		    set arg_alt "<image>"
		    set arg_align bottom
		    set type label
		    set args [lindex $args 1]
		    foreach arg $args {
			catch "set arg_[lindex $arg 0] [lindex $arg 1]"
		    }

		    if [info exists arg_src] {
			set name  [ManualBuildURL [winfo parent [winfo parent $win]] $arg_src]
			ManualParseURL $name protocol file query target
			set widget $win.img$labnum
			incr labnum
			set image 0
			if {[info exists ManualImages($name)]} {
			    set img $ManualImages($name)
			    $win window create insert -align $arg_align -pady 2 -padx 2 \
				-create "$type $widget -image $img"
			    set image 1
			} else {
			    ManualParseURL $name protocol file query target
			    $win window create insert -align $arg_align -pady 2 -padx 2 \
				-create "if \{!\[catch \"set code::ManualImages($name) \[image create photo -file $file\]\"\]\} \
				    \{$type $widget -image \$code::ManualImages($name)\} \
				    else \{$type $widget -text $arg_alt\}"
			    if {0 && ![catch "image create photo -file $file" img]} {
			        # $widget configure -image $img
			        $win image create insert -align $arg_align -image $img -pady 2 -padx 2
				set ManualImages($name) $img
			    }
			    set image 1
			} 
			if {!$image} {
			    $type $widget
			    $widget configure -text $arg_alt -highlightthickness 0
			    catch "$widget configure -bd $arg_border"
			    $win window create insert -align $arg_align -window $widget -pady 2 -padx 2
			}
		    }
		}
	    }
	}
    }
}

#
# ManualGetURL - get the contents of a URL
#
proc code::ManualGetURL {protocol file iserr} {
    upvar $iserr err
    variable Manual

    set err 0
    set s {}
    switch -exact $protocol {
        file {
            if [catch "open \"$file\"" f] {
                tk_messageBox \
    	            -parent . \
	            -icon error \
	            -message $f \
	            -type ok
		set err 1
	    } elseif [catch "read $f" s] {
	        tk_messageBox \
	            -parent . \
	            -icon error \
	            -message $s \
	            -type ok
	        set err 1
	        close $f
	    } else {
	        close $f
	    }
	}
	http {
	    set token [http::geturl http:$file]
	    http::wait $token
	    upvar #0 $token state
	    if {$state(status) != "ok"} {
	        tk_messageBox \
	            -parent . \
	            -icon error \
	            -message "Can't get http:/$file" \
	            -type ok
		set err 1
	    }
	    set s $state(body)
	    unset state
	}
	default {
	    tk_messageBox \
	        -parent . \
	        -icon error \
	        -message "Protocol $protocol: not supported" \
	        -type ok
	    set err 1
	}
    }
    return $s
}

#
# HTMLstartup - return the command to open an html window
#
proc code::HTMLstartup {doc} {
    return "HTML [$doc.contents cget -contents]"
}

#
# HTML - open an HTML window
#
proc code::HTML {name args} {
    variable Manual

    set type "Manual"
    set w [CODEfind $type $name]
    if {$w != {}} {
	# have an HTML window, raise and return
	DOCstate $w
	$w raise
	return $w
    }
    # a new html window
    set w [eval TEXT $args]
    set t $w.contents
    $w menu entryconfigure Close -command "code::HTMLcleanup $w"
    $w configure -type $type -startupproc "code::HTMLstartup" -raiseproc code::MANUALraise
    $w configure -image {} -icontext $type -title $name
    $t configure -contents $name -highlightthickness 0 -positionproc {} \
	-deleteproc code::HTMLnull -insertproc code::HTMLnull
    $t configure -scrollbar auto
    set Manual($w,entrytext) ""
    # set up cursor
    set Manual($w,cursor) left_ptr
    bind $t <Enter> "$w config -cursor $code::Manual($w,cursor)"
    # set up link events
    $t tag bind link <Enter> "code::ManualEnter $w %W %x %y"
    $t tag bind link <Motion> "code::ManualEnter $w %W %x %y"
    $t tag bind link <Leave> "code::ManualLeave $w %W %x %y"
    $t tag bind link <ButtonRelease-1> "code::ManualGoto $w %W %x %y"
    if {[llength [allDocs]] == 1} {
        $w configure -state maximized
    }
    if {$name != {}} {
        ManualOpenURL $w $name 
    }
    return $w
}

#
# HTMLnull - a donothing insert or delete proc (makes window readonly)
#
proc code::HTMLnull {args} {
    return 0; # don't update widget
}

#
# HTMLcleanup - clean up and destroy an HTML window
#
proc code::HTMLcleanup {w} {
    variable Manual

    foreach entry [array names Manual $w,*] {
	unset Manual($entry)
    }
    destroy $w
}

#
# ManualOpenURL put the contents of a URL in a document
#
proc code::ManualOpenURL {w name {remember 1} {force 0}} {
    variable Manual
    variable manual

    if {$w == {}} {
	set w [HTML $name]
    }
    set text $w.contents
    ManualParseURL $name protocol file query target
    ManualParseURL $Manual($w,entrytext) uprotocol ufile uquery utarget

    set protocol [string tolower $protocol]
    set uprotocol [string tolower $uprotocol]
    set err 0
    if {$force || $protocol != $uprotocol || $file != $ufile} {
	# this is a new document
        set oldcursor $Manual($w,cursor)
        set Manual($w,cursor) watch
        $text config -cursor $Manual($w,cursor)
        set s [ManualGetURL $protocol $file err]

	# insert s into text widget
	if {!$err} {
            if {[string match http:* $manual]} {
	        regsub ^$manual http:$file {} tree
	    } else {
	        regsub ^$manual $file {} tree
	    }
	    # remove comtents.html
	    regsub {/contents.html} $tree {} tree
	    set t $Manual(tree)
	    ManualTree:open $t $tree
	    set dir [file split $tree]
	    set path [lindex $dir 0]
	    set dir [lrange $dir 1 end]
	    foreach element $dir {
	        set path $path$element
	        ManualTree:open $t $path
	        append path /
	    }
	    
	    ManualTree:show $t $tree
	    ManualTree:setselection $t $tree
            set insertproc [$text cget -insertproc]
            $text configure -insertproc {}
            set deleteproc [$text cget -deleteproc]
            $text configure -deleteproc {}
            $text delete 1.0 end
	    eval $text mark unset [$text mark names]
	    set pfont [Preference Manual fontproportional]
	    set ffont [Preference Manual fontfixed]
	    set tail [file tail $file]
	    wm title . $tail
	    $w configure -title $tail
	    $w configure -icontext $tail
	    $text configure -contents $name
            set Manual($w,entrytext) $name
            set Manual($w,file) $name
	    set Manual($w,source) $s
            html::insert -unknown code::ManualHandler \
		-proportional [lindex $pfont 0] \
		-psize [lindex $pfont 1] \
		-fixed [lindex $ffont 0] \
		-fsize [lindex $ffont 1] \
		-linkcolor [Preference Manual colorlink] \
		$text.text $s 
	    colorText $text $file 
            $text mark set insert 1.0
	    $text configure -insertproc $insertproc
	    $text configure -deleteproc $deleteproc
            set title [wm title .]
	    $w configure -title $title
	    $w configure -icontext $title
            SearchUpdate $w $text $title $file
            if {$remember} {
                ManualHistoryInsert $w $title $name
                set remember 0
            }
	    ManualIndexBuild
 	}
        set Manual($w,cursor) $oldcursor
        $text config -cursor $Manual($w,cursor)
    }

    # goto a target
    if {!$err && $target != ""} {
	if {![catch {$text see N:$target}]} {
	    $text mark set insert [$text index N:$target]
	} else {
	    # look for an index target that isn't in HTML yet
	    if {[regexp {^IDX([0-9]*\.[0-9]*).*} $target all target]} {
	        $text mark set insert $target
		$text see $target
	    }
	}

        set Manual($w,entrytext) $name
        set title [wm title .]
	$w configure -title $title
        SearchUpdate $w $text $title $file
        if {$remember} {
            ManualHistoryInsert $w $title $name
        }
    }
    MANUALraise $w
}

#
# ManualOpen put up the open dialog box and open a file
#
proc code::ManualOpen {w {file ""} {remember 1}} {
    if {$file == ""} {
	set initialdir [Preference Manual dirfileopen]
	if {$initialdir == {} || ![file isdirectory $initialdir]} {
	    set initialdir [pwd]
	}
        set f [tk_getOpenFile -initialdir $initialdir \
            -filetypes "{{HTML Files} {.html .htm}} {{All Files} *}"]
        if {$f != ""} {
	    Preference Manual dirfileopen [file dirname $f]
            set f [ManualPrefix $f]
        }
    } else {
        set f $file
    }
    if {$f != ""} {
	ManualOpenURL $w "$f" $remember
    }
}

#
# ManualPrefix - add a file: to non-http: file names
#
proc code::ManualPrefix {file} {
    if {$file != {} && ![string match http:* $file]} {
        set file file:$file
    }
    return $file
}
#
# ManualEnter - A link has been entered
#
proc code::ManualEnter {doc w x y} {
    variable Manual
    set text $doc.contents
    status . [ManualBuildURL $doc [html::link $w $x $y]]
    if {$Manual($doc,cursor) == "watch"} {
	return
    }
    set Manual($doc,cursor) hand2
    $text config -cursor $Manual($doc,cursor)
}

#
# ManualLeave - A link has been left
#
proc code::ManualLeave {doc w x y} {
    variable Manual

    code::status . ""
    if {$Manual($doc,cursor) == "watch"} {
	return
    }
    set Manual($doc,cursor) left_ptr
    $doc.contents config -cursor $Manual($doc,cursor)
}

#
# ManualGoto - goto a link in the text
#
proc code::ManualGoto {doc w x y} {
    ManualOpenURL $doc [ManualBuildURL $doc [html::link $w $x $y]]
    set Manual($doc,cursor) left_ptr
    $doc.contents config -cursor $Manual($doc,cursor)
}

#
# Copyright (C) 1997,1998 D. Richard Hipp
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
# 
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA  02111-1307, USA.
#
# Author contact information:
#   drh@acm.org
#   http://www.hwaci.com/drh/
#
# $Revision: 1.75 $
#

#
# Create a new tree widget.  $args become the configuration arguments to
# the canvas widget from which the tree is constructed.
#
proc code::ManualTree:create {w args} {
  global code::ManualTree
  eval canvas $w $args
  bind $w <Destroy> "code::ManualTree:delitem $w /"
  ManualTree:dfltconfig $w /
  ManualTree:buildwhenidle $w
  set ManualTree($w:selection) {}
  set ManualTree($w:selidx) {}
  set ManualTree($w:font) {}
}

# Initialize a element of the tree.
# Internal use only
#
proc code::ManualTree:dfltconfig {w v} {
  global code::ManualTree
  set ManualTree($w:$v:children) {}
  set ManualTree($w:$v:open) 0
  set ManualTree($w:$v:icon) {}
  set ManualTree($w:$v:text) {}
  set ManualTree($w:$v:file) {}
  set ManualTree($w:$v:italic) 0
  set ManualTree($w:$v:y) 0
}

#
# Pass configuration options to the tree widget
#
proc code::ManualTree:config {w args} {
  eval $w config $args
}

#
# Insert a new element $v into the tree $w.
#
proc code::ManualTree:newitem {w v args} {
  global code::ManualTree 

    set dir [file dirname $v]
    set n [file tail $v]
    if {![info exists ManualTree($w:$dir:open)]} {
        ManualTree:newitem $w $dir -text [file tail $dir]
        # error "parent item \"$dir\" is missing"
    }

  set i [lsearch -exact $ManualTree($w:$dir:children) $n]
  if {$i>=0} {
    error "item \"$v\" already exists"
  }
  lappend ManualTree($w:$dir:children) $n
  ManualTree:dfltconfig $w $v
  foreach {op arg} $args {
    switch -exact -- $op {
      -image {set ManualTree($w:$v:icon) $arg}
      -text {set ManualTree($w:$v:text) $arg}
      -file {set ManualTree($w:$v:file) $arg}
      -italic {set ManualTree($w:$v:italic) $arg}
    }
  }
  ManualTree:buildwhenidle $w
}

#
# Delete element $v from the tree $w.  If $v is /, then the widget is
# deleted.
#
proc code::ManualTree:delitem {w v} {
  global code::ManualTree
  if {![info exists ManualTree($w:$v:open)]} return
  if {[string compare $v /]==0} {
    # delete the whole widget
    catch {destroy $w}
    foreach t [array names ManualTree $w:*] {
      unset ManualTree($t)
    }
    return
  }
  foreach c $ManualTree($w:$v:children) {
    ManualTree:delitem $w $v/$c
  }
  unset ManualTree($w:$v:open)
  unset ManualTree($w:$v:children)
  unset ManualTree($w:$v:icon)
  unset ManualTree($w:$v:text)
  unset ManualTree($w:$v:file)
  unset ManualTree($w:$v:italic)
  unset ManualTree($w:$v:y)
  set dir [file dirname $v]
  set n [file tail $v]
  set i [lsearch -exact $ManualTree($w:$dir:children) $n]
  if {$i>=0} {
    set ManualTree($w:$dir:children) [lreplace $ManualTree($w:$dir:children) $i $i]
  }
  ManualTree:buildwhenidle $w
}

#
# Change the selection to the indicated item
#
proc code::ManualTree:setselection {w v} {
  global code::ManualTree
  set ManualTree($w:selection) $v
  ManualTree:drawselection $w
}

# 
# Retrieve the current selection
#
proc code::ManualTree:getselection w {
  global code::ManualTree
  return $ManualTree($w:selection)
}

#
# Bitmaps used to show which parts of the tree can be opened.
#
set maskdata "#define solid_width 9\n#define solid_height 9"
append maskdata {
  static unsigned char solid_bits[] = {
   0xff, 0x01, 0xff, 0x01, 0xff, 0x01, 0xff, 0x01, 0xff, 0x01, 0xff, 0x01,
   0xff, 0x01, 0xff, 0x01, 0xff, 0x01
  };
}
set data "#define open_width 9\n#define open_height 9"
append data {
  static unsigned char open_bits[] = {
   0xff, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x7d, 0x01, 0x01, 0x01,
   0x01, 0x01, 0x01, 0x01, 0xff, 0x01
  };
}
image create bitmap ManualTree:openbm -data $data -maskdata $maskdata \
  -foreground black -background white
set data "#define closed_width 9\n#define closed_height 9"
append data {
  static unsigned char closed_bits[] = {
   0xff, 0x01, 0x01, 0x01, 0x11, 0x01, 0x11, 0x01, 0x7d, 0x01, 0x11, 0x01,
   0x11, 0x01, 0x01, 0x01, 0xff, 0x01
  };
}
image create bitmap ManualTree:closedbm -data $data -maskdata $maskdata \
  -foreground black -background white

# Internal use only.
# Draw the tree on the canvas
proc code::ManualTree:build w {
  global code::ManualTree
  $w delete all
  catch {unset ManualTree($w:buildpending)}
  set ManualTree($w:y) 30
  ManualTree:buildlayer $w / 10
  $w config -scrollregion [$w bbox all]
  ManualTree:drawselection $w
  set all [$w bbox all]
  if {$all == {}} {
      return
  }
  set width [expr {[lindex $all 2] - [lindex $all 0] + 5}]
  # set min, max width
  set max [expr {[winfo width [winfo toplevel $w]] / 2}]
  if {$width < 50} {
      set width 50
  } elseif {$width > $max} {
      set width $max
  }
  $w config -width $width
}

# Internal use only.
# Build a single layer of the tree on the canvas.  Indent by $in pixels
proc code::ManualTree:buildlayer {w v in} {
  global code::ManualTree
  if {$v=="/"} {
    set vx {}
  } else {
    set vx $v
  }
  # remember this placement
  set ManualTree($w:$v:y) [expr {$ManualTree($w:y) - 50}]
  set start [expr {$ManualTree($w:y)-10}]
  foreach c $ManualTree($w:$v:children) {
    set y $ManualTree($w:y)
    incr ManualTree($w:y) 17
    $w create line $in $y [expr {$in+10}] $y -fill gray50 
    set icon $ManualTree($w:$vx/$c:icon)
    set x [expr {$in+12}]
    if {[string length $icon]>0} {
      set k [$w create image $x $y -image $icon -anchor w -tags x]
      incr x 20
      set ManualTree($w:tag:$k) $vx/$c
    }
    set ManualTree($w:$vx/$c:y) [expr {$y - 50}]
    if {$ManualTree($w:$vx/$c:text) != {}} {
	set text $ManualTree($w:$vx/$c:text)
    } else {
	set text $c
    }
    set font [Preference Manual fontproportional]
    if {$ManualTree($w:$vx/$c:italic)} {
	if {[llength $font] == 1} {
	    set font [concat $font {} italic]
	} else {
	    lappend font italic
	}
    }
    set j [$w create text $x $y -text $text \
	-font $font -anchor w -tags x]
    set ManualTree($w:tag:$j) $vx/$c
    set ManualTree($w:$vx/$c:tag) $j
    if {[string length $ManualTree($w:$vx/$c:children)]} {
      if {$ManualTree($w:$vx/$c:open)} {
         set j [$w create image $in $y -image ManualTree:openbm]
         $w bind $j <1> "set \"code::ManualTree($w:$vx/$c:open)\" 0; code::ManualTree:build $w"
         ManualTree:buildlayer $w $vx/$c [expr {$in+18}]
      } else {
         set j [$w create image $in $y -image ManualTree:closedbm]
         $w bind $j <1> "set \"code::ManualTree($w:$vx/$c:open)\" 1; code::ManualTree:build $w"
      }
    }
  }
  if {[info exists y]} {
      set j [$w create line $in $start $in [expr {$y+2}] -fill gray50 ]
      $w lower $j
  }
}

# Open a branch of a tree
#
proc code::ManualTree:open {w v} {
  global code::ManualTree
  update idletasks
  if {[info exists ManualTree($w:$v:open)] && $ManualTree($w:$v:open)==0
      && [info exists ManualTree($w:$v:children)] 
      && [string length $ManualTree($w:$v:children)]>0} {
    set ManualTree($w:$v:open) 1
    ManualTree:build $w
  }
}

# Show a branch of a tree
#
proc code::ManualTree:show {w v} {
  global code::ManualTree

  update idletasks
  set all [$w bbox all]
  if {[catch {expr $ManualTree($w:$v:y).0 / ([lindex $all 3].0 - [lindex $all 1].0)} yview]} {
      return
  }
  $w yview moveto $yview
}

proc code::ManualTree:close {w v} {
  global code::ManualTree
  if {[info exists ManualTree($w:$v:open)] && $ManualTree($w:$v:open)==1} {
    set ManualTree($w:$v:open) 0
    ManualTree:build $w
  }
}

# Internal use only.
# Draw the selection highlight
proc code::ManualTree:drawselection w {
  global code::ManualTree
  if {[string length $ManualTree($w:selidx)]} {
    $w delete $ManualTree($w:selidx)
  }
  set v $ManualTree($w:selection)
  if {[string length $v]==0} return
  if {![info exists ManualTree($w:$v:tag)]} return
  set bbox [$w bbox $ManualTree($w:$v:tag)]
  if {[llength $bbox]==4} {
    set sc [Preference General colorselection]
    set i [eval $w create rectangle $bbox -fill $sc -outline {{}}]
    set ManualTree($w:selidx) $i
    $w lower $i
  } else {
    set ManualTree($w:selidx) {}
  }
}

# Internal use only
# Call ManualTree:build then next time we're idle
proc code::ManualTree:buildwhenidle w {
  global code::ManualTree
  if {![info exists ManualTree($w:buildpending)]} {
    set ManualTree($w:buildpending) 1
    after idle "code::ManualTree:build $w"
  }
}

#
# Return the full pathname of the label for widget $w that is located
# at real coordinates $x, $y
#
proc code::ManualTree:labelat {w x y} {
  set x [$w canvasx $x]
  set y [$w canvasy $y]
  global code::ManualTree
  foreach m [$w find overlapping $x $y $x $y] {
    if {[info exists ManualTree($w:tag:$m)]} {
      return $ManualTree($w:tag:$m)
    }
  }
  return ""
}
