#
# CODE preferences
#
# Default CODE preferences are set here.
#
set code::PreferenceFile ""
set code::PreferenceDelayedCallbacks ""
set code::PreferenceDelayCallbacks 0
set code::Build(PROJECT) {}
set code::Build(OBJECT) {}

array set code::PreferenceProjectWizard {
Start {
  "The project wizard"
"The project wizard will step you through the process of creating and modifying a project.

You can press Finish at any time and use the CODE menus to configure aspects of the project by hand."
  {} ProjectName Finish
}
ProjectName {
  "Choose the project's name"
"The project's name will be the default name of the project file (with a .code extension) as well as the base name of any files, such as startup code, that CODE might create for the project."
  Start ProjectBase Finish
  {
    variable Configure
    set l [label $c.label -text "Project Name:"]
    grid $l -column 0 -row 0

    set Configure(Build,name) [Preference Build name]
    set e [entry $c.entry -textvariable code::Configure(Build,name) -width 40]
    bind $e <Return> code::WizardCheck
    grid $e -column 1 -row 0 -sticky ew
    grid columnconfigure $c 1 -weight 1
  }
  {
    variable Configure
    PreferenceSetIfChanged Build name $Configure(Build,name)
  }
}
ProjectBase {
  "Choose the project's base project"
"Your project's base project is a set of definitions that your project will inherit. You can use the standard CODE base file, or choose one that more closely corresponds to your project's requirements.

You can use the Project Base button to choose a predefined project base file."
  ProjectName ProjectProcessor Finish
  {
    variable PreferenceImport
    variable ProjectImport
    variable INTROL
    proc ProjectImportProc {} {
        variable PreferenceImport
        variable ProjectImport
        variable INTROL

        set ProjectImport [file nativename [eval file join [subst -nobackslashes -nocommands $PreferenceImport]]]
    }
    set l [label $c.label -text "Base Project:"]
    grid $l -column 0 -row 0

    ProjectImportProc
    trace variable PreferenceImport w "code::ProjectImportProc; #"
    set e [entry $c.entry -textvariable code::ProjectImport -width 40]
    bind $e <Return> code::WizardCheck
    grid $e -column 1 -row 0 -sticky ew
    grid columnconfigure $c 1 -weight 1
    set b [WizardButton $c.button ProjectBase]
    grid $b -column 2 -row 0
  }
  {
    variable PreferenceImport
    variable ProjectImport
    if {![file exists $ProjectImport]} {
	error "$ProjectImport does not exist."
    }
    set expanded [BuildFileName $ProjectImport]
    regsub -all " " $expanded "\\ " expanded
    regsub -all / $expanded " " expanded
    regsub -all {\$\{INTROL\}} $expanded \"\${INTROL}\" expanded
    set PreferenceImport $expanded
  }
}
ProjectProcessor {
  "Choose a processor and variant"
"Here you can select the microcontoller that you are targeting with your project and you can also select which member of the microcontroller's family you will be working with.

The Processor and Variant buttons will bring up menus of processors supported by CODE."
  ProjectBase ProjectResult Finish
  {
    variable Configure
    set Configure(Build,processor) [Preference Build processor]
    if {$Configure(Build,processor) == {}} {
        set Configure(Build,processor) Unknown
    }
    set b [WizardButton $c.proc Processor code::Configure(Build,processor)]
    grid $b -column 1 -row 1
    set l [label $c.lproc -text "Processor"]
    grid $l -column 1 -row 0
    set Configure(Build,variant) [Preference Build variant]
    if {$Configure(Build,variant) == {}} {
        set Configure(Build,variant) Unknown
    }
    set b [WizardButton $c.var Variant code::Configure(Build,variant)]
    grid $b -column 2 -row 1
    set l [label $c.lvar -text "Variant"]
    grid $l -column 2 -row 0
  }
}
ProjectResult {
  "Choose the project's result"
"A CODE project can produce a program, a library, or just a project containing other projects. You can specify a project result as \"None\" to indicate that the project doesn't directly produce any result.

Normally, your projects will build a program that can be run on the target microcontroller."
  ProjectProcessor ProjectNotes Finish
  {
    variable Configure
    set Configure(Build,target) [Preference Build target]
    set l [label $c.lproc -text "Result:"]
    grid $l -column 0 -row 1
    set b [WizardButton $c.proc ProjectResult code::Configure(Build,target)]
    grid $b -column 1 -row 1
  }
  {
    variable Configure
    PreferenceSetIfChanged Build target $Configure(Build,target)
  }
}
ProjectNotes {
  "Add notes to your project"
"A project's notes can be used to hold any information you'd like to keep about a project.

The notes are kept in the project file and can be modified at any time by choosing Edit Project Notes... from the Project menu."
  ProjectResult MemoryMap Finish
  {
    set b [WizardButton $c.notes ProjectNotes]
    grid $b -column 1 -row 1
  }
}
MemoryMap {
  "Set up the project's memory map"
"The project's memory map controls where your project's code and data will be placed in the target microcontroller's memory.

The memory map controls how CODE generates the linker command file for your project. Normally the linker command file is not saved on the disk but is given to the linker directly when the project is built. You can view the linker command file by selecting View LD File from the Project menu."
  ProjectNotes Environment Finish
  {
    set b [WizardButton $c.mem MemoryMap]
    grid $b -column 1 -row 1
  }
}
Environment {
  "Set up the project's execution environment"
"The project's execution environment is set up in the start up code generated by CODE. You can modify the execution environment for any special needs your project may have.

The start up code is generated into an assembly file called <your_project_name>start.s"
  MemoryMap ProgramOptions Finish
  {
    set b [WizardButton $c.env ConfigureEnvironment]
    grid $b -column 1 -row 1
  }
}
ProgramOptions {
  "Choose global build options for your project"
"You can specify build options that will be used globally for files in your project. Individual files added to your project may have their own build options also."
  Environment ProjectScripts Finish
  {
    set b [WizardButton $c.c GlobalCOptions]
    grid $b -column 1 -row 1
    set b [WizardButton $c.a GlobalAOptions]
    grid $b -column 2 -row 1
    set b [WizardButton $c.l GlobalLOptions]
    grid $b -column 3 -row 1
  }
}
ProjectScripts {
  "Set up global command scripts"
"You can define command scripts that will be executed as your project is built. The global scripts are executed for all project files. You can also specify scripts for individual project files.

You don't normally have to define any build scripts."
  ProgramOptions AddFiles Finish
  {
    set b [WizardButton $c.pb PreBuild]
    grid $b -column 1 -row 1
    set b [WizardButton $c.pt PreTranslate]
    grid $b -column 2 -row 1
    set b [WizardButton $c.at PostTranslate]
    grid $b -column 3 -row 1
    set b [WizardButton $c.pl PreLink]
    grid $b -column 1 -row 2
    set b [WizardButton $c.ab PostBuild]
    grid $b -column 2 -row 2
  }
}
AddFiles {
  "Add files to your project"
"You project will usually contain one or more source files or other project files. you can add files now, or later from the CODE environment.

You can add files one at a time, or use a folder window to select multiple files."
  ProjectScripts Dependencies Finish
  {
    set b [WizardButton $c.a AddToProject]
    grid $b -column 1 -row 1
    set b [WizardButton $c.f Folder]
    grid $b -column 2 -row 1
  }
}
Dependencies {
  "Find #include dependencies"
"CODE will automatically find the #include files that any C source files in your project depend on."
  AddFiles ProjectOutput Finish
  {
    set b [WizardButton $c.a AllDepends]
    grid $b -column 1 -row 1
  }
}
ProjectOutput {
  "Set up your project's output files"
"You can specify other output files that your project will produce in addition to the default object file created by CODE."
   Dependencies Finish Finish
  {
    set b [WizardButton $c.o ProjectOutput]
    grid $b -column 1 -row 1
  }
}
Finish {
  "Your project is ready"
"Your project is ready to use. You can modify any aspect of your project in the CODE environment by using menu options or running the wizard again."
  ProjectOutput {} Close
}
Close {
    ""
    ""
    {} {} {}
    {
      # this destroys the window
      PreferencesSave
      PreferenceWizardClose
    }
}
}

#
# PreferenceFile - return the name of the current project file
#
proc code::PreferenceFile {} {
    variable PreferenceFile

    return $PreferenceFile
}
#
# PreferenceCloseCurrent - close the current project
#
proc code::PreferenceCloseCurrent {} {
    variable Preferences
    variable PreferenceChanged

    # close any open documents
    CODEclose
    if {![PreferencesSave 0 1]} {
        CODEstartup
	return
    }
    # start with a completely clean state
    Preferences {} 0 1 0
    CODEstartup
}

#
# PreferenceNew - forget preferences and start the project wizard
#
proc code::PreferenceNew {} {
    variable Preferences

    # close any open documents
    CODEclose
    if {![PreferencesSave 0 1]} {
        CODEstartup
	return
    }
    Preferences {} 0 1 0
    PreferenceWizard
}

#
# PreferenceWizard - show the preference wizard
#
proc code::PreferenceWizard {} {
    set box .wizard
    set geometry [Preference Geometry wizard]
    if {[winfo exists $box]} {
        if {$geometry != {}} {
	    # the window has had its geometry change saved
	    setGeometry $box $geometry
	}

	WizardShow Start PreferenceProjectWizard
        wm deiconify $box
	raise $box
        # this code seems to get around a bug in Tk (wm geometry for
        # a hidden window)
        global tcl_platform
        if {$tcl_platform(platform) == "windows" && $geometry != {}} {
            setGeometry $box $geometry
        }
	return
    }

    # create the wizard
    toplevel $box 
    wm transient $box .
    wm protocol $box WM_DELETE_WINDOW "code::PreferenceWizardClose"
    set f [frame $box.buttons]
    set backBtn [button $f.back -text Back -width 10 -state disabled]
    set forwardBtn [button $f.forward -text Forward -width 10 -state disabled]
    set extraBtn [button $f.extra -text Finish -width 10]
    grid $backBtn -in $f -row 0 -column 0 
    grid $forwardBtn -in $f -row 0 -column 1 
    grid $extraBtn -in $f -row 0 -column 2 
    grid $f -in $box -row 2 -column 0 -sticky ew -padx 5 -pady 5
    set f [frame $box.frame]
    grid $f -in $box -row 1 -column 0 -sticky new
    grid rowconfigure $box 1 -weight 1
    grid columnconfigure $box 0 -weight 1
    set t [scrolltext $box.text -scrollbar auto -wrap word -state disabled \
	-height 10]
    grid $t -in $box -row 0 -column 0 -sticky nsew

    if {$geometry != {}} {
        # the window has had its geometry change saved
        setGeometry $box $geometry
    }
    WizardShow Start PreferenceProjectWizard
}

#
# PreferenceWizardClose - destroy a wizard after saving it's geometry
#
proc code::PreferenceWizardClose {} {

    set w .wizard
    if {![winfo exists $w]} {
	return
    }
    if {![catch {wm geometry $w} geometry]} {
	# save only the window's position, not its size
        Preference Geometry wizard $geometry
    }
    destroy $w
}

#
# PreferenceBase - View/Change the base project
#
proc code::PreferenceBase {} {
    variable PreferenceChanged
    variable PreferenceImport
    variable PreferenceFile
    variable PreferenceBase
    variable PreferenceBaseSelect
    variable INTROL
    variable FileIcons

    set box .projectbase
    toplevel $box 
    wm transient $box .
    wm protocol $box WM_DELETE_WINDOW "set code::PreferenceBase {}"
    set f [frame $box.buttons]
    set okBtn [button $f.ok -text Ok -width 6 \
        -command "set code::PreferenceBase {}"]
    set changeBtn [button $f.change -text Change -width 6 -state disabled \
        -command "set code::PreferenceBase Change"]
    grid $okBtn -in $f -row 0 -column 0 
    grid $changeBtn -in $f -row 0 -column 1 
    grid $f -in $box -row 2 -column 0 -sticky ew -padx 5 -pady 5
    set l [label $box.l -textvariable code::PreferenceBaseName]
    grid $l -in $box -row 0 -column 0 -sticky ew -padx 5 -pady 5
    set f [frame $box.window]
    grid $f -in $box -row 1 -column 0 -sticky nsew -padx 5 -pady 5
    grid rowconfigure $box 1 -weight 1
    grid columnconfigure $box 0 -weight 1
    set t [treecontrol $f.tree -text "" -width 30 -selectmode single \
	-onselect "code::PreferenceBaseSelect $f $changeBtn" \
	-bd 2 -highlightthickness 0]
    grid $t -in $f -row 1 -column 0 -sticky ns
    grid rowconfigure $f 1 -weight 1
    set text [scrolltext $f.text -scrollbar auto -width 40 -wrap none \
	-bd 2 -highlightthickness 0]
    grid $text -in $f -row 1 -column 1 -sticky nsew
    grid columnconfigure $f 1 -weight 1
    # the default project
    set default {"$INTROL" tcltk code CODE.code}
    $t insert current -image $FileIcons(codefile.icon) \
	-text "Current: [file rootname [lindex $PreferenceImport end]]" \
	-user $PreferenceImport
    $t insert default -image $FileIcons(codefile.icon) \
	-text "Default: [file rootname [lindex $default end]]" \
	-user $default

    # add other base project files from Configure
    PreferenceBaseAdd $t [file join $INTROL Configure]

    $t selection set current
    PreferenceBaseSelect $f $changeBtn
    placewindow $box widget .
    wm title $box "Base Project"

    set oldFocus [focus]
    set oldGrab [grab current $box]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $box

    tkwait variable code::PreferenceBase
    catch {focus $oldFocus}
    grab release $box
    destroy $box
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    if {$PreferenceBase == {}} {
	return
    }
    # changing the base project
    CODEclose
    set name [Preference Build name]
    set PreferenceImport $PreferenceBaseSelect
    set PreferenceChanged 1
    if {$name == ""} {
	set saveas 1
    } else {
	set saveas 0
    }

    if {![PreferencesSave $saveas]} {
        CODEstartup
	return
    }
    # reload the project
    Preferences $PreferenceFile 0 1 0 $PreferenceBaseSelect
    if {!$saveas} {
	# restore the original name
        Preference Build name $name 1
    }
    CODEstartup
}

#
# PreferenceBaseAdd - add base files to the project base tree
#
proc code::PreferenceBaseAdd {t dir {need {}}} {
    variable FileIcons

    # match .code and .cod in either case
    set projects [lsort -dictionary \
	[concat [glob -nocomplain $dir/*.\[cC\]\[oO\]\[dD\]\[eE\]] \
	[glob -nocomplain $dir/*.\[cC\]\[oO\]\[dD\]]]]
    if {$projects != {}} {
	set parent {}
	foreach dir $need {
	    # add any needed folders
	    set dir [file tail $dir]
	    regsub -all " " $dir "" id
	    if {[catch {$t nodecget $parent$id -user}]} {
                $t insert $parent$id -text $dir \
		    -image $FileIcons(directory.icon) \
		    -parent $parent
	    append parent $id
	    }
	}
	# add these projects
	foreach project $projects {
	    set name [file rootname [file tail $project]]
	    set expanded [BuildFileName $project]
	    regsub -all " " $expanded "\\ " expanded
	    regsub -all / $expanded " " expanded
	    regsub -all {\$\{INTROL\}} $expanded \"\${INTROL}\" expanded
	    regsub -all " " $name "" id
            $t insert $parent$id -text $name \
		-image $FileIcons(codefile.icon) \
		-parent $parent \
		-user $expanded
	}
    }
    foreach file [glob -nocomplain $dir/*] {
	if {[file isdirectory $file]} {
	    # add subdirectories
            PreferenceBaseAdd $t $file "$need [list [file tail $file]]"
	}
    }
}

#
# PreferenceBaseSelect - A Base project has been selected
#
proc code::PreferenceBaseSelect {f b} {
    variable PreferenceImport
    variable PreferenceBaseName
    variable PreferenceBaseSelect
    variable INTROL

    set text $f.text
    set t $f.tree
    set node [$t selection get]
    $t see $node
    # get the project name
    set project [$t nodecget $node -user]
    if {$node != "current" && $PreferenceImport != $project} {
       $b configure -state normal
    } else {
       $b configure -state disabled
    }
    if {$project == {}} {
	# this isn't a project, must be a folder
        $b configure -state disabled
        $text configure -state normal
        $text delete 1.0 end
        $text configure -state disabled
        set PreferenceBaseName "No base project selected"
	$t expand $node
	return
    }
    # remember in case we change to it
    set PreferenceBaseSelect $project

    # show the notes for the selected base project
    set name [subst -nobackslashes -nocommands $project]
    set name [eval file join $name]
    set fd [open $name]
    array set values [read $fd]
    close $fd
    set value "No description available"
    if {[info exists values(Build)]} {
        foreach "item value" $values(Build) {
	    # find notes
	    if {$item == "notes"} {
	        break
	    }
        }
    }
    $text configure -state normal
    $text delete 1.0 end
    $text insert insert $value
    $text configure -state disabled

    set name [file nativename $name]
    if {$node != "current" && $PreferenceImport != $project} {
	set which Selected
    } else {
	set which Current
    }
    set PreferenceBaseName "$which: $name"
}

#
# PreferenceBuild - build the project
#
proc code::PreferenceBuild {w} {
    if {[catch {Build $w build} result]} {
        tk_messageBox \
    	    -parent $w \
	    -icon error \
	    -message "Can't build: $result." \
	    -type ok
    }
}

#
# PreferenceProjectName - get the current project name
#
proc code::PreferenceProjectName {} {
    variable PreferenceChanged

    set name [Preference Build name]
    if {$name == {}} {
	set name Unknown
    }
    if {[info exists PreferenceChanged] && $PreferenceChanged} {
        append name { [Modified]}
    }
    set maintitle "$name"
    return $maintitle
}

#
# PreferenceEdit - set up the CODE preferences
#
proc code::PreferenceEdit {{w {}}} {
    variable UI

    if {!$UI} {
	# no user interface
	return
    }

    set top .preferences
    set geometry [Preference Geometry preferences]
    if {[winfo exists $top]} {
        if {$geometry != {}} {
	    # the window has had its geometry change saved
	    setGeometry $top $geometry
	}
	if {$w != {}} {
            Notebook:raise $top.nb .
	}
        wm deiconify $top
	raise $top
        # this code seems to get around a bug in Tk (wm geometry for
        # a hidden window)
        global tcl_platform
        if {$tcl_platform(platform) == "windows" && $geometry != {}} {
            setGeometry $top $geometry
        }
	Preference General showpreferences 1
	return
    }

    # create the Preferences window
    set n [CODENotebook Preferences $top preferences]
    wm resizable $top 0 0
    Notebook:config $n -width 450 -height 350
    if {$w != {}} {
        Notebook:raise $top.nb .
    }
    bind $top <F1> "code::OpenHelp Applications Preferences.html"
}

#
# PreferenceSetIfChanged - set a preference if it has changed
#
proc code::PreferenceSetIfChanged {cat elt value} {
    if {$value != [Preference $cat $elt]} {
        Preference $cat $elt $value 1
    }
}

#
# PreferenceClose - withdraw a window after saving its geometry
#
proc code::PreferenceClose {w which} {

    if {![winfo exists $w]} {
	return
    }
    if {![catch {wm geometry $w} geometry]} {
	# save only the window's position, not its size
	regexp {([0-9]*x[0-9]*)(.*)} $geometry foo foo geometry
        Preference Geometry ${which} $geometry
    }
    Preference General show$which {} 1
    wm withdraw $w
}

#
# PreferenceProcessorInfo - set up processor information
#
proc code::PreferenceProcessorInfo {} {
    variable INTROL
    variable Preferences
    variable Configure

    # check for the list already read
    if {[info exists Preferences(processors)]} {
	return $Preferences(processors)
    }

    # get the list of processor directories from the Configure directory
    set dirs [lsort [glob -nocomplain [file join $INTROL Configure *]]]

    set processors {}
    foreach dir $dirs {
        # read the definitions
        set name [file join $dir Processor.cfg]
        if {[catch {open $name} f]} {
	    # no .cfg file
	    continue
	}
        set list [read $f]
        close $f

	# get the processor id
	set processor [file tail $dir]
	lappend processors $processor

	# set Build information, if it hasn't been changed
	foreach "name value" $list {
	    switch -exact $name {
	        clock -
	        crystal {
		    set Preferences([set processor],$name) $value
	        }
	        default {
	            PreferenceSet Preferences(Build,[set processor]$name) $value
	        }
	    }
	}
    }
    # set up initial processor
    set Configure(Build,processor) [Preference Build processor]
    if {$Configure(Build,processor) == {}} {
	set Configure(Build,processor) Unknown
    } else {
	# catch any old ccoptions, asoptions, and ldoptions

	foreach one {ccoptions asoptions ldoptions CFLAGS ASFLAGS LDFLAGS} {
	    if {[info exists Preferences(Build,$one)]} {
		Preference Build [Preference Build processor]$one \
		    $Preferences(Build,$one) 1
		unset Preferences(Build,$one)
	    }
	}
    }
    set Preferences(processors) $processors
    return $processors
}

#
# PreferenceSet - set a preference if it hasn't already been changed
#
proc code::PreferenceSet {what value} {
    variable Preferences

    if {[info exists $what]} {
	return
    }
    set $what $value
}

#
# PreferenceSetProc - set a new processor if it has changed
#
proc code::PreferenceSetProc {proc} {
    variable INTROL
    variable Configure
    variable Preferences
    variable PreferenceChangedList

    if {$proc == [Preference Build processor]} {
	return
    }

    # reset important values because changing the processor changes
    # many aspects of a project

    # initialize Build
    Build . setproc $proc
    Preference Build processor $proc
    # make sure the debugger knows
    Preference Build variant {} 1 
    Preference System sections {} 1
    Preference System memory {} 1
    Preference System vectors {} 1
    set Configure(Build,variant) Unknown
    if {![catch {DebuggerDbg processor $proc}]} {
        set Configure(Debugger,targetname) [DebuggerDbg target]
        set Configure(Debugger,target) [DebuggerDbg target -id]
        catch {Preference DebuggerInternal target $Configure(Debugger,target)}
    }

    # lose old module information
    foreach name [array names Preferences Module,*] {
	unset Preferences($name)
	catch {unset Configure($name)}
        catch {unset PreferenceChangedList($name)}
    }
    DdfLoad
    if {$proc != "Other"} {
        # Get module information
        set name [file join $INTROL Libraries Assembly [Preference Build [set proc]gen] module.cfg]
        set f [open $name]
        set list [read $f]
        close $f
        foreach "name value" $list {
	    set Preferences(Module,$name) $value
	    set Configure(Module,$name) $value
        }
    }

    # set up processor specific values
    foreach name [array names Preferences $proc,*] {
	regsub $proc $name System sysname
	set Preferences($sysname) $Preferences($name)
	set Configure($sysname) $Preferences($name)
    }

    set variants [PreferenceVariantInfo]
    # clear any existing memory map
    Preference System memory {} 1
    if {[llength $variants] == 1} {
	# only one variant, use it
	PreferenceSetVariant $variants
    } else {
        PreferenceRebuild
    }
}

#
# PreferenceRebuild - rebuild any open windows
#
proc code::PreferenceRebuild {} {
    # create a new memory window if one exists
    ConfigureEditMemory 1
    # and the library window
    ConfigureEditEnvironment 1
    # and option windows
    ConfigureEditOptionsCheck
}

#
# PreferenceProcessorMenu - create the processor menu
#
proc code::PreferenceProcessorMenu {m} {
    if {![winfo exists $m]} {
        menu $m -tearoff 0 -postcommand "code::PreferenceProcessorMenuBuild $m"
    }
    return $m
}

#
# PreferenceProcessorMenuBuild - build a menu of processors
#
proc code::PreferenceProcessorMenuBuild {m} {
    variable Configure

    $m delete 0 end
    set Configure(Build,processor) [Preference Build processor]
    # build the menu entries
    foreach proc [PreferenceProcessorInfo] {
	$m add radiobutton -label $proc \
	-variable code::Configure(Build,processor) \
	-command "code::PreferenceSetProc $proc"
    }
    return $m
}

#
# PreferenceResultMenu - create the project results menu
#
proc code::PreferenceResultMenu {m} {
    if {![winfo exists $m]} {
        menu $m -tearoff 0 -postcommand "code::PreferenceResultMenuBuild $m"
    }
    return $m
}

#
# PreferenceResultMenuBuild - build a menu of project results
#
proc code::PreferenceResultMenuBuild {m} {
    variable Configure

    $m delete 0 end
    set Configure(Build,target) [Preference Build target]
    # build the menu entries
    foreach "result name" { program Program library Library {} None } {
	$m add radiobutton -label $name \
	-value $result -variable code::Configure(Build,target) \
	-command { code::Preference Build target $code::Configure(Build,target) 1 }
    }
    return $m
}

#
# PreferenceVariantInfo - set up variant information
#
proc code::PreferenceVariantInfo {} {
    variable INTROL
    variable Preferences
    variable Configure

    # get the current processor
    set processor [Preference Build processor]

    # make sure we have the processor info
    PreferenceProcessorInfo

    if {$processor == "Other"} {
	return {}
    }
    # check for the list already read
    if {[info exists Preferences($processor,variants)]} {
	return $Preferences($processor,variants)
    }

    # read the list
    set name [file join $INTROL Libraries Assembly $Preferences(Build,[set processor]gen) variants]
    set f [open $name]
    set list [read $f]
    close $f

    set list [concat [Preference $processor myvariants] $list]
    set variants {}
    foreach variant $list {
	# only get variants for the current processor
	if {[lindex $variant 0] != $processor} {
	    continue
	}

	# get the variant name

	set var [lindex $variant 1]
	# set Build information
	set Preferences(Build,[set var]ddf) [lindex $variant 2]
	set Preferences(Build,[set var]import) [lindex $variant 3]
	set Preferences(Build,[set var]memory) [lindex $variant 4]
	set Preferences(Build,[set var]defaultmemory) [lindex $variant 5]
	set Preferences(Build,[set var]module) [lindex $variant 6]
	set Preferences(Build,[set var]sections) [lindex $variant 7]
	set Preferences(Build,[set var]variables) [lindex $variant 8]
	set Preferences(Build,[set var]vectors) [lindex $variant 9]
	set Preferences(Build,[set var]vectorsimport) [lindex $variant 10]
	lappend variants $var
    }
    # set up initial variant
    set Configure(Build,variant) [Preference Build variant]
    if {$Configure(Build,variant) == {}} {
	set Configure(Build,variant) Unknown
    }
    set Preferences(variants) $variants
    return $variants
}

#
# PreferenceVariantMenu - build a menu of processor variants
#
proc code::PreferenceVariantMenu {m} {
    if {![winfo exists $m]} {
        menu $m -tearoff 0 -postcommand "code::PreferenceVariantMenuBuild $m"
    }
    return $m
}

#
# PreferenceVariantMenuBuild - build the variant menu
#
proc code::PreferenceVariantMenuBuild {m} {
    variable Configure

    $m delete 0 end
    set proc [Preference Build processor]
    if {$proc == {}} {
	$m add command \
	    -label "You have to select a processor first."
	return
    }

    set Configure(Build,variant) [Preference Build variant]
    # build the menu entries
    foreach variant [PreferenceVariantInfo] {
	$m add radiobutton -label $variant \
	-variable code::Configure(Build,variant) \
	-command "code::PreferenceSetVariant $variant"
    }
}

#
# PreferenceSetVariant - set a new variant
#
proc code::PreferenceSetVariant {variant} {
    Preference Build variant $variant
    DdfLoad
    PreferenceRebuild
}

#
# PreferenceTargetMenu - build a menu of debugger targets
#
proc code::PreferenceTargetMenu {m} {
    if {![winfo exists $m]} {
        menu $m -tearoff 0 -postcommand "code::PreferenceTargetMenuBuild $m"
    }
    return $m
}

#
# PreferenceTargetMenuBuild - build the target menu
#
proc code::PreferenceTargetMenuBuild {m} {
    $m delete 0 end
    set proc [Preference Build processor]
    if {$proc == {}} {
	$m add command \
	    -label "You have to select a processor first."
	return
    }

    # build the menu entries
    # get the list of targets
    DebuggerDbg processor [Preference Build processor]
    set list [DebuggerDbg target -all]
    if {$list == {}} {
        tk_messageBox \
	    -icon info \
	    -message "No targets for the $proc are available." \
	    -type ok
	    return
    }

    foreach var $list {
	set id [lindex $var 0]
	set desc [lindex $var 1]
	$m add radiobutton \
	    -label $desc \
	    -variable code::Configure(Debugger,target) \
	    -value $id \
	    -command "code::Preference DebuggerInternal target $id; \
		set code::Configure(Debugger,targetname) \"$desc\""
    }
}

#
# PreferenceOpen - open a new project file
#
proc code::PreferenceOpen {{file {}}} {
    variable PreferenceFile

    if {$file != {} && $PreferenceFile == $file} {
	#already open
	return
    }

    CODEclose
    if {![PreferencesSave 0 1]} {
	return
    }

    if {$file == {}} {
	set initialdir [Preference Configure dirfileopen]
	if {$initialdir == {} || ![file isdirectory $initialdir]} {
	    set initialdir [pwd]
	}
        set file [tk_getOpenFile -title "Select a CODE project file" \
	    -initialdir $initialdir \
            -defaultextension .code \
            -filetypes {
                {"CODE Projects" {.code .cod}}
            } ]
        if {$file == {}} {
	    return
        }
	Preference Configure dirfileopen [file dirname $file]
    }

    Preferences $file
    # get any initial documents
    CODEstartup
}

#
# Preferences - try to find saved preferences
#
proc code::Preferences {{filename {}} {import 0} {new 0} {subproject 0}
    {baseproject {"$INTROL" tcltk code CODE.code}}} {
    global tcl_platform env
    variable INTROL
    variable PreferenceFile
    variable PreferenceImport
    variable PreferenceChanged
    variable PreferenceChangedList
    variable Preferences
    variable Configure
    variable PreferenceDelayedCallbacks
    variable PreferenceDelayCallbacks
    variable Build
    variable UI

    # see if we are running with a user interface
    if {[info commands .] == "."} {
	set UI 1
    } else {
	set UI 0
    }
    # find code files to try
    set filetoopen {}
    if {$filename != {}} {
	# got an explicit file name
	regsub -all {[\\]} $filename / filename
	# give a default .code extension, if none present
	if {!$import && [file extension $filename] == {}} {
	    set filename $filename.code
	}
	set extension [string tolower [file extension $filename]]
	if {!$import && [string compare $extension ".code"] != 0 && \
	    [string compare $extension ".cod"] != 0} {
		# not a project file, open later
		set filetoopen $filename
		set filename {}
	}
	if {$filename != {} && ![file exists $filename]} {
	    # file not found
	    if {!$UI} {
		puts stderr "BUILD: cannot find project file $filename"
		exit 1
	    }
	    if {$subproject} {
		error "Cannot find project file $filename"
	    }
            tk_messageBox \
		-icon error \
	        -message "Cannot find project file $filename" \
	        -type ok
	    set filename {}
	} else {
	    set codefile $filename
	}
    } elseif {!$UI} {
	puts stderr "BUILD: no project name specified"
	exit 1
    }

    if {$filename == {}} {
        set basefile [subst -nobackslashes -nocommands $baseproject]
	set codefile [eval file join $basefile]
	set PreferenceImport $baseproject
    }

    if {$codefile != {}} {
        # read the project file

	if {[catch {open $codefile r} fd]} {
	    error $fd
        }

        set pref [read $fd]
        close $fd
    } 

    if {!$import} {
	# clear only the file name for now
        set PreferenceFile {}

	# clear any existing preferences
        set PreferenceChanged 0
        catch {unset PreferenceChangedList}
        catch {unset Preferences}
	set PreferenceDelayedCallbacks {} 
	set PreferenceDelayCallbacks 1
    }
    
    # change to the project directory, if a project file
    if {!$import && $filename != {}} {
	# remember the project directory
        cd [file dirname $filename]
    }

    foreach "type value" $pref {
	if {$type == "import"} {
	    # import from another file

	    # import from a project file
	    set joinedvalue [eval file join $value]
	    set sub [subst -nobackslashes -nocommands $joinedvalue]
	    regsub -all {[\\]} $sub / sub
	    Preferences $sub 1
	    # remember the latest import
	    set PreferenceImport $value
	    continue
	}
	if {$import || $filename == {}} {
	    # don't remember imported values
	    foreach "elt val" $value {
	        set Preferences($type,$elt) $val
	    }
	} else {
	    # not imported, keep track of changes
	    foreach "elt val" $value {
		Preference $type $elt $val 1
	    }
	}
    }

    if {!$import} {
        # read the coderc file to override global definitions
        if {$tcl_platform(platform) == "windows"} {
            if {[file exists "~/coderc"]} {
                Preferences ~/coderc 1
            }
        } else {
            if {[file exists "~/.coderc"]} {
                Preferences ~/.coderc 1
            }
        }
    }

    if {$UI && $import == 0} {
        # create any defined fonts
	# get an acceptable default size, default tool relief
	set tool [Preference General flattools]
        if {$tcl_platform(platform) == "windows"} {
	    if {$tool == {}} {
		set Preferences(General,flattools) 1
		set Configure(General,flattools) 1
	    }
	    set defaultsize 10
	} else {
	    if {$tool == {}} {
		set Preferences(General,flattools) 0
		set Configure(General,flattools) 0
	    }
	    set defaultsize 12
	}
        foreach temp [array names Preferences *,font*] {
	    regexp {(.*),(.*)} $temp all type element
	    set font $Preferences($temp)
	    set changed 0
	    set name [lindex $font 0]
	    set size [lindex $font 1]
	    if {$size == {}} {
		set changed 1
	        set size $defaultsize
	    }
	    set weight [lindex $font 2]
	    if {$weight == {}} {
		set changed 1
	        set weight normal
	    }
	    set slant [lindex $font 3]
	    if {$slant == {}} {
		set changed 1
	        set slant roman
	    }
	    if {$changed} {
		# update the font entry
		set changed [info exists PreferenceChangedList($type,$element)]
		Preference $type $element [list $name $size $weight $slant]
		if {!$changed} {
		    # don't write out if just filling in defaults
                    catch {unset PreferenceChangedList($type,$element)}
		}
	    }
        }
    }
    set PreferenceChanged 0
    if {!$import} {
	if {$filename != {}} {
	    # named file
	    set PreferenceFile [file tail $filename]
	    # change to the project directory
	    set dir [file dirname $filename]
	    Preference Directory lastdir [pwd]
            # remember the project directory
            set Build(PROJECT) [pwd]
	    set Build(OBJECT) [BuildRealFileName [Preference Build OBJECT]]
	    if {$Build(OBJECT) == {}} {
		# use the project directory by default
		set Build(OBJECT) $Build(PROJECT)
	    }
	} else {
	    # should we restore the last directory?
	    if {!$new && $filetoopen == {} && \
	      [Preference Directory gotolastdir]} {
		set last [Preference Directory lastdir]
		if {$last != {}} {
		    # try to go back there
		    catch {cd $last}
		}
	    }
	}

	# update any Configure variables
        foreach name [array names Preferences *,*] {
	    if {[info exists Configure($name)]} {
	        set Configure($name) $Preferences($name)
	    }
        }
        # get processor information
        PreferenceProcessorInfo
        set Configure(Build,variant) Unknown
        set Configure(Debugger,target) Unknown
        set proc [Preference Build processor]

        if {$proc != {}} {
	    # have a processor, check variant and target
	    PreferenceVariantInfo
            set Configure(Debugger,target) [Preference DebuggerInternal target]
            if {$UI && $Configure(Debugger,target) != {}} {
	        DebuggerDbg processor $proc
	        catch {DebuggerDbg target [Preference DebuggerInternal target]}
	        set Configure(Debugger,targetname) [DebuggerDbg target]
            }
    
            # Get module information that hasn't changed
            if {$proc != "Other"} {
                set name [file join $INTROL Libraries Assembly [Preference Build [set proc]gen] module.cfg]
                set f [open $name]
                set list [read $f]
                close $f
                foreach "name value" $list {
	            if {![info exists Preferences(Module,$name)]} {
	                set Preferences(Module,$name) $value
	                set Configure(Module,$name) $value
	            }
	        }
	    }

            # set up processor specific values
            foreach name [array names Preferences $proc,*] {
	        regsub $proc $name System sysname
                set Preferences($sysname) $Preferences($name)
                set Configure($sysname) $Preferences($name)
            }
	    DdfLoad 0
        }

	PreferenceDoDelayedCallbacks

        # initialize Build
        Build . setproc $proc

	# rebuild or close any open windows
	if {$UI && !$subproject} {
	    PreferenceRebuild
	}

	if {!$UI || $subproject} {
	    # don't display info
            set PreferenceChanged 0
	    return {}
	}

        if {$filetoopen == {} && [Preference General showpreferences] != {}} {
	    PreferenceEdit
        }
    set PreferenceChanged 0
    return $filetoopen
    }
}

#
# tk_chooseFont - chose a font
#
proc tk_chooseFont {args} {
    set w .__tk_font
    upvar #0 $w data

    tkFontDialog_Config $w $args
    tkFontDialog_InitValues $w

    if ![winfo exists $w] {
	toplevel $w -class tkFontDialog
	tkFontDialog_BuildDialog $w
    }
    wm transient $w $data(-parent)

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	- [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	- [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w
    wm title $w $data(-title)

    set oldFocus [focus]
    set oldGrab [grab current $w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $w
    focus $data(okBtn)

    tkwait variable [set w](selectFont)
    set font $data(selectFont)
    catch {focus $oldFocus}
    grab release $w
    destroy $w
    unset data
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    return $font
}

# tkFontDialog_InitValues --
#
#	Get called during initialization
#
proc tkFontDialog_InitValues {w} {
    upvar #0 $w data
    global tcl_platform

    # Set the initial font, specified by -initialfont, or the
    # first font in the families list
    set data(initialFont)  [lindex $data(-initialfont) 0]
    set data(initialSize)  [lindex $data(-initialfont) 1]
    if {[catch {expr int($data(initialSize))}]} {
        if {$tcl_platform(platform) == "windows"} {
	    set data(initialSize) 14
	} else {
	    set data(initialSize) 12
	}
        set style 1
    } else {
        set style 2
    }
    set bold 0
    set italic 0
    foreach arg [lrange $data(-initialfont) $style end] {
	switch -exact [string tolower $arg] {
	    bold { set bold 1 }
	    italic { set italic 1 }
	    roman { set italic 0 }
	    normal { set bold 0 }
	    default { error "bad font style" }
	}
    }
    if {$bold && $italic} {
	set data(initialStyle) "Bold Italic"
    } elseif {$bold} {
	set data(initialStyle) "Bold"
    } elseif {$italic} {
	set data(initialStyle) "Bold"
    } else {
        set data(initialStyle) Regular
    }
    set data(finalFont) $data(initialFont)
    set data(finalStyle) $data(initialStyle)
    set data(finalSize) $data(initialSize)
}

# tkFontDialog_Config  --
#
#	Parses the command line arguments to tk_chooseFont
#
proc tkFontDialog_Config {w argList} {
    upvar #0 $w data

    # 1: the configuration specs
    #
    set specs {
	{-initialfont "" "" ""}
	{-parent "" "" "."}
	{-title "" "" "Font"}
    }

    # 2: parse the arguments
    #
    tclParseConfigSpec $w $specs "" $argList

    if ![string compare $data(-title) ""] {
	set data(-title) " "
    }
    if ![winfo exists $data(-parent)] {
	error "bad window path name \"$data(-parent)\""
    }

    set fonts [lsort [font families -displayof $data(-parent)]]
    set data(families) {}
    set length 0
    foreach n $fonts {
	if {[string length $n] > $length} {
	    set length [string length $n]
	}
        lappend data(families) [tkFontToupper $n]
    }
    set data(widest) $length
    if ![string compare $data(-initialfont) ""] {
	set data(-initialfont) [lindex $data(families) 0]
    }
}

#
# tkFontToupper - capitalize the first letter of each word in string
#
proc tkFontToupper {s} {
    set res ""
    foreach word $s {
	lappend res \
	    [string toupper [string index $word 0]][string range $word 1 end]
    }
    return $res
}

# tkFontDialog_BuildDialog --
#
#	Build the dialog.
#
proc tkFontDialog_BuildDialog {w} {
    upvar #0 $w data

    set f [frame $w.buttons]
    set data(okBtn) [button $f.ok -text Ok -width 6 \
        -command "tkFontDialog_OkCmd $w"]
    set data(cancelBtn) [button $f.cancel -text Cancel -width 6 \
        -command "tkFontDialog_CancelCmd $w"]
    grid $data(okBtn) -in $f -row 0 -column 0 
    grid $data(cancelBtn) -in $f -row 1 -column 0 
    grid $f -in $w -row 1 -column 3 -rowspan 3 -sticky n -padx 5
    label $w.font -text Font
    label $w.style -text "Font style"
    label $w.size -text "Size"
    grid $w.font -in $w -row 0 -column 0 -sticky w -padx 5 
    grid $w.style -in $w -row 0 -column 1 -sticky w -padx 5
    grid $w.size -in $w -row 0 -column 2 -sticky w -padx 5 
    entry $w.fontentry -textvariable [set w](initialFont) -width 1
    entry $w.styleentry -textvariable [set w](initialStyle) -width 1
    entry $w.sizeentry -textvariable [set w](initialSize) -width 1
    grid $w.fontentry -in $w -row 1 -column 0 -sticky ew -padx 5
    grid $w.styleentry -in $w -row 1 -column 1 -sticky ew -padx 5
    grid $w.sizeentry -in $w -row 1 -column 2 -sticky ew -padx 5
    set ff [frame $w.fontframe]
    set sf [frame $w.styleframe]
    set zf [frame $w.sizeframe]
    grid $ff -in $w -row 2 -column 0 -sticky ew -padx 5
    grid $sf -in $w -row 2 -column 1 -sticky ew -padx 5
    grid $zf -in $w -row 2 -column 2 -sticky ew -padx 5
    listbox $ff.lb -height 8 -width $data(widest) -yscrollcommand "$ff.sb set" \
	-exportselection 0
    bind $ff.lb <ButtonRelease-1> "set [set w](finalFont) \
	\[$ff.lb get \[$ff.lb curselection]]; \
	set [set w](initialFont) \[set [set w](finalFont)]; \
        tkFontSetPreview $w"
    eval $ff.lb insert 0 $data(families)
    set index 0
    foreach n [$ff.lb get 0 end] {
	if {[string compare $n $data(finalFont)] == 0} {
	    $ff.lb selection set $index
	    $ff.lb see $index
	}
	incr index
    }
    scrollbar $ff.sb -command "$ff.lb yview" -orient vertical
    grid $ff.lb -in $ff -row 0 -column 0 -sticky nsew
    grid $ff.sb -in $ff -row 0 -column 1 -sticky ns
    listbox $sf.lb -height 8 -width 12 -yscrollcommand "$sf.sb set" \
	-exportselection 0
    bind $sf.lb <ButtonRelease-1> "set [set w](finalStyle) \
	\[$sf.lb get \[$sf.lb curselection]]; \
	set [set w](initialStyle) \[set [set w](finalStyle)]; \
        tkFontSetPreview $w"
    $sf.lb insert 0 Regular Italic Bold "Bold Italic"
    set index 0
    foreach n [$sf.lb get 0 end] {
	if {[string compare $n $data(finalStyle)] == 0} {
	    $sf.lb selection set $index
	    $sf.lb see $index
	}
	incr index
    }
    scrollbar $sf.sb -command "$sf.lb yview" -orient vertical
    grid $sf.lb -in $sf -row 0 -column 0 -sticky nsew
    grid $sf.sb -in $sf -row 0 -column 1 -sticky ns
    listbox $zf.lb -height 8 -width 3 -yscrollcommand "$zf.sb set" \
	-exportselection 0
    bind $zf.lb <ButtonRelease-1> "set [set w](finalSize) \
	\[$zf.lb get \[$zf.lb curselection]]; \
	set [set w](initialSize) \[set [set w](finalSize)]; \
        tkFontSetPreview $w"
    $zf.lb insert 0 8 9 10 11 12 14 16 18 20 22 24 26 28 36 48 72
    set index 0
    foreach n [$zf.lb get 0 end] {
	if {[string compare $n $data(finalSize)] == 0} {
	    $zf.lb selection set $index
	    $zf.lb see $index
	}
	incr index
    }
    scrollbar $zf.sb -command "$zf.lb yview" -orient vertical
    grid $zf.lb -in $zf -row 0 -column 0 -sticky nsew
    grid $zf.sb -in $zf -row 0 -column 1 -sticky ns

    set f [frame $w.pre -bd 2 -relief groove]
    grid $f -in $w -row 3 -column 0 -columnspan 4 -padx 5 -pady 5 -sticky ew
    label $f.l -text Preview
    set data(preview) [label $f.p -text "ABCDEFGHIJKLMNOPQRSTUVWXYZ\nabcdefghijklmnopqrstuvwxyz\n0123456789" -width 0]
    #set data(preview) [label $f.p -textvariable [set w](finalFont) -width 0]
    tkFontSetPreview $w

    grid $f.l -in $f -row 0 -column 0 -sticky w
    grid $f.p -in $f -row 1 -column 0 -sticky ew
    grid columnconfigure $f 0 -weight 1 -minsize 200
    grid rowconfigure $f 1 -minsize 70 -weight 0

    wm protocol $w WM_DELETE_WINDOW "tkFontDialog_CancelCmd $w"
}

#
# tkFontSetPreview - set the preview label
#
proc tkFontSetPreview {w} {
    upvar #0 $w data

    $data(preview) config -font [tkFontGetSpec $w]
}

#
# tkFontGetSpec - get the tk front specifier
#
proc tkFontGetSpec {w} {
    upvar #0 $w data

    switch -exact $data(finalStyle) {
	Regular { 
	    set slant roman
	    set weight normal
	}
	Italic { 
	    set slant italic
	    set weight normal
	}
	Bold { 
	    set slant roman
	    set weight bold
	}
	"Bold Italic" { 
	    set slant italic
	    set weight bold
	}
    }
    return [list $data(finalFont) $data(finalSize) $slant $weight]
}

# user hits OK button
#
proc tkFontDialog_OkCmd {w} {
    upvar #0 $w data

    set data(selectFont) [tkFontGetSpec $w]
}

# user hits Cancel button
#
proc tkFontDialog_CancelCmd {w} {
    upvar #0 $w data

    set data(selectFont) ""
}

#
# PreferencesSaveAs - save the current configuration to a specific file
#
proc code::PreferencesSaveAs {} {
    PreferencesSave 1
}

#
# PreferencesSave - save the current preferences if they have changed
#
proc code::PreferencesSave {{saveas 0} {check 0}} {
    variable Preferences
    variable PreferenceFile
    variable PreferenceImport
    variable PreferenceChanged
    variable PreferenceChangedList
    variable installed
    variable Build

    # save the global preferences
    PreferenceSaveUser

    # check for changed project preferences
    if {!$saveas && !$PreferenceChanged} {
	return 1
    }

    if {$saveas || $PreferenceFile == {}} {
	# clear the PreferenceFile for saveas
	set filename {}
	# no preference file exists, use project name, if any
	if {!$saveas} {
	    set filename [Preference Build name]
	    if {$filename != {}} {
	        set ext [file extension $filename]
	        if {$ext != ".code" && $ext != ".cod"} {
	            set filename $filename.code
	        }
	    }
	    if {$filename == {} && [Preference Build processor] == {}} {
	        # not named and no processor, don't bother saving
		return 1
	    }
	}
	while {$filename == {}} {
	    # try to get a default name
            set newname [file tail [pwd]]
	    if {[file exists $newname.code] || [file exists $newname.cod]} {
		set newname {}
	    }
	    set filename [tk_getSaveFile -title "Save CODE project as" \
	        -initialdir [pwd] \
		-defaultextension .code \
		-initialfile $newname \
                -filetypes {
	            {"CODE Projects" {.code .cod}}
	        } ]
	    if {$filename == ""} {
		if {$saveas} {
		    return 1
		} 
	        set result [tk_messageBox \
	            -icon question \
	            -message "Discard changes to the current project?" \
	            -default no \
	            -type yesnocancel]
	        if {$result == "cancel"} {
		    return 0
	        }
	        if {$result == "yes"} {
		    return 1
	        }
	    }
	}
    } else {
	if {$check} {
	    # ask if we should save the project file
	    set result [tk_messageBox \
	        -icon question \
	        -message "Save changes to $PreferenceFile?" \
	        -default yes \
	        -type yesnocancel]
	    if {$result == "cancel"} {
	        return 0
	    }
	    if {$result == "no"} {
		return 1
	    }
	}

	set filename $PreferenceFile
    }

    if {$saveas && [Preference System definestartup]} {
	# remove the old project's startup code from the project
        set Build(NAME) [Preference Build name]
	set startname [BuildSubstitute [Preference System startfilename]]
        Configure . deletefile $startname
    }

    if {[file dirname $filename] == "."} {
	# this is the current project file
	if {$Build(PROJECT) == {}} {
            set Build(PROJECT) [pwd]
	}
	set filename [file join $Build(PROJECT) $filename]
    }
    while {[catch {open $filename w} fd]} {
	 if {!$code::installed} {
	     # don't complain
	     return 1
	 }
         set res [tk_messageBox \
		-icon error \
	        -message "Cannot open $filename for writing: $fd" \
	        -type abortretryignore]
	 if {$res == "abort"} {
	     # failed
             return 0
	 } elseif {$res == "ignore"} {
	     # just continue
	     return 1
	 }
    }

    if {[catch {puts $fd "import [list $PreferenceImport]"} msg]} {
        tk_messageBox \
   	    -icon error \
	    -message "Error writing $filename: $msg" \
	    -type ok
        return 0
    }

    if {$saveas && [Preference Build name] == {}} {
	# get the project name
        Preference Build name [file rootname [file tail $filename]]
    }
    set dir [file dirname $filename]
    if {$dir != $Build(PROJECT)} {
	# project directory has changed
        foreach file [Preference Build files] {
	    # substitute names for new project directory
	    set file  [lindex $file 0]
	    set orig [BuildRealFileName $file]
	    ConfigureRename $orig $orig $dir
        }
    }

    # set new project directory
    set Build(PROJECT) $dir
    set changedlist [lsort [array names PreferenceChangedList]]
    set type ""
    append changedlist " end,empty"
    foreach change "$changedlist" {
	regexp {(.*),(.*)} $change all thistype element
	if {$thistype != "end" && \
	  ![info exists Preferences($thistype,$element)]} {
	    # this has been removed for backward compatability (e.g. asoptions)
	    continue
	}
	if {$thistype != $type} {
	    # changing preference category
	    if {$type != ""} {
		# save old type
                if {[catch {puts $fd [list $type $value]} msg]} {
                    tk_messageBox \
   	                -icon error \
	                -message "Error writing $filename: $msg" \
	                -type ok
                    return 0
                }
	    }
	    if {$thistype == "end"} {
		# end of list
		break
	    }
	    set value {}
	    set type $thistype
	}
	lappend value $element $Preferences($type,$element)
    }
    if {[catch {close $fd} msg]} {
        tk_messageBox \
   	    -icon error \
	    -message "Error closing $filename: $msg" \
	    -type ok
        return 0
    }

    # remember the preference file name
    set PreferenceFile [file tail $filename]
    set PreferenceChanged 0
    DirectoryRefresh
    return 1
}

#
# PreferenceSaveUser - save the current preferences for next time
#
proc code::PreferenceSaveUser {} {
    global tcl_platform
    variable PreferenceImport
    variable Preferences

    foreach cat [Preference Configure global] {
        append result "\n$cat \{\n"
	foreach name [array names Preferences $cat,*] {
	    regexp "$cat,(.*)" $name all elt
	    append result "    $elt [list [Preference $cat $elt]]\n"
	}
	append result "\}\n"
    }

    if {$tcl_platform(platform) == "windows"} {
        # use a file in the user's
	# home directory
	set filename "~/coderc"
        if {[catch {open $filename w} fd]} {
            set res [tk_messageBox \
		-icon error \
	        -message "Cannot save user information: $fd" \
	        -type ok]
	    return
        }

        # write the file
	puts -nonewline $fd $result
	close $fd
	return
    } else {
	# save to ~/.coderc
	set filename "~/.coderc"
        if {[catch {open $filename w} fd]} {
            set res [tk_messageBox \
	    	-icon error \
	        -message "Cannot save user information: $fd" \
	        -type ok]
	    return
	}
	# write the file
	puts -nonewline $fd $result
	close $fd
    }
}

#
# Preference - query or set a CODE preference
#
proc code::Preference {category option {newvalue {}} {force 0}} {
    global code::Preferences code::Configure
    global code::PreferenceChanged code::PreferenceChangedList
    global code::PreferenceCallback
    global code::PreferenceDelayCallbacks
    global code::PreferenceDelayedCallbacks

    if {!$force && $newvalue == {}} {
	# look up a value
        if {![info exists Preferences($category,$option)]} {
	    return {}
        }
        return $Preferences($category,$option)
    }

    if {[info exists Preferences($category,$option)] && \
      $Preferences($category,$option) == $newvalue} {
	# unchanged
	return $newvalue
    }

    # set a new preference value
    set Preferences($category,$option) $newvalue
    set Configure($category,$option) $newvalue

    if {[info exists Preferences(Build,processor)]} {
        set proc $Preferences(Build,processor)
        if {$category == "System" && [info exists Preferences($proc,$option)]} {
	    # processor specific options
            set PreferenceChangedList($proc,$option) 1
            set Preferences($proc,$option) $newvalue
        }
    }

    if {$PreferenceDelayCallbacks} {
	# delay callbacks until they have all occured
        foreach one [array names PreferenceCallback $category,*] {
	    if {[lsearch -exact $PreferenceDelayedCallbacks $PreferenceCallback($one)] == -1} {
	        lappend PreferenceDelayedCallbacks $PreferenceCallback($one)
	    }
        }
    } else {
	# do the callback immediately
        foreach one [array names PreferenceCallback $category,*] {
	    if {[catch {eval $PreferenceCallback($one)}]} {
		 unset PreferenceCallback($one)
	    }
        }
    }

    if {[lsearch [Preference Configure global] $category] != -1} {
	# changing a global preference
	return $newvalue
    }
    set PreferenceChanged 1
    set PreferenceChangedList($category,$option) 1
    return $newvalue
}

#
# PreferenceDoDelayedCallbacks - do delayed callbacks
#
proc code::PreferenceDoDelayedCallbacks {} {
    global code::PreferenceDelayCallbacks
    global code::PreferenceDelayedCallbacks

    foreach one $PreferenceDelayedCallbacks {
        catch {eval $one}
    }
    set PreferenceDelayCallbacks 0
    set PreferenceDelayedCallbacks {}
}

#
# PreferenceSearch - return a list of matching preference names
#
proc code::PreferenceSearch {category option} {
    global code::Preferences

    set names [array names Preferences $category,$option]
    if {$names == {}} {
	return {}
    }
    set list {}
    foreach name $names {
	regexp (.*),(.*) $name all category option
	lappend list $category
	lappend list $option
    }
    return $list
}

#
# PreferenceWhenChanged - set up a callback for changed preferences
#
proc code::PreferenceWhenChanged {category tag cmd} {
    global code::PreferenceCallback

    if {[info exists PreferenceCallback($category,$tag)]} {
	append PreferenceCallback($category,$tag) "; $cmd"
    } else {
	set PreferenceCallback($category,$tag) $cmd
    }
}

#
# PreferenceCallbackForget - remove a set of preference callbacks
#
proc code::PreferenceCallbackForget {tag} {
    global code::PreferenceCallback

    foreach one [array names PreferenceCallback *,$tag] {
	unset PreferenceCallback($one)
    }
}

#
# setFileTypes - set up the FileTypes database
#
proc code::setFileTypes {} {
    global code_library
    variable Preferences
    variable FileIcons

    if {[info exists FileIcons(iconheight)]} {
	return
    }
    set FileIcons(iconheight) 0
    set FileIcons(iconwidth) 0
    
    foreach type [array names Preferences FileTypes,*] {
	set list $Preferences($type)
	set icon [lindex $list 0]
	if {![info exists FileIcons($icon)]} {
	    regsub \.icon $icon .gif giffile
	    set giffile [file join $code_library icons $giffile]
	    if {[file exists $giffile]} {
                set img [image create photo -file $giffile]
	    } elseif {$type == "FileTypes,directory"} {
                set img [image create photo -data {
    R0lGODdhEAAQAPIAAAAAAHh4eLi4uPj4APj4+P///wAAAAAAACwAAAAAEAAQAAADPVi63P4w
    LkKCtTTnUsXwQqBtAfh910UU4ugGAEucpgnLNY3Gop7folwNOBOeiEYQ0acDpp6pGAFArVqt
    hQQAO///
	        }]

	    } else {
		# use a generic file icon
	        set giffile [file join $code_library icons file.gif]
	        if {[file exists $giffile]} {
                    set img [image create photo -file $giffile]
	        } else {
	            set img [image create photo -data {
    R0lGODdhEAAQAPIAAAAAAHh4eLi4uPj4+P///wAAAAAAAAAAACwAAAAAEAAQAAADPkixzPOD
    yADrWE8qC8WN0+BZAmBq1GMOqwigXFXCrGk/cxjjr27fLtout6n9eMIYMTXsFZsogXRKJf6u
    P0kCADv/
	            }]
	        }
	    }
	    set width [image width $img]
	    set height [image height $img]
	    if {$width > $FileIcons(iconwidth)} {
		set FileIcons(iconwidth) $width
	    }
	    if {$height > $FileIcons(iconheight)} {
		set FileIcons(iconheight) $height
	    }
	    set FileIcons($icon) $img
	}
    }
}
