#
#	Search - Find, Replace, Goto in a text widget
#

array set code::Search {
    file {}
    canreplace 0
    count 0
}

#
# Search - open the search window
#
proc code::Search {w t type title {replace 0}} {
    variable Search

    # Things that change each time Search is entered
    set Search(owner) $w
    # set the text widget
    set Search(text) $t

    # if a selection exists, use it
    if {   ![catch {winfo parent [selection own -displayof $t]} tmp]
	&& $tmp == $t
	&& ![catch {selection get -displayof $t} sel]} {
	if {$type == "Goto"} {
            set Search(goto) $sel
	} elseif {$type == "Find"} {
            set Search(pattern) $sel
	} elseif {$type == "Replace"} {
            set Search(rpattern) $sel
	}
    } 

    set top .search
    set geometry [Preference Geometry search]
    if {[winfo exists $top]} {
        if {$geometry != {}} {
	    # the window has had its geometry change saved
	    setGeometry $top $geometry
	}
        SearchUpdate $w $t $title
	if {!$replace} {
	    # can't use replace
	    set Search(canreplace) 0
	    Notebook:pageconfig $top.nb Replace -state disabled
	    if {$type == "Replace"} {
	        set type Find
	    }
	} else {
	    set Search(canreplace) 1
	    Notebook:pageconfig $top.nb Replace -state normal
	}
        Notebook:raise $top.nb $type
	switch $type {
	    Find {
                focus $Search(findcombo)
		$Search(findcombo) selection range 0 end
	    }
	    Replace {
                focus $Search(replacefindcombo)
		$Search(replacefindcombo) selection range 0 end
	    }
	    Goto {
                focus $Search(gotocombo)
		$Search(gotocombo) selection range 0 end
	    }
	}
        wm deiconify $top
	raise $top
	wm transient $top $w
        # this code seems to get around a bug in Tk (wm geometry for
        # a hidden window)
        global tcl_platform
        if {$tcl_platform(platform) == "windows" && $geometry != {}} {
            setGeometry $top $geometry
        }
	return
    }

    # Set initial options
    set Search(direction) Down
    set Search(matchcase) 0
    set Search(matchpattern) 0
    if {![info exists Search(pattern)]} {
	set Search(pattern) {}
    }
    if {![info exists Search(rpattern)]} {
	set Search(rpattern) {}
    }
    if {![info exists Search(goto)]} {
	set Search(goto) {}
    }
    set Search(gotowhat) Line

    # create the Search window
    CODENotebook Search $top search
    wm transient $top $w
    Notebook:config $top.nb -height 135 -width 420

    SearchUpdate $w $t $title
    if {!$replace} {
        # can't use replace
	set Search(canreplace) 0
	Notebook:pageconfig $top.nb Replace -state disabled
	if {$type == "Replace"} {
	    set type Find
	}
    } else {
	set Search(canreplace) 1
	Notebook:pageconfig $top.nb Replace -state normal
    }
    Notebook:raise $top.nb $type
    switch $type {
        Find {
            focus $Search(findcombo)
        }
        Replace {
            focus $Search(replacefindcombo)
        }
        Goto {
            focus $Search(gotocombo)
        }
    }
    bind $top <F1> "code::OpenHelp Applications Search.html"
}

#
# SearchMark - set a mark in a text widget
#
proc code::SearchMark {w t {file {}}} {
    variable Search
    variable SearchMarks

    if {$w == "."} {
        set box .mark
    } else {
	set box $w.mark
    }

    # check for a current selection
    set Search(mark) {}
    if {![catch {selection get} sel]} {
	set Search(mark) $sel
    }
    toplevel $box 
    wm transient $box $w
    wm protocol $box WM_DELETE_WINDOW "set code::Search(mark) {}"
    set f [frame $box.buttons]
    set okBtn [button $f.ok -text Ok -width 6]
    set cancelBtn [button $f.cancel -text Cancel -width 6 \
        -command "set code::Search(mark) {}"]
    grid $okBtn -in $f -row 0 -column 0 
    grid $cancelBtn -in $f -row 0 -column 1 
    grid $f -in $box -row 2 -column 0 -sticky ew -padx 5 -pady 5
    set c [combobox $box.combo \
        -maxheight 0 \
	-width 27 \
	-editable true \
        -state normal]

    $c configure -value $Search(mark)
    $c configure -command "set code::Search(mark) \[$c get]; #"
    $okBtn configure -command "set code::Search(mark) \[$c get]"

    set marks [$t mark names]
    $c list delete 0 end
    foreach mark $marks {
        if {$mark == "current" || $mark == "anchor"} {
	    continue
        }
	    $c list insert end $mark
    }

    grid $c -in $box -row 1 -column 0 -sticky ew -padx 5 -pady 5
    set l [label $box.l -text "Enter or select the mark name"]
    grid $l -in $box -row 0 -column 0 -sticky ew -padx 5 -pady 5

    placewindow $box widget $w
    wm title $box Mark
    bind $box <F1> "code::OpenHelp Applications SearchMark.html"

    set oldFocus [focus]
    set oldGrab [grab current $box]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $box
    focus $c

    tkwait variable code::Search(mark)
    catch {focus $oldFocus}
    grab release $box
    destroy $box
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    set mark $Search(mark)
    if {$mark == {}} {
	return {}
    }
    if {$mark == "current" || $mark == "insert" || $mark == "anchor"} {
	# can't set these
	tk_messageBox -icon error \
	    -message "'$mark' is a reserved bookmark name" -type ok
	return {}
    }
    # set a mark
    $t mark set $mark insert
    if {$file != {}} {
	# remember marks for this file
	set SearchMarks($file) [$t dump -mark 1.0 end]
    }
    SearchUpdate $w $t $file
    return "$mark [$t index insert]"
}

#
# SearchUpdate - set information in the Search widget
#
proc code::SearchUpdate {w t {title {}} {file {}} {replace 0} {force 0}} {
    variable Search
    variable SearchMarks

    if {$file != {}} {
        # named file, get any pre-existing marks
        if {[file dirname $file] == "."} {
            # use the full name
            set file [file join [pwd] $file]
        }
        if {[info exists SearchMarks($file)]} {
            foreach "mark name index" $SearchMarks($file) {
                if {   $name == "index"
		    || $name == "current"
	            || $name == "anchor"} {
                    continue
	        }
	        $t mark set $name $index
	    }
	}
    }

    set top .search
    if {[winfo exists $top] && ($w == $Search(owner) || $force)} {
	# the search window exists and this toplevel owns it
	if {$t == {}} {
	    # window being closed
	    wm withdraw $top
	    return
	}
	set Search(text) $t
	if {!$replace} {
	    # can't use replace
	    set Search(canreplace) 0
	    Notebook:pageconfig $top.nb Replace -state disabled
	} else {
	    set Search(canreplace) 1
	    Notebook:pageconfig $top.nb Replace -state normal
	}
	set marks [$t mark names]
	set c $Search(gotocombo)
	$c list delete 0 end
	set old [$c get]
	set found 0
	foreach mark $marks {
	    if {$mark == "current" || $mark == "anchor"} {
		continue
	    }
	    $c list insert end $mark
	    if {$mark == $old} {
		set found 1
	    }
	}
	if {!$found && $Search(gotowhat) == "Mark"} {
	    $c configure -editable true
	    $c delete 0 end
	    $c configure -editable false
	}
	if {$title != {}} {
	    wm title $top "Search: [file nativename $title]"
	}
    }
}

#
# SearchNext - find the next thing
#
proc code::SearchNext {combo args} {
    variable Search

    # get the search pattern
    set pattern [$combo get]
    if {$pattern == {}} {
	return
    }
    if {   ![info exists Search(findcombo)] \
	|| ![info exists Search(replacefindcombo)]} {
	# too early
	return
    }

    set list [$combo list get 0 end]
    if {[lsearch -exact $list $pattern] == -1} {
	# keep combo boxes for find and replace in sync
	foreach combo {findcombo replacefindcombo} {
	    $Search($combo) delete 0 end
	    $Search($combo) insert 0 $pattern
	    $Search($combo) list insert 0 $pattern
	}
    }

    set t $Search(text)
    set current [$t index insert]
    set direction $Search(direction)
    switch $direction {
	All -
	Down {
	    set dir "-forwards"
	    set end end
	}
	Up {
	    set dir "-backwards"
	    set end 1.0
	}
    }
    if {$Search(matchpattern)} {
	set type "-regexp"
    } elseif {$Search(matchcase)} {
	set type -exact
    } else {
	set type -nocase
    }
    while {1} {
        set found [$t search -count code::Search(count) $type $dir $pattern $current $end]
        if {$found == $current} {
	    # start the search one character over
            set found [$t search -count code::Search(count) $type $dir $pattern "$current + 1c" $end]
        }
        if {$found == {} && $direction == "All"} {
	    # not found, search from beginning
            set found [$t search -count code::Search(count) $type -forward $pattern 1.0 $current]
        }
        if {$found == {}} {
	    # not found
            switch $direction {
	        All {
	            tk_messageBox \
		        -parent .search \
		        -icon info \
	                -message "\"$pattern\" not found." \
	                -type ok
		    return
		}
	        Down {
	            set result [tk_messageBox \
		        -parent .search \
	                -icon question \
	                -message "\"$pattern\" not found, search from beginning?" \
	                -default yes \
	                -type yesno]
		    if {$result == "no"} {
			return
		    }
	            set end "$current + 1c"
		    set current 1.0
		    set direction "All"
		    continue
	        }
	        Up {
	            set result [tk_messageBox \
		        -parent .search \
	                -icon question \
	                -message "\"$pattern\" not found, search from end?" \
	                -default yes \
	                -type yesno]
		    if {$result == "no"} {
			return
		    }
	            set end $current
		    set current "end"
		    set direction "All"
		    continue
	        }
            }
        }
        $t mark set insert $found
        $t see insert
        $t tag remove sel 1.0 end
        $t tag add sel $found "$found + $code::Search(count)c"
        return
    }
}

#
# SearchReplace - replace the current selection with the replacement string
#
proc code::SearchReplace {args} {
    variable Search

    if {!$Search(canreplace)} {
	return
    }
    set t $Search(text)
    # get the replacement string
    set replace [$Search(replacecombo) get]
    set list [$Search(replacecombo) list get 0 end]
    if {[lsearch -exact $list $replace] == -1} {
        $Search(replacecombo) list insert 0 $replace
    }
    while {1} {
	set list [$t tag ranges sel]
	if {$list == {}} {
	    break
	}
	set start [lindex $list 0]
	set end [lindex $list 1]
	$t delete $start $end
	$t insert $start $replace
    }
}

#
# SearchReplaceAll - find the next thing
#
proc code::SearchReplaceAll {} {
    variable Search

    if {!$Search(canreplace)} {
	return
    }
    set t $Search(text)

    # get the search pattern
    set pattern [$Search(replacefindcombo) get]
    if {$pattern == {}} {
	return
    }
    set list [$Search(replacefindcombo) list get 0 end]
    if {[lsearch -exact $list $pattern] == -1} {
	# keep combo boxes for find and replace in sync
	foreach combo {findcombo replacefindcombo} {
	    $Search($combo) delete 0 end
	    $Search($combo) insert 0 $pattern
	    $Search($combo) list insert 0 $pattern
	}
    }
    # get the replacement string
    set replace [$Search(replacecombo) get]
    set list [$Search(replacecombo) list get 0 end]
    if {[lsearch -exact $list $replace] == -1} {
        $Search(replacecombo) list insert 0 $replace
    }

    # start at the beginning
    set current 1.0
    set dir "-forwards"
    set end end
    if {$Search(matchpattern)} {
	set type "-regexp"
    } elseif {$Search(matchcase)} {
	set type -exact
    } else {
	set type -nocase
    }
    while {1} {
        set found [$t search -count code::Search(count) $type $dir $pattern $current $end]
        if {$found == $current} {
	    # start the search one character over
            set found [$t search -count code::Search(count) $type $dir $pattern "$current + 1c" $end]
        }
        if {$found == {}} {
	    # not found
	    break
        }
        $t mark set insert $found
        $t tag remove sel 1.0 end
        $t tag add sel $found "$found + $code::Search(count)c"
	$t delete $found "$found + $code::Search(count)c"
	$t insert $found $replace
	# start search after current replacement
	set current insert
    }
   $t see insert
}

#
# SearchGoto - go to a target
#
proc code::SearchGoto {args} {
    variable Search

    if {![info exists Search(gotocombo)]} {
	return
    }
    set target [$Search(gotocombo) get]
    if {$target == {}} {
	return
    }
    set t $Search(text)
    set what $Search(gotowhat)
    switch $what {
	Line {
	    # go to a line
	    if {[catch {expr double($target)} value]} {
		# not a number
		if {$target != "end"} {
	            tk_messageBox \
		        -parent .search \
		        -icon error \
	                -message "'$target' is a bad goto target" \
	                -type ok
		    return
		}
		set value end
	    }
            $t mark set insert $value
            $t see insert
            $t tag remove sel 1.0 end
	    # to get around a disabling bug
	    SearchWhat {} Line
        }
	Mark {
            $t mark set insert $target
            $t see $target
            $t tag remove sel 1.0 end
	    # to get around a disabling bug
	    SearchWhat {} Mark
	}
    }
}

#
# SearchWhat - gotowhat hash changed
#
proc code::SearchWhat {w what} {
    variable Search

    if {$what == "Line"} {
        set state disabled
        set edit true
        help [winfo toplevel $Search(gotocombo)] $Search(gotocombo) \
	    "Enter a line number or 'end' for the end of the file"
    } else {
	set state normal
	set edit false
        help [winfo toplevel $Search(gotocombo)] $Search(gotocombo) \
	    "Select a text mark"
	set marks [$Search(gotocombo) list get 0 end]
	if {[lsearch $marks [$Search(gotocombo) get]] == -1} {
            $Search(gotocombo) delete 0 end
	}
    }

    if {$what != $Search(gotowhat)} {
        $Search(gotocombo) delete 0 end
    }

    $Search(gotocombo) configure -state $state
    $Search(gotocombo) configure -editable $edit
}

#
# SearchForwardBrace - search forward for a brace
#
proc code::SearchForwardBrace {t start ch find {op 0}} {
    set index $start
    set level 0
    set end [$t index end]
    while {[$t compare $index <= $end]} {
	set index [$t search -regexp \[$find$ch\] $index end]
	if {$index == {}} {
	    return {}
	}
	set this [$t get $index]
	if {$op} {
	    # only find operators
	    set tags [$t tag names $index]
	    if {[lsearch $tags operator] == -1} {
		set this {}
	    }
	} 
	if {$this == $ch} {
	    incr level
	} elseif {$this == $find} {
	    incr level -1
	    if {$level == 0} {
		return $index
	    }
	}
	set index [$t index "$index + 1c"]
	if {$index == $end} {
	    return {}
	}
    }
    return {}
}

#
# SearchBackBrace - search backward for a brace
#
proc code::SearchBackBrace {t end ch find {op 0}} {
    set index $end
    set level 0
    while {[$t compare $index >= 1.0]} {
	set index [$t search -regexp -backwards \[$ch$find\] $index 1.0]
	if {$index == {}} {
	    return {}
	}
	set this [$t get $index]
	if {$op} {
	    # only find operators
	    set tags [$t tag names $index]
	    if {[lsearch $tags operator] == -1} {
		set this {}
	    }
	} 
	if {$this == $ch} {
	    incr level
	} elseif {$this == $find} {
	    incr level -1
	    if {$level == 0} {
		return $index
	    }
	}
	if {$index == 1.0} {
	    return {}
	}
    }
    return {}
}

#
# SearchMatchBrace - match braces and set insertion point
#
proc code::SearchMatchBrace {t {update {}}} {
    # are we at a brace
    set ch [$t get insert]

    if {![regexp "\[]\[\}\{)(]" $ch]} {
	# not found, look later on line
	set index [$t search -regexp {[]\})]} insert "insert lineend"]
	if {$index == {}} {
	    return
	}
	$t mark set insert $index
        # color the brace
	set end [$t index "insert + 1 c"]
	set tags [$t tag names insert]
	set op [expr [lsearch $tags operator] != -1]
        SearchBrace $t [$t get insert] $end $op
	return
    }

    set tags [$t tag names insert]
    set op [expr [lsearch $tags operator] != -1]
    set back 0
    switch -exact -- $ch {
	"\}" {
	    set find "\{"
	    set back 1
	}
	"\{" {
	    set find "\}"
	}
	"]" {
	    set find {[}
	    set back 1
	}
	{[} {
	    set find "]"
	}
	")" {
	    set find "("
	    set back 1
	}
	"(" {
	    set find ")"
	}
    }

    if {$back} {
	# search backwards
	set end [$t index "insert + 1 c"]
        set index [SearchBackBrace $t $end $ch $find $op]
    } else {
	# search forward
        set index [SearchForwardBrace $t [$t index insert] $ch $find $op]
	if {$index != {}} {
	    # set up for coloring
	    set end [$t index "$index + 1 c"]
	    set ch $find
	}
    }
   
    if {$index == {}} {
	return
    }
    $t mark set insert $index
    $t see insert
    if {$update != {}} {
        eval $update $t
    }
    # color the brace
    SearchBrace $t $ch $end $op
}

#
# SearchBrace - do brace match coloring
#
proc code::SearchBrace {t ch {end {}} {op 0}} {
    variable Search

    set w [winfo toplevel $t]

    if {[Preference General bracetime] == 0} {
        return
    }

    set find {}
    switch -exact -- $ch {
	"\}" {
	    set find "\{"
	}
	"]" {
	    set find {[}
	}
	")" {
	    set find "("
	}
    }
    if {$find == {}} {
	return
    }
    
    # cancel any old brace color
    catch {after cancel $Search($w,brace)}
    $t tag remove brace 1.0 end

    if {$end == {}} {
	# use the current insertion point
        set end [$t index insert]
    }

    set start [SearchBackBrace $t $end $ch $find $op]
    if {$start == {}} {
	return
    }

    $t tag configure brace -foreground [Preference General colorbrace]
    $t tag add brace $start $end
    set Search($w,brace) \
	[after [Preference General bracetime] "$t tag remove brace 1.0 end"]
}

