#
# CODE - build the main code window
#

package require mkWidgets

# set debug flag
set code::CODEdebug 1

# keys that can be bound to CODE events
set code::CODEallkeys {
    Control {
	a b c d e f g h i j k l m n o p q r s t u v w x y z
	A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
    }
    Alt {
	0 1 2 3 4 5 6 7 8 9 F1 F2 F3 F4 F5 F6 F7 F8 F9
	a b c d e f g h i j k l m n o p q r s t u v w x y z
	A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
    }
    {} {
        F1 F2 F3 F4 F5 F6 F7 F8 F9 Tab
    }
    Shift {
        0 1 2 3 4 5 6 7 8 9 F1 F2 F3 F4 F5 F6 F7 F8 F9 Tab
    }
    Control-Shift {
        0 1 2 3 4 5 6 7 8 9 F1 F2 F3 F4 F5 F6 F7 F8 F9
    }
}

#
# MAINevents - define main window events
#
# All CODE operations go through events
#
# For each event (lindex value):
#	0 - Menu name (used in menus)
#	1 - Help string (shown in tool tips and status bar)
#	2 - Conditions to enable (used to set state)
#		debug - the debugger is active
#		running - the debugged program is running
#		stopped - the debugged program is stoped
#		nocomplex - the complex breakpoint editor is not opened
#		hascontext - a debugger context is available
#		notdisassembly - not in a disassembly area
#		isdisplay - in a debugger display window
#		iswatch - in a debugger watch window
#		isterminal - in a terminal window
#		foundid - an identifier has been found in the debug context
#		hasfile - a file name is associated with this event
#		onefile - a single file name has been given
#		issource - the file(s) are program source
#		isobject - the file(s) are object files
#		canmakeobject - the file(s) are or can be object files
#		isddf - the file(s) are device definitions files
#		hasredo - redo is available
#		hasundo - undo is available
#		istext - document is a text widget
#		hasid - line has an identifier
#		editable - document is editable text
#		hasselection - the current editor has a selection
#		hasclipboard - the current clipboard has contents
#		haserrors - document has errors
#		ismodified - document is modified
#		hasmacro - line has a macro
#		hasmacrofile - definition of macro is known
#		hasexpansions - line contains macro expansions
#		isinproject - file(s) in project
#		hasdepends - the file is in a project and has dependencies
#		notinproject - file(s) not in project
#		canaddtoproject - file is appropriate for project inclusion
#		options - file(s) can have project options set
#		isdirectory - current window is a directory window
#	3 - noshow flag (popup menu only, if not {}, don't show if disabled)
#	4 - {} for normal button {check <variable>} for checkbutton, 
#	    cascade <list> to make a cascade of events,
#	    or cascade command (command to make a cascaded menu)
#	5 - Tool button name (name used in a toolbar)
#	6 - Code to perform operation
#	7 - non-{} if this should be broadcast. "all": broadcast to all windows
#
set code::MAINevents {
State
    { "State"
      "Update the debugger state in all windows"
      {}
      {}
      {}
      "State"
      { event generate . <<State>>; break }
      all
    }
DebugCommands
    { "Debug Commands"
      "Debugger commands"
      {}
      noshow 
      { cascade {
	    Download Reset Rgo Go Stop Step Next Istep GotoLine
	  }
      }
      "Debug Commands"
      {}
    }
Reset
    { "Reset"
      "Reset the target processor"
      debug
      {}
      {}
      "Reset"
      { code::DebuggerReset; break }
    }
Download
    { "Download"
      "Download the debugger object file"
      { debug stopped }
      {}
      {}
      "Download"
      { code::DebuggerLoad; break }
    }
Target
    { "Target"
      "Select a debugger target"
      { nodebugorstopped }
      {}
      "code::PreferenceTargetMenu .target"
      "Target"
      { .target post %X %Y
        focus .target
	bind .target <FocusOut> ".target unpost"
	break
      }
    }
Go
    { "Go"
      "Start or continue the target program"
      { debug stopped }
      {}
      {}
      "Go"
      { code::DebuggerGo go; break }
    }
Rgo
    { "Reset and Go"
      "Reset the target program and go"
      { debug stopped }
      {}
      {}
      "Rgo"
      { code::DebuggerReset 1; break }
    }
Stop
    { "Stop"
      "Stop the target program"
      { debug running }
      {}
      {}
      "Stop"
      { code::DebuggerStop; break }
    }
Step
    { "Step"
      "Step a line into functions"
      { debug stopped }
      {}
      {}
      "Step"
      { code::DebuggerGo step; break }
    }
Next
    { "Next"
      "Step a line over functions"
      { debug stopped }
      {}
      {}
      "Next"
      { code::DebuggerGo step -over; break }
    }
Istep
    { "Istep"
      "Step a machine instruction"
      { debug stopped }
      {}
      {}
      "Istep"
      { code::DebuggerGo step -instruction; break }
    }
Evaluate
    { "Evaluate"
      "Evaluate $selection"
      { debug stopped hasselection }
      noshow 
      {}
      "Evaluate"
      { code::DebuggerEvaluate %W; break }
    }
Mixed
    { "Mixed"
      "Show disassembly after the current source line"
      { debug }
      noshow 
      { check code::Debugger(domixed) }
      "Mixed"
      { code::Preference Debugger mixed $code::Debugger(domixed); break }
    }
AutoMixed
    { "Auto Mixed"
      "Show disassembly if the PC is not at the start of the line "
      { debug }
      noshow 
      { check code::Debugger(automixed) }
      "Auto Mixed"
      { code::Preference Debugger automixed $code::Debugger(automixed); break }
    }
GotoSelection
    { "Goto Selection"
      "Go to the line containing $selection"
      { debug stopped hasselcontext }
      noshow 
      {}
      "Goto Selection"
      { code::DebuggerGo selection %W; break }
    }
BreakSelection
    { "Breakpoint Selection"
      "Set a breakpoint at $selection"
      { debug stopped hasselcontext }
      noshow 
      {}
      "Breakpoint Selection"
      { code::DebuggerSetBreakpoint selection %W; break }
    }
PCSelection
    { "Set PC to Selection"
      "Set the PC the address of $selection"
      { debug stopped hasselcontext }
      noshow 
      {}
      "Set PC to Selection"
      { code::DebuggerSetPC selection %W; break }
    }
GotoLine
    { "Goto Line"
      "Go to the line containing the cursor"
      { debug stopped hascontext }
      noshow 
      {}
      "Goto Line"
      { code::DebuggerGo widget %W; break }
    }
BreakAtCommands
    { "Breakpoint Commands"
      "Debugger breakpoint commands"
      {}
      noshow 
      { cascade {
	    BreakAt BreakAtRemove BreakAtInactivate BreakAtActivate
	  }
      }
      "Breakpoint Commands"
      {}
    }
BreakAt
    { "Breakpoint"
      "Set, inactivate, or remove a breakpoint on this line"
      { debug stopped hascontext notdisassembly }
      noshow 
      {}
      "Breakpoint"
      { code::DebuggerSetBreakpoint widget %W; break }
    }
BreakAtRemove
    { "Remove Breakpoints"
      "Remove all line breakpoints"
      { debug stopped }
      noshow 
      {}
      "Remove Breakpoints"
      { code::DebuggerAtBreakpoints remove; break }
    }
BreakAtInactivate
    { "Inactivate Breakpoints"
      "Inactivate all line breakpoints"
      { debug stopped }
      noshow 
      {}
      "Inactivate Breakpoints"
      { code::DebuggerAtBreakpoints inactivate; break }
    }
BreakAtActivate
    { "Activate Breakpoints"
      "Activate all line breakpoints"
      { debug stopped }
      noshow 
      {}
      "Activate Breakpoints"
      { code::DebuggerAtBreakpoints activate; break }
    }
BreakComplexCommands
    { "Complex Breakpoint Commands"
      "Debugger complex breakpoint commands"
      {}
      noshow 
      { cascade {
	    BreakComplexAt BreakComplexSelection BreakComplex BreakComplexEdit
	    BreakComplexRemove
	    BreakComplexInactivate BreakComplexActivate
	  }
      }
      "Complex Breakpoint Commands"
      {}
    }
BreakComplexAt
    { "Complex Breakpoint Line"
      "Define a complex breakpoint on this line"
      { debug stopped hascontext nocomplex }
      noshow 
      {}
      "Complex Breakpoint Line"
      { code::DebuggerSetBreakpoint widget %W 0; break }
    }
BreakComplexSelection
    { "Complex Breakpoint Selection"
      "Define a complex breakpoint on $selection"
      { debug stopped hasselcontext nocomplex }
      noshow 
      {}
      "Complex Breakpoint Selection"
      { code::DebuggerSetBreakpoint selection %W 0; break }
    }
BreakComplex
    { "Complex Breakpoint..."
      "Define a new complex breakpoint"
      { debug stopped nocomplex }
      noshow 
      {}
      "Complex Breakpoint"
      { code::DebuggerComplexBreakpoint; break }
    }
BreakComplexEdit
    { "Edit Complex Breakpoint"
      "Edit a complex breakpoint"
      { debug stopped nocomplex }
      noshow 
      "code::DebuggerComplexMenu .complex"
      "Edit Complex Breakpoint"
      { .complex post %X %Y
        focus .complex
	bind .complex <FocusOut> ".complex unpost"
	break
      }
    }
BreakComplexRemove
    { "Remove Complex Breakpoints"
      "Remove all complex breakpoints"
      { debug stopped nocomplex }
      noshow 
      {}
      "Remove Complex Breakpoints"
      { code::DebuggerComplexBreakpoints remove; break }
    }
BreakComplexInactivate
    { "Inactivate Complex Breakpoints"
      "Inactivate all complex breakpoints"
      { debug stopped nocomplex }
      noshow 
      {}
      "Inactivate Complex Breakpoints"
      { code::DebuggerComplexBreakpoints inactivate; break }
    }
BreakComplexActivate
    { "Activate Complex Breakpoints"
      "Activate all complex breakpoints"
      { debug stopped nocomplex }
      noshow 
      {}
      "Activate Complex Breakpoints"
      { code::DebuggerComplexBreakpoints activate; break }
    }
PCLine
    { "Set PC"
      "Set the PC the line containing the cursor"
      { debug stopped hascontext }
      noshow 
      {}
      "Set PC"
      { code::DebuggerSetPC widget %W; break }
    }
Modules
    { "Modules..."
      "Set up debugger script modules"
      { debug }
      noshow
      {}
      "Modules"
      { code::DebuggerModuleSelect; break }
    }
GotoTarget
    { "Goto Target..."
      "Go to a specific target"
      { debug stopped }
      noshow
      {}
      "Goto Target"
      { code::DebuggerGoto; break }
    }
Path
    { "Path"
      "View or add to the source file serach path"
      {}
      {}
      "code::DebuggerPathMenu .path"
      "Path"
      { .path post %X %Y
        focus .path
	bind .path <FocusOut> ".path unpost"
	break
      }
    }
SetAddress
    { "Set Address..."
      "Set the address of this display window"
      { debug isdisplay }
      noshow
      {}
      "Set Address"
      { code::DebuggerSetAddress; break }
    }
Stdio
    { "Program Stdio"
      "Open the standard input/output window"
      debug
      {}
      {}
      "Stdio"
      { code::STDIO; break }
    }
SetupTerminal
    { "Setup Terminal"
      "Set up this terminal window's parameters"
      { isterminal }
      noshow
      {}
      "Setup Terminal"
      { code::TERMINALsetup; break }
    }
Terminal
    { "Terminal"
      "Open a terminal window"
      {}
      {}
      {}
      "Terminal"
      { code::TERMINAL; break }
    }
Registers
    { "Registers"
      "Open the register display window"
      debug
      {}
      {}
      "Registers"
      { code::REGISTERS; break }
    }
Disassembly
    { "Disassembly"
      "Open a disassembly window"
      debug
      {}
      {}
      "Disassembly"
      { code::DISASSEMBLE; break }
    }
Memory
    { "Memory"
      "Open a memory window"
      debug
      {}
      {}
      "Memory"
      { code::MEMORY; break }
    }
Trace
    { "Trace"
      "Enable instruction tracing"
      { debug }
      noshow 
      { check code::Debugger(traceenable) }
      "Trace"
      { code::TRACEcontrol; break }
    }
TraceToolbar
    { "Trace Tools"
      "Add a toolbar with the trace commands"
      { debug }
      noshow 
      {}
      "Trace Tools"
      { code::TRACEtoolbar; break }
    }
TraceCommands
    { "Trace Commands"
      "Trace buffer commands"
      { debug }
      noshow 
      { cascade {
	    TraceToolbar {}
	    TraceGoBack TraceStepBack TraceNextBack TraceIstepBack
	    TraceGoForward TraceStepForward TraceNextForward
	    TraceIstepForward
	  }
      }
      "Trace Commands"
      {}
    }
TraceGoForward
    { "Go Forward"
      "Start or continue the traced program"
      { debug stopped traceon }
      {}
      {}
      "Go Forward"
      { code::DebuggerGo go -forward; break }
    }
TraceStepForward
    { "Step Forward"
      "Step a traced line into functions"
      { debug stopped traceon }
      {}
      {}
      "Step Forward"
      { code::DebuggerGo step -forward; break }
    }
TraceNextForward
    { "Next Forward"
      "Step a traced line over functions"
      { debug stopped traceon }
      {}
      {}
      "Next Forward"
      { code::DebuggerGo step -over -forward; break }
    }
TraceIstepForward
    { "Istep Forward"
      "Step a traced insruction"
      { debug stopped traceon }
      {}
      {}
      "Istep Forward"
      { code::DebuggerGo step -instruction -forward; break }
    }
TraceGoBack
    { "Go Back"
      "Start or continue the traced program backward"
      { debug stopped traceon }
      {}
      {}
      "Go Back"
      { code::DebuggerGo go -back; break }
    }
TraceStepBack
    { "Step Back"
      "Step a traced line backward into functions"
      { debug stopped traceon }
      {}
      {}
      "Step Back"
      { code::DebuggerGo step -back; break }
    }
TraceNextBack
    { "Next Back"
      "Step a traced line backward over functions"
      { debug stopped traceon }
      {}
      {}
      "Next Back"
      { code::DebuggerGo step -over -back; break }
    }
TraceIstepBack
    { "Istep Back"
      "Step a traced insruction backward"
      { debug stopped traceon }
      {}
      {}
      "Istep Back"
      { code::DebuggerGo step -instruction -back; break }
    }
TraceBuffer
    { "Trace Buffer"
      "Open a trace window"
      debug
      {}
      {}
      "Trace Buffer"
      { code::TRACE; break }
    }
WatchCommands
    { "Watch Commands"
      "Watch window commands"
      {}
      noshow 
      { cascade {
	    Watch WatchRegisters
	  }
      }
      "Watch Commands"
      {}
    }
Watch
    { "Watch"
      "Add a watch expression"
      {}
      {}
      {}
      "Watch"
      { code::WATCHadd; break }
    }
WatchRegisters
    { "Watch Registers"
      "Add processor registers to this watch window"
      { debug iswatch }
      noshow
      {}
      "Watch Registers"
      { code::WATCHregisters; break }
    }
WatchWindow
    { "Watch Window"
      "Open an expression watch window"
      debug
      noshow
      {}
      "Watch Window"
      { code::WATCH; break }
    }
Help
    { "Help"
      "Open the manual page for the current window"
      {}
      {}
      {}
      "Help"
      { code::CODEhelp; break }
    }
About
    { "About"
      "About CODE"
      {}
      {}
      {}
      "About"
      { code::signon; break }
    }
OpenFile
    { "Open..."
      "Open a file"
      {}
      {}
      {}
      "Open..."
      { code::OpenFile; break }
    }
OpenFiles
    { "Open Selected"
      "Open $filenative"
      hasfile
      {}
      {}
      "Open Selected"
      { code::OpenFile $code::CODEcurrentfiles; break }
    }
ReopenFile
    { "Reopen"
      "Open a recently opened file"
      {}
      {}
      "code::MainMenuReopen .reopen"
      "Reopen"
      { .reopen post %X %Y
        focus .reopen
	bind .reopen <FocusOut> ".reopen unpost"
	break
      }
    }
Editor
    { "Editor"
      "Open a new editor window"
      {}
      {}
      {}
      "Editor"
      { code::EDIT; break }
    }
OpenView
    { "Open View"
      "Open another window to $filenative"
      { istext onefile }
      {}
      {}
      "Open View"
      { code::EDIT [lindex $code::CODEcurrentfiles 0]; break }
    }
SaveFile
    { "Save $filetail"
      "Save $filenative"
      ismodified
      {}
      {}
      "Save"
      { %W save; break }
    }
SaveAsFile
    { "Save As..."
      "Save $filenative to a specified file"
      istext
      {}
      {}
      "Save As"
      { %W saveas; break }
    }
SaveCopyFile
    { "Save Copy As..."
      "Save a copy of $filenative to a specified file"
      istext
      {}
      {}
      "Save Copy"
      { %W savecopyas; break }
    }
SaveAllFile
    { "Save All"
      "Save all modified files"
      {}
      {}
      {}
      "Save All"
      { code::EDITsaveall; break }
    }
Options
    { "Options..."
      "Edit $filenative build options"
      options
      {}
      {}
      "Options"
      { code::OPTIONS $code::CODEcurrentfiles; break }
    }
AddProject
    { "Add to Project"
      "Add $filenative to the current project"
      { hasfile canaddtoproject notinproject }
      {}
      {}
      "Add File"
      { code::ConfigureAddProject $code::CODEcurrentfiles; break }
    }
RemoveProject
    { "Remove from Project"
      "Remove $filenative from the current project"
      { hasfile isinproject }
      {}
      {}
      "Remove File"
      { code::ConfigureRemoveProject $code::CODEcurrentfiles; break }
    }
Preferences
    { "Preferences..."
      "Change CODE preferences"
      {}
      {}
      {}
      "Preferences"
      { code::PreferenceEdit .; break }
    }
SavePreferences
    { "Save Preferences"
      "Save your current preferences for next time"
      {}
      {}
      {}
      "Save Preferences"
      { code::PreferenceSaveUser; break }
    }
Exit
    { "Exit"
      "Exit CODE"
      {}
      {}
      {}
      "Exit"
      { ::exit; break }
    }
Whatis
    { "What is $word?"
       "Open the manual page for $word"
       { istext hasid }
       noshow
       {}
       "What is?"
       { code::EDITshow %W manual; break }
    }
Redo
    { "Redo"
      "Redo last change"
      hasredo
      {}
      {}
      "Redo"
      { %W redo; break }
    }
Undo
    { "Undo"
      "Undo last change"
      hasundo
      {}
      {}
      "Undo"
      { %W undo; break }
    }
Cut
    { "Cut"
      "Cut to buffer"
      { editable hasselection }
      {}
      {}
      "Cut"
    }
Copy
    { "Copy"
      "Copy to buffer"
      { hasselection }
      {}
      {}
      "Copy"
    }
Paste
    { "Paste"
      "Paste from buffer"
      { editable hasclipboard }
      {}
      {}
      "Paste"
    }
Find
    { "Find..."
      "Search for a string or pattern"
       istext
       {}
       {}
       "Find"
       { code::EDITsearch Find; break }
    }
Replace
    { "Replace..."
      "Search for a string or pattern and replace it"
      editable
      {}
      {}
      "Replace"
      { code::EDITsearch Replace; break }
    }
Goto
    { "Goto..."
      "Goto a specified place in the file"
       istext
       {}
       {}
       "Goto"
       { code::EDITsearch Goto; break }
    }
Mark
    { "Mark..."
      "Place a bookmark at the insertion point"
      istext
      {}
      {}
      "Mark"
      { code::EDITmark; break }
    }
EditorCommands
    { "Editor Operations"
      "Operations that can be done in a editor window"
      {}
      noshow 
      { cascade {
	    Start End Match Transpose InsertAbove InsertNext Join
	    DeleteLine KillLine OpenLine SelectLine ToUpper ToLower
	    Tab ReverseTab
	  }
      }
      "Editor Operations"
      {}
    }
Start
    { "Start"
      "Move the insertion point to the start of the current line"
      istext
      {}
      {}
      "Start"
    }
End
    { "End"
      "Move the insertion point to the end of the current line"
      istext
      {}
      {}
      "End"
    }
Match
    { "Match Brace"
       "Find start and end of braces"
       istext
       {}
       {}
       "Match Brace"
       { code::SearchMatchBrace %W; break }
    }
Transpose
    { "Transpose"
      "Transpose the two characters folowing the insertion point"
      editable
      noshow
      {}
      "Transpose"
    }
InsertAbove
    { "Insert Above"
       "Insert an empty line above the current line"
       editable
       noshow
       {}
       "Insert Above"
       { %W mark set insert [%W index "insert linestart"];
	  %W insert insert "\n";
	  %W mark set insert "insert - 1c";
	  break
       }
    }
InsertNext
    { "Insert Below"
       "Insert an empty line below the current line"
       editable
       noshow
       {}
       "Insert Below"
       { %W mark set insert [%W index "insert lineend"];
	  %W insert insert "\n";
	  break
       }
    }
Join
    { Join
      "Join the next line to the current line"
      editable
      noshow
      {}
      "Join"
      { %W mark set insert [%W index "insert lineend"]
	  while {[string is space [%W get insert]]} {
	      %W delete insert
	  }
	  %W insert insert " "
	  break
      }
    }
DeleteLine
    { "Delete Line"
      "Delete the current line"
      editable
      noshow
      {}
      "Delete Line"
      { %W mark set insert [%W index "insert linestart"]
	  %W delete [%W index "insert linestart"] [%W index "insert lineend + 1c"]
	  break
      }
    }
KillLine
    { "Kill Line"
      "Delete from the insertion point to the end of the current line"
      editable
      noshow
      {}
      "Kill Line"
    }
OpenLine
    { "Open Line"
      "Move text after the insertion point to the next line"
      editable
      noshow
      {}
      "Open Line"
    }
SelectLine
    { "Select Line"
      "Select the current line"
      istext
      {}
      {}
      "Select Line"
      { %W tag remove sel 1.0 end
	  %W tag add sel [%W index "insert linestart"] [%W index "insert lineend"]
	  break
      }
    }
ToUpper
    { "To Upper"
      "Convert selection to upper case"
      { editable hasselection }
      noshow
      {}
      "To Upper"
      { if {[%W tag nextrange sel 1.0 end] != ""} {
	    set %W::temp [%W get sel.first sel.last]
	    %W delete sel.first sel.last
	    %W insert insert [string toupper ${%W::temp}]
	    unset %W::temp
	    break
	  }
      }
    }
ToLower
    { "To Lower"
      "Convert selection to lower case"
      { editable hasselection }
      noshow
      {}
      "To Lower"
      { if {[%W tag nextrange sel 1.0 end] != ""} {
	    set %W::temp [%W get sel.first sel.last]
	    %W delete sel.first sel.last
	    %W insert insert [string tolower ${%W::temp}]
	    unset %W::temp
	    break
	}
      }
    }
Tab
    { "Insert Tab"
      "Insert a tab at the current position or tab selected text over"
      { editable }
      noshow
      {}
      "Insert Tab"
      { if {[%W tag nextrange sel 1.0 end] != ""} {
	    code::TEXTtab %W 1
	    break
	} else {
            %W insert insert \t
            focus %W
            break
        }
      }
    }
ReverseTab
    { "Reverse Tab"
      "Remove a tab at the current position or tab selected text back"
      { editable }
      noshow
      {}
      "Reverse Tab"
      { if {[%W tag nextrange sel 1.0 end] != ""} {
	    code::TEXTtab %W 0
	    break
	} else {
	    if {[%W get "insert - 1c"] == "\t"} {
                %W delete "insert - 1c"
	    }
            focus %W
            break
        }
      }
    }
ExpandLine
    { "Macro Expand Line"
      "View the current source line with all macros expanded"
      { istext hasexpansions }
      noshow
      {}
      "Expand Line"
      { code::EDITshow %W line; break }
    }
Expand
    { "Expand $word"
      "View the expansion of $word"
      { istext hasmacro }
      noshow
      {}
      "Expand Macro"
      { code::EDITshow %W macro; break }
    }
GotoMacro
    { "Goto definition of $word"
      "Open an edit window showing the definition of $word"
      { istext hasmacro }
      noshow
      {}
      "Definition"
      { code::EDITshow %W mgoto; break }
    }
Declaration
    { "Goto declaration of $word"
      "Open an edit window showing the declaration of $word"
      { istext foundid }
      noshow
      {}
      "Declaration"
      { code::EDITshow %W igoto; break }
    }
FileCommands
    { "File Commands"
      "Operations on the current file"
      {}
      noshow 
      { cascade {
	    Check Translate Link Execute {}
	    AddProject Options FileDepends AddDepend RemoveDepend
	    Headers FilePreScript FilePostScript RemoveProject
	  }
      }
      "File Commands"
      {}
    }

Check
    { "Check File"
      "Check $filenative for syntax and semantic errors"
      { hasfile issource }
      {}
      {}
      "Check"
      { code::CODEbuild check $code::CODEcurrentfiles; break }
    }
Translate
    { "Translate"
      "Translate $filenative into object code"
      { hasfile issource }
      {}
      {}
      "Translate"
      { code::CODEbuild object $code::CODEcurrentfiles; break }
    }
Headers
    { "Headers"
      "Generate #include headers from $filenative"
      { hasfile isddf }
      {}
      {}
      "Headers"
      { code::CODEbuild headers $code::CODEcurrentfiles; break }
    }
Link
    { "Link"
      "Link $filenative into a program"
      { hasfile canmakeobject }
      {}
      {}
      "Link"
      { code::CODEbuild program $code::CODEcurrentfiles; break }
    }
Execute
    { "Execute"
      "Execute $filenative after building"
      { hasfile canmakeobject }
      {}
      {}
      "Execute"
      { code::CODEbuild execute $code::CODEcurrentfiles; break }
    }
Color
    { "Color Text"
      "Color or re-color program syntax elements"
      editable
      noshow
      {}
      "Color"
      { code::EDITcolor %W; break }
    }
NextError
    { "Next Error"
      "Go to the next build error"
      { haserrors }
      {}
      {}
      "Next Error"
      { code::EDITnexterror; break }
    }
LastError
    { "Last Error"
      "Go to the last build error"
      { haserrors }
      {}
      {}
      "Last Error"
      { code::EDITlasterror; break }
    }
BuildProject
    { "Build Project"
      "Build the current project"
      {}
      {}
      {}
      "Build"
      { code::CODEbuild build; break }
    }
ForceRebuild
    { "Force Rebuild"
      "Force rebuild of the entire current project"
      {}
      {}
      {}
      "Force Rebuild"
      { code::CODEbuild buildforce; break }
    }
Processor
    { "Processor"
      "Select a processor family"
      { nodebugorstopped }
      {}
      "code::PreferenceProcessorMenu .processor"
      "Processor"
      { .processor post %X %Y
        focus .processor
	bind .processor <FocusOut> ".processor unpost"
	break
      }
    }
Variant
    { "Variant"
      "Select a member of the processor family"
      { nodebugorstopped }
      {}
      "code::PreferenceVariantMenu .variant"
      "Variant"
      { .variant post %X %Y
        focus .variant
        bind .variant <FocusOut> ".variant unpost"
        break
      }
    }
ProjectName
    { "Project Name..."
      "Set the name of the project"
      {}
      {}
      {}
      "Project Name"
      { code::BuildProjectName; break }
    }
ProjectBase
    { "Project Base..."
      "View/Set the project opon which this project is based"
      {}
      {}
      {}
      "Project Base"
      { code::PreferenceBase; break }
    }
ConfigureEnvironment
    { "Configure Environment"
      "Configure the startup code and runtime environment"
      {}
      {}
      {}
      "Configure Environment"
      { code::ConfigureEditEnvironment; break } 
    }
MemoryMap
    { "Memory Map"
      "Edit the target system memory map and linker command file"
      {}
      {}
      {}
      "Memory Map"
      { code::ConfigureEditMemory; break } 
    }
OptionCommands
    { "Program Options"
      "Set global program options"
      {}
      noshow 
      { cascade {
	    GlobalCOptions GlobalAOptions GlobalLOptions
	  }
      }
      "Program Options"
      {}
    }
GlobalCOptions
    { "Compiler Options..."
      "Global C compiler options"
      {}
      {}
      {}
      "Global Compiler Options"
      { code::ConfigureEditGlobalOptions [code::Preference Build processor]ccoptions; break }
    }
GlobalAOptions
    { "Assembler Options..."
      "Global assembler options"
      {}
      {}
      {}
      "Global Assembler Options"
      { code::ConfigureEditGlobalOptions [code::Preference Build processor]asoptions; break }
    }
GlobalLOptions
    { "Linker Options..."
      "Global linker options"
      {}
      {}
      {}
      "Global Linker Options"
      { code::ConfigureEditGlobalOptions [code::Preference Build processor]ldoptions; break }
    }
ProjectNotes
    { "Edit Project Notes..."
      "Edit notes associated with this project"
      {}
      {}
      {}
      "Edit Notes"
      { code::EDITvariable Build notes {Notes about this project}; break }
    }
ScriptCommands
    { "Scripts"
      "Edit the projects global scripts"
      { debug }
      noshow 
      { cascade {
	    PreBuild PreTranslate PostTranslate PreLink PostBuild
	  }
      }
      "Scripts"
      {}
    }
PreBuild
    { "Pre-Build..."
      "Edit the project pre-build script"
      {}
      {}
      {}
      "Pre-Build"
      { code::EDITvariable Build prescript {The script run before this project is built}; break }
    }
PreTranslate
    { "Pre-Translate..."
      "Edit the project pre-translate script"
      {}
      {}
      {}
      "Pre-Translate"
      { code::EDITvariable Build pretranslatescript {The script run before each source file is translated}; break }
    }
PostTranslate
    { "Post-Translate..."
      "Edit the project post-translate script"
      {}
      {}
      {}
      "Post-Translate"
      { code::EDITvariable Build posttranslatescript {The script run after each source file is translated}; break }
    }
PreLink
    { "Pre-Link..."
      "Edit the project pre-link script"
      {}
      {}
      {}
      "Pre-Link"
      { code::EDITvariable Build prelinkscript {The script run before this project is linked}; break }
    }
PostBuild
    { "Post-Build..."
      "Edit the project post-build script"
      {}
      {}
      {}
      "Post-Build"
      { code::EDITvariable Build postscript {The script run after this project is built}; break }
    }
AllDepends
    { "All Dependencies"
      "Find #include file dependencies in the project's C files"
      {}
      {}
      {}
      "All Dependencies"
      { code::ConfigureDoDepends; break }
    }
FileDepends
    { "Dependencies"
      "Find #include file dependencies in $filenative"
      { hasfile isinproject }
      {}
      {}
      "Dependencies"
      { code::ConfigureDepends $code::CODEcurrentfiles; break }
    }
AddDepend
    { "Add Dependency..."
      "Add a dependency to $filenative"
      { hasfile isinproject }
      {}
      {}
      "Add Dependency"
      { code::ConfigureAddDependency $code::CODEcurrentfiles; break }
    }
RemoveDepend
    { "Remove Dependency"
      "Remove a dependency from $filenative"
      { onefile isinproject hasdepends }
      {}
      { code::ConfigureRemoveDependMenu .remdepend }
      "Remove Dependency"
      { .remdepend post %X %Y
        focus .remdepend
        bind .remdepend <FocusOut> ".remdepend unpost"
        break
      }
    }
FilePreScript
    { "File Pre-Script..."
      "Edit the pre-translate script for $filenative"
      { onefile isinproject }
      {}
      {}
      "File Pre-Script"
      { code::EDITvariable Prescript [lindex $code::CODEcurrentfiles 0] "The pre-translate script for [file tail [lindex $code::CODEcurrentfiles 0]]"; break }
    }
FilePostScript
    { "File Post-Script..."
      "Edit the post-translate script for $filenative"
      { onefile isinproject }
      {}
      {}
      "File Post-Script"
      { code::EDITvariable Postscript [lindex $code::CODEcurrentfiles 0] "The post-translate script for [file tail [lindex $code::CODEcurrentfiles 0]]"; break }
    }
ProjectResult
   { "Project Result"
     "Set the type of object this project produces"
     {}
     {}
     "code::PreferenceResultMenu .result"
     "Project Result"
     { .result post %X %Y
       focus .result
       bind .result <FocusOut> ".result unpost"
       break
     }
   }
AddToProject
    { "Add to Project..."
      "Add a file to the current project"
      {}
      {}
      {}
      "Add to Project"
      { code::ConfigureAddProject {}; break }
    }
CloseProject
    { "Close Project"
      "Close the current project"
      {}
      {}
      {}
      "Close Project"
      { code::PreferenceCloseCurrent; break }
    }
SaveProject
    { "Save Project"
      "Save the current project"
      {}
      {}
      {}
      "Save Project"
      { code::PreferencesSave; break }
    }
SaveProjectAs
    { "Save Project As..."
      "Save the current project to a specified file"
      {}
      {}
      {}
      "Save Project As"
      { code::PreferencesSaveAs; break }
    }
GenerateStartup
    { "Generate Startup Code"
      "Save the generated starup code file"
      {}
      {}
      {}
      "Generate Startup"
      { code::CODEbuild generatestartup; break }
    }
ViewLDFile
    { "View LD File"
      "View the generated linker command file in the editor"
      {}
      {}
      {}
      "View LD File"
      { code::CODEbuild viewld; break }
    }
GenerateLDFile
    { "Generate LD File"
      "Save the generated linker command file"
      {}
      {}
      {}
      "Generate LD"
      { code::CODEbuild generateld; break }
    }
ViewMakefile
    { "View Makefile"
      "View a generated makefile in the editor"
      {}
      {}
      {}
      "View Makefile"
      { code::CODEbuild makefile; break }
    }
GenerateMakefile
    { "Generate Makefile"
      "Save the generated Makefile"
      {}
      {}
      {}
      "Generate Makefile"
      { code::CODEbuild generatemakefile; break }
    }
NewProject
    { "New Project..."
      "Start a new project"
      {}
      {}
      {}
      "New Project"
      { code::PreferenceNew; break }
    }
ProjectWizard
    { "Project Wizard..."
      "Start the project wizard"
      {}
      {}
      {}
      "Project Wizard"
      { code::PreferenceWizard; break }
    }
Command
    { "Command Prompt"
      "Open the command prompt window"
      {}
      {}
      {}
      "Command"
      { code::COMMAND; break }
    }
ProjectFiles
    { "Project Files"
      "Open the project files window"
      {}
      {}
      {}
      "Project Files"
      { code::FILES; break }
    }
ProjectOutput
    { "Project Output"
      "Open the project output window"
      {}
      {}
      {}
      "Project Output"
      { code::PROJECT Output; break }
    }
ProjectTools
    { "Project Tools"
      "Open the project tools window"
      {}
      {}
      {}
      "Project Tools"
      { code::PROJECT Tools; break }
    }
ProjectVariant
    { "Project Variant"
      "Open the project variant window"
      {}
      {}
      {}
      "Project Variant"
      { code::PROJECT Variant; break }
    }
MacroExpansions
    { "Macro Expansions"
      "Open the macro expansion window"
      {}
      {}
      {}
      "Macro Expansions"
      { code::EDITmacro; break }
    }
Folder
    { "Folder Browser"
      "Open a folder browser window"
      {}
      {}
      {}
      "Folder"
      { code::DIRECTORY; break }
    }
Rename
    { "Rename"
      "Rename $filenative"
      { isdirectory onefile }
      {}
      {}
      "Rename"
      { code::DirectoryRename [lindex $code::CODEcurrentfiles 0]; break }
    }
Delete
    { "Delete"
      "Delete $filenative"
      { isdirectory hasfile }
      {}
      {}
      "Delete"
      { code::DirectoryDelete $code::CODEcurrentfiles; break }
    }
NewFolder
    { "New Folder"
      "Delete $filenative"
      { isdirectory }
      {}
      {}
      "New Folder"
      { code::DirectoryNewFolder; break }
    }
SelectAll
    { "Select All"
      "Select all files in window"
      { isdirectory }
      {}
      {}
      "Select All"
      { code::DirectorySelectAll; break }
    }
SelectFiles
    { "Select Files..."
      "Select files that match a pattern"
      { isdirectory }
      {}
      {}
      "Select Files"
      { code::DirectorySelectPattern; break }
    }
Refresh
    { "Refresh"
      "Refresh folder window from disk"
      { isdirectory }
      {}
      {}
      "Refresh"
      { code::DirectoryRefresh $code::CODEdocument; break }
    }
FolderCommands
    { "Folder Commands"
      "Folder browser commands"
      {}
      noshow 
      { cascade {
	    GotoProject GotoOutput SetOutput Refresh NewFolder SelectAll
	    SelectFiles Rename Delete
	  }
      }
      "Folder Commands"
      {}
    }
GotoProject
    { "Goto Project"
      "Go to the project folder"
      { isdirectory }
      {}
      {}
      "Goto Project"
      { code::DirectoryGotoProject; break }
    }
GotoOutput
    { "Goto Output"
      "Go to the project output folder"
      { isdirectory }
      {}
      {}
      "Goto Output"
      { code::DirectoryGotoOutput; break }
    }
SetOutput
    { "Set Output"
      "Set the project output folder to $filenative"
      { isdirectory }
      {}
      {}
      "Goto Output"
      { code::DirectorySetOutput $code::CODEdocument; break }
    }
}

# Predefined text widget bindings to use for various text widget operations
set code::TEXTusebindings {
    Start	Control-a
    End		Control-e
    KillLine	Control-k
    OpenLine	Control-o
    Transpose	Control-t
    Cut		<Cut>
    Copy	<Copy>
    Paste	<Paste>
}

tk appname Introl-CODE
wm withdraw .

# set the initial document number
set code::CODEID 0

# make INTROL sane on all targets (forward slashed)
set code::INTROL [eval file join [file split $env(INTROL)]]

# check to see if we are installed
set code::installed [file exists [file join $code::INTROL installed]]

# check to see if we have an accepted license
set code::license [file exists [file join $code::INTROL license]]

# check to see if the manual is re-directed
set code::manual [file join $code::INTROL manual]
if {![catch {open $code::manual r} fd]} {
    if {[catch {read $fd} code::manual]} {
	unset code::manual
	close $fd
	unset fd
    } else {
	set code::manual [string trim $code::manual]
        close $fd
	unset fd
    }
} else {
    # no re-direct file
    set code::manual [file join $code::INTROL]
}

#
# handle event to do checking of CODE event conditions
# A CODE event will only be issued if its conditions are valid
#
rename event _event

proc event {cmnd widget args} {
    global code::CODEevents

    if {$cmnd == "generate"} {
	regexp <<(.*)>> [lindex $args 0] all event
	if {[info exists event] && [info exists CODEevents($event)]} {
	    if {![code::ToolGenerate check $event]} {
		# conditions don't hold
		return
	    }
	    set info $CODEevents($event)
	    set broadcast [lindex $info 7]
	    switch $broadcast {
		all {
		   # send this to all subwindows
		    foreach doc [allDocs] {
			_event generate $doc <<$event>>
		    }
		return
		}
	    }
	}
    }
    eval _event $cmnd $widget $args
}

# move exit out of the way
rename exit _exit

#
# redefine exit to do cleanup
#
proc exit {{code 0}} {
    # save the current main window configuration
    if {![catch {wm geometry .} geometry]} {
        code::Preference Geometry geometry $geometry
    }
    # save preferences
    code::CODEclose
    code::CODEupdatetoolbars
    if {![code::PreferencesSave 0 1]} {
	return
    }

    _exit $code
}

proc arrangeDocs {sHow} {
  ::Document::Arrange $sHow
}

proc allDocs {} {
  return [::Document::Documents]
}

proc allToolbars {parent} {
  return [::Toolbar::Toolbars $parent]
}

#
# CODEtitle - set the CODE title bar
#
proc code::CODEtitle {} {
    variable Debugger

    set title "Introl-CODE Project: [PreferenceProjectName]"
    set rest "[Preference Build processor]"
    if {$rest != {}} {
	set rest ", $rest"
    }
    if {![catch {DebuggerDbg target -id} target]} {
	append rest "-$target"
	if {$Debugger(running)} {
	    append rest "... running"
	} else {
	    append rest "... stopped"
	}
    }
    append title $rest
    wm title . $title
}

#
# DOCraise - called when an document is raised
#
proc code::DOCraise {w} {
    variable CODEdocument
    variable CODEcurrentfiles
    variable DOCraise

    if {[info exists DOCraise]} {
	return
    }
    set DOCraise 1
    set CODEdocument $w
    set CODEcurrentfiles {}

    CODEtitle
    if {$w == {}} {
        .status itemconf Position -text ""
	focus .
        ToolGenerate toolbar
	unset DOCraise
        return
    }
	
    if {[catch {$w cget -type} type]} {
        ToolGenerate toolbar
	unset DOCraise
	return
    }
    if {   ![string match Edit* $type]
	&& [lsearch {Show Command Disassembly Memory} $type] == -1} {
        ToolGenerate toolbar
	unset DOCraise
	return
    }
    set t $w.contents
    if {[catch {$t focus}]} {
        ToolGenerate toolbar
	unset DOCraise
	return
    }
    set file [$t cget -contents]
    if {[string match Edit* $type]} {
        set CODEcurrentfiles [list $file]
    }
    SearchUpdate $w $t $file $file [string match Edit* $type] 1
    ToolGenerate toolbar
    unset DOCraise
}

#
# colorText - color text in a widget depending on a file's type
#
proc code::colorText {t file {cpp {}} \
  {insert 1.0} {endinsert 0.0} {delete 0}} {
    set extension [string tolower [file extension $file]]
    set type [Preference FileTypes file$extension]
    if {$type == {}} {
	return {}
    }
    set type [lindex $type 3]
    set types [cpptypes]
    if {[lsearch $types $type] != -1} {
	if {$cpp == {}} {
	    # open the preprocessor context
	    # define processor
	    set proc [Preference Build processor]
	    if {$proc != {}} {
	        set cpp [cppopen -processor $proc -type $type -string "empty" $file]
	    } else {
	        set cpp [cppopen -type $type -string "empty" $file]
	    }
	    if {$type == "c"} {
	        foreach dir [BuildCIncludes $file] {
	            cppinclude $cpp $dir
	        }
	        if {$proc != {}} {
	            set name __CC[Preference Build ${proc}E]__
	            cppdefine $cpp $name 1
	        }
	        foreach define [BuildCDefines $file] {
	            eval cppdefine $cpp $define
	        }
	    }
	}
	if {$delete} {
	    # deleting text
	    set ranges [cppranges -delete \
		-string [$t get 1.0 end] $cpp $insert $endinsert]
	} else {
	    # inserting text
	    set ranges [cppranges \
		-string [$t get 1.0 end] $cpp $insert $endinsert]
	}
	# puts $ranges
        # remove old tags
	set none [lindex $ranges 0]
        foreach tag [cppclasses] {
	    $t tag remove $tag [lindex $none 1] [lindex $none 2]
        } 
	foreach taglist [lrange $ranges 1 end] {
	    eval $t tag add $taglist
	}
	return $cpp
    } else {
	# we don't know how to color this
    }
    return {}
}

#
# CODEbuild - do a build operation on a file
#
proc code::CODEbuild {command {file {}}} {
    if {[catch {Build . $command $file} result]} {
	tk_messageBox \
	    -parent . \
	    -icon error \
	    -message "Can't build: $result." \
	    -type ok
	return
    }
    if {$result != {}} {
        status . "Made $result"
    }
}

#
# CODEfind - find an document, if any, that already has the specified
# contents and type
#
proc code::CODEfind {type contents} {
    foreach doc [allDocs] {
	if {[string match $type [$doc cget -type]]} {
	    if {[catch {$doc.contents cget -contents} mycontents]} {
		# no contents option
		return $doc
	    } elseif {[string equal $mycontents $contents]} {
	        return $doc
	    }
	}
    }
    return {}
}

#
# CODEproject - set the project field in the status bar
#
proc code::CODEproject {} {
    CODEtitle
    .status itemconf Project -text "Project: [PreferenceProjectName]"
    wm iconname . [PreferenceProjectName]
}

proc code::CODEpreferences {} {
    variable CODEdebug
    variable CODEevents
    variable CODEbindkeys
    variable CODEkeys
    variable CODEallkeys
    variable CODEalt

    catch {unset CODEbindkeys}
    catch {unset CODEkeys}
    array set CODEbindkeys [Preference General ${CODEalt}bindkeys]
    # clear the key used array
    foreach "prefix list" $CODEallkeys {
	foreach key $list {
	    if {$prefix != {}} {
		# prefixed keys
	        set CODEkeys($prefix-$key) 0
	    } else {
		# plain keys
	        set CODEkeys($key) 0
	    }
	}
    }
    foreach event [array names CODEbindkeys] {
	set key $CODEbindkeys($event)
	if {$key != {}} {
	    # this key is used
	    set CODEkeys($key) 1
	}
    }
    foreach bind [array names CODEevents] {
	set command [lindex $CODEevents($bind) 6]
	if {[string match *%W* $command]} {
	    # document commands don't work in .
	    if {![info exists CODEbindkeys($bind)]} {
		# make sure the key array is set
	        if {$CODEdebug} {
		    puts "$bind has no bindkey entry"
	        }
	        set CODEbindkeys($bind) {}
	    }
	    continue
	}
	bind . <<$bind>> $command
	if {[info exists CODEbindkeys($bind)]} {
	    if {$CODEbindkeys($bind) != {}} {
	        # bind keys to . 
	        bind . <$CODEbindkeys($bind)> "event generate . <<$bind>>"
	    }
	} else {
	    # entry doesn't exist
	    if {$CODEdebug} {
		puts "$bind has no bindkey entry"
	    }
	    set CODEbindkeys($bind) {}
	}
    }
}

#
# CODEdestroytoolbar - destroy a toolbar
#
proc code::CODEdestroytoolbar {bar} {
    if {[tk_messageBox \
      -parent . \
      -icon question \
      -message "Really delete this toolbar?" \
      -type yesno] == "yes"} {
        destroy $bar
        CODEupdatetoolbars
    }
}

#
# CODEupdatetoolbars - update current toolbar info
#
proc code::CODEupdatetoolbars {} {
    variable CODEbuttons
    variable CODEalt

    set toolbars ""
    foreach bar [allToolbars .] {
	if {![string match .tbar* $bar]} {
	    # update user definable toolbars only
	    continue
	}
	set thisbar ""
	foreach button [$bar names] {
	    if {$button != {}} {
	        append thisbar " $CODEbuttons($button)"
	    } else {
	        append thisbar " {}\n"
	    }
	}
	set thisbar "\n[$bar cget -side] {\n$thisbar }\n"
	# save in reverse order
	set toolbars $thisbar$toolbars
    }
    
    PreferenceSetIfChanged Control ${CODEalt}toolbars $toolbars
}

#
# CODEtooladd - add one or more tools to a toolbar
#
proc code::CODEtooladd {bar list selected} {
    variable CODEevents
    variable CODEbuttons
    variable CODEtoolwidgets
    variable CODEtoolchanged

    set filenative "current file"
    set word "word"
    set selection "selection"
    foreach row $selected {
	if {$row == 1} {
	    # adding a separator
	    set sep 0
	    while {[catch {$bar add separator sep$sep}]} {
		# find an open separator
		incr sep
	    }
	    continue
	}
	set button [lindex [lindex [$list get -columns button $row] 0] 0]
	set event $CODEbuttons($button)
	set info $CODEevents($event)
        set name [lindex $info 0]
        set help [lindex $info 1]
        set predicates [lindex $info 2]
        set noshow [lindex $info 3]
        set cascade [lindex $info 4]
        set button [lindex $info 5]
        set widget [$bar add button but$event -text $button \
	    -highlightbackground gray]
        set help [subst -nocommands -nobackslashes $help]
        .ttip add $widget $help
	set CODEtoolwidgets($event) $widget
    }
    # delete added buttons from the list
    foreach row $selected {
	if {$row == 1} {
	    # separator
	    continue
	}
	$list delete $row
    }
    set CODEtoolchanged 1
}

#
# CODEtoolselect - enable or disable button depending on selection
#
proc code::CODEtoolselect {list button} {
    if {[$list selection get] == {}} {
        $button configure -state disabled
    } else {
        $button configure -state normal
    }
}

#
# CODEdeletetool - delete a tool from a toolbar
#
proc code::CODEdeletetool {bar} {
    variable CODEdocument
    variable CODEtoolwidgets

    # get the current button
    set button [$bar buttonmenu]
    if {[regexp {.*but([^.]+)} $button all event]} {
        .ttip forget $button
        $bar delete but$event
        unset CODEtoolwidgets($event)
    } elseif {[regexp {.*(sep[^.]+)} $button all separator]} {
        $bar delete $separator
    } else {
	# no match
	return
    }
    CODEupdatetoolbars
    if {$CODEdocument != {}} {
        # set state of new buttons
        $CODEdocument raise
    }
}

#
# CODEaddtool - add a tool to a toolbar
#
proc code::CODEaddtool {bar} {
    variable CODEevents
    variable CODEbuttons
    variable CODEtoolcontrol
    variable CODEtoolchanged
    variable CODEdocument

    set w .
    set box .addtool
    toplevel $box 
    wm transient $box $w
    wm resizable $box 0 0
    wm protocol . WM_DELETE_WINDOW "set code::CODEtoolcontrol {}"
    set f [frame $box.buttons]
    set l [listcontrol $box.list]
    grid $l -in $box -row 0 -column 0 -columnspan 2 -sticky ew -padx 2 -pady 2

    set addBtn [button $f.add -text Add -width 6 -state disabled]
    set closeBtn [button $f.close -text Close -width 6 \
        -command "set code::CODEtoolcontrol {}"]
    grid $addBtn -in $f -row 0 -column 0 
    grid $closeBtn -in $f -row 0 -column 1 
    grid $f -in $box -row 2 -column 0 -columnspan 2 -sticky ew -padx 2 -pady 2

    $l configure -onselect "code::CODEtoolselect $l $addBtn"
    $addBtn configure -command "code::CODEtooladd $bar $l \[$l selection get]"
    $l column insert button end -text Button -width 200 -minsize 32
    $l column insert desc end -text Description -width 500 -minsize 32
    $l bind <Double-1> "code::CODEtooladd $bar $l \[$l selection get]"

    set CODEtoolchanged 0
    set filenative "current file"
    set word "word"
    set selection "selection"
    set current [$bar names]
    $l insert end { "<Separator>" "Add a button separator bar" }
    foreach button [lsort [array names CODEbuttons]] {
	set event $CODEbuttons($button)
        set info $CODEevents($event)
        set help [lindex $info 1]
	if {[lsearch $current $button] != -1} {
	    # already here
	    continue
	}
	set help [subst -nocommands -nobackslashes $help]
	$l insert end [list $button $help]
    }

    placewindow $box widget $w
    wm title $box "Add Tool Buttons"

    set oldFocus [focus]
    set oldGrab [grab current $box]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $box
    focus $l

    tkwait variable code::CODEtoolcontrol
    catch {focus $oldFocus}
    grab release $box
    destroy $box
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    if {$CODEtoolchanged} {
        CODEupdatetoolbars
        ToolGenerate toolbar
    }
}

#
# CODEtoolbar - create a toolbar
#
proc code::CODEtoolbar {side} {
    variable CODEtoolcount

    set tbar .tbar$CODEtoolcount
    toolbar $tbar -side $side
    $tbar menu add separator
    $tbar menu add command -label "Add Tool..." \
	-command "code::CODEaddtool .tbar$CODEtoolcount"
    $tbar menu add command -label "Delete" \
	-command "code::CODEdestroytoolbar .tbar$CODEtoolcount"
    $tbar buttonmenu add command -label "Delete Tool" \
	-command "code::CODEdeletetool .tbar$CODEtoolcount"
    return $tbar
}

#
# CODEcontrol - set up control menus and toolbars
#
proc code::CODEcontrol {{alt {}} {altlist {}}} {
    variable menustatus
    variable CODEevents
    variable CODEbindkeys
    variable CODEkeys
    variable CODEtoolwidgets
    variable CODEtoolnames
    variable CODEtoolcount
    variable CODEdocument
    variable CODEalt
    variable CODEdebug
    if {$CODEdebug} {
        variable CODEeventsused
	CODEpreferences; # this eliminates accelerator key warnings
    }

    set CODEalt $alt
    # create (or recreate) the menu bar
    catch {destroy .menu}
    menu .menu -tearoff 0
    . configure -menu .menu
    # the first few menus are generated from programmable information
    set count 0
    foreach "name underline help list " [concat [Preference Global ${alt}menus] [Preference Control ${alt}menus]] {
        # special hokey cases.
        if {$name == "Help"} {
            set mb help
        } elseif {$name == "System"} {
            set mb system
        } elseif {$name == "Apple"} {
            set mb apple
        } else {
	    set mb mb$count
	    incr count
        }
        set m .menu.$mb
        menu $m -tearoff 0 -postcommand "code::ToolGenerate mainmenu $m \{$list\}"
	set menustatus(.,$name) $help
        .menu add cascade -label $name -menu $m -underline $underline
	# reserve the accelerator in both upper and lower case
	set accel "Alt-[string toupper [string index $name $underline]]"
	if {$CODEdebug && $CODEkeys($accel)} {
	    puts "$accel is already in use"
	}
	set CODEkeys($accel) 1
	set accel "Alt-[string tolower [string index $name $underline]]"
	if {$CODEdebug && $CODEkeys($accel)} {
	    puts "$accel is already in use"
	}
	set CODEkeys($accel) 1
	if {$CODEdebug} {
            foreach entry $list {
                if {$entry == {}} {
 	            continue
                }
	        if {$CODEdebug} {
		    # check for events used
		    CODEcheckevents $entry
	        }
	    }
	}
    }

    if {$CODEdebug} {
	# make sure all events have a menu entry
        foreach event [array names CODEevents] {
	     if {![info exists CODEeventsused($event)]} {
		 puts "$event has no menu entry"
	     }
	}
    }

    # the following menus are fixed
    set list {
        Files "Open a file in this project" 1 "code::MainMenuFiles"
        Window "Manipulate and display CODE windows"  0 "code::MainMenuWindow"
        Help "Browse the CODE Manual" 0 "code::MainMenuHelp"
    }

    if {$altlist != {}} {
	set list $altlist
    }

    foreach "name help underline post" $list {
        # special hokey cases.
        if {$name == "Help"} {
            set mb help
        } elseif {$name == "System"} {
            set mb system
        } elseif {$name == "Apple"} {
            set mb apple
        } else {
	    set mb mb$count
	    incr count
        }
        set m .menu.$mb
        menu $m -tearoff 0 -postcommand "$post $m"
	set menustatus(.,$name) $help
        .menu add cascade -label $name -menu $m -underline $underline
	# reserve the accelerator in both upper and lower case
	set accel "Alt-[string toupper [string index $name $underline]]"
	if {$CODEdebug && $CODEkeys($accel)} {
	    puts "$accel is already in use"
	}
	set CODEkeys($accel) 1
	set accel "Alt-[string tolower [string index $name $underline]]"
	if {$CODEdebug && $CODEkeys($accel)} {
	    puts "$accel is already in use"
	}
	set CODEkeys($accel) 1
    }

    # destroy any old toolbars
    foreach bar [info commands .tbar*] {
	destroy $bar
    }
    catch {unset CODEtoolwidgets}
    catch {destroy .ttip}
    tooltip .ttip
    # set tools when a state change occurs
    set CODEtoolcount 0
    set filenative "current file"
    set word "word"
    set selection "selection"
    foreach "side toolbar" [Preference Control ${alt}toolbars] {
	CODEtoolbar $side
	set sep 0
	foreach event $toolbar {
	    if {$event == {}} {
		# separator
		.tbar$CODEtoolcount add separator sep$sep
		incr sep
		continue
	    }
	    set info $CODEevents($event)
            set name [lindex $info 0]
            set help [lindex $info 1]
            set predicates [lindex $info 2]
            set noshow [lindex $info 3]
            set buttontype [lindex $info 4]
            set button [lindex $info 5]
	    if {[lindex $buttontype 0] == "check"} {
		set variable [lindex $buttontype 1]
                set widget [.tbar$CODEtoolcount add checkbutton but$event -text $button \
	 	    -highlightbackground gray]
		set value 0

		if {[info exists $variable]} {
		    set value [set $variable]
		    # remove any old trace
		    unset $variable
		}
		set $variable $value
		.tbar$CODEtoolcount set but$event [set $variable]
		trace variable [lindex $buttontype 1] w ".tbar$CODEtoolcount set but$event \$[lindex $buttontype 1]; #"
	    } else {
                set widget [.tbar$CODEtoolcount add button but$event -text $button \
	 	    -highlightbackground gray]
	    }
            set help [subst -nocommands -nobackslashes $help]
            .ttip add $widget $help
	    set CODEtoolwidgets($event) $widget
	}
	incr CODEtoolcount
    }
    if {[info exists CODEdocument]} {
        ToolGenerate toolbar
    }
}

#
# CODEcheckevents - mark events used in menus
#
proc code::CODEcheckevents {list} {
    variable CODEevents
    variable CODEeventsused

    foreach entry $list {
	if {$entry == {}} {
	    continue
	}
        set CODEeventsused($entry) 1
	set info $CODEevents($entry)
        set buttontype [lindex $info 4]
	if {[lindex $buttontype 0] == "cascade"} {
	    # mark cascaded events as used
	    CODEcheckevents [lindex $buttontype 1]
	}
    }
}

#
# CODEcloseall - close all open documents
#
proc code::CODEcloseall {} {
    foreach doc [allDocs] {
	if {[$doc cget -type] != "Command"} {
            if {[catch {$doc.contents destroy $doc}]} {
		$doc menu invoke Close
	    }
	} else {
            $doc configure -state withdrawn
	}
    }
}

#
# CODEclose - close all open documents
#
proc code::CODEclose {} {
    variable CODEdocument
    variable CODEalt

    # close all open documents
    set doclist "\n"
    set topdoc {}
    foreach doc [allDocs] {
	set startup [$doc startup]
	if {$startup == {}} {
	    continue
	}
	if {   $topdoc == {}
	    && $CODEdocument == $doc
	    && [$doc cget -state] != "minimized"
	    && [$doc cget -state] != "withdrawn"} {
	    # remember the top document
	    set topdoc "\[$startup"
	    # add the appropriate document options
	    foreach option { -x -y -iX -iY -width -height -state -ontop } {
	        append topdoc " $option [list [$doc cget $option]]"
	    }
	    append topdoc "] raise\n"
	} elseif {[$doc cget -state] == "withdrawn" || [$doc cget -state] == "minimized"} {
	    # create a command to re-create the withdrawn or minimized document before
	    # other windows
	    # add the appropriate document options
	    foreach option { -x -y -iX -iY -width -height -state -ontop } {
	        append startup " $option [list [$doc cget $option]]"
	    }
	    set doclist \n$startup$doclist
	} else {
	    # create a command to re-create the document
	    append doclist $startup
	    # add the appropriate document options
	    foreach option { -x -y -iX -iY -width -height -state -ontop } {
	        append doclist " $option [list [$doc cget $option]]"
	    }
	    append doclist "\n"
	}
	if {[$doc cget -type] != "Command"} {
	    if {![catch {$doc.contents close} value] && $value == 0} {
	        # abort exit
	        return
	    }
	    if {[catch {$doc.contents destroy $doc}]} {
	        $doc menu invoke Close
	    }
	} else {
            $doc configure -state withdrawn
	}
    }
    if {$topdoc != {}} {
        append doclist "$topdoc"
    }
    if {[PreferenceFile] != {}} {
	# a project file is open
        Preference Build ${CODEalt}startup $doclist 1
    } else {
	# no project file is open
        Preference CODE ${CODEalt}startup $doclist 1
    }
}

#
# CODE - set up the initial CODE window
#
proc code::CODE {{filename {}}} {
    variable installed
    variable license
    variable INTROL
    variable CODEdocument
    variable PreferenceFile
    variable CODEevents
    variable MAINevents
    variable CODEbuttons
    variable CODEalt
    variable CODEstarted

    # use main definitions
    set CODEalt {}

    array set CODEevents $MAINevents
    # build the button array
    foreach event [array names code::CODEevents] {
        set code::CODEbuttons([lindex $code::CODEevents($event) 5]) $event
    }

    pack [frame .work -bg gray50] -fill both -expand 1
    # set CODE preferences
    set filetoopen [Preferences $filename]
    wm protocol . WM_DELETE_WINDOW "::exit"
    statusbar .status -ticks 10
    .status add Project -width 200
    trace variable code::PreferenceFile w "code::CODEproject; #"
    trace variable code::PreferenceChanged w "code::CODEproject; #"

    CODEproject
    .status add Position -width 150
    CODEpreferences 
    setFileTypes
    CODEcontrol

    set geometry [Preference Geometry geometry]
    if {$geometry != {}} {
	# the window has had its geometry change saved
	setGeometry . $geometry
    }
    PreferenceWhenChanged General . "code::CODEpreferences"
    PreferenceWhenChanged Control . "code::CODEcontrol"
    wm deiconify .
    tkwait visibility .
    DOCraise {}
    signon 1

    if {[winfo ismapped .signon]} {
	raise .signon
    }
    # this code seems to get around a bug in Tk (wm geometry for
    # a hidden window)
    global tcl_platform
    if {$tcl_platform(platform) == "windows" && $geometry != {}} {
        setGeometry . $geometry
    }

    # remember the main window's geometry
    bind . <Configure> "code::PreferenceMove . geometry"

    if {!$license} {
        # display the code license
        License

        # check for archive file
        if {![catch {glob [file join $code::INTROL code-*]} zipfile]} {
            # A CODE download-able archive exists, ask if it should be deleted
            if {[tk_messageBox \
                -icon question \
                -message "CODE has detected the downloadable archive \"$zipfile\".
This file can be deleted now. Should it be deleted and CODE marked as
installed?" \
                -type yesno] == "yes"} {
                    file delete -force $zipfile
            }

            unset zipfile
        }
    }
    if {!$installed} {
        # mark as installed
        Installed
    }
    if {$filetoopen != {}} {
	if {[catch {OpenFile $filetoopen} msg]} {
	    tk_messageBox \
	        -parent . \
	        -icon error \
	        -message "$msg" \
	        -type ok
	}
    } elseif {$filename != {}} {
	# add the project to recent files
	CODEaddrecent $PreferenceFile
    }

    bind .work <Configure> "arrangeDocs icons"
    CODEstartup
    set CODEstarted 1

    # initialize debugger preferences
    DebuggerPreferences
    PreferenceWhenChanged Debugger . "code::DebuggerPreferences"
    PreferenceWhenChanged DebuggerInternal . "code::DebuggerPreferences"
    PreferenceWhenChanged General . "code::DebuggerPreferences"
    PreferenceWhenChanged Build . "code::DebuggerPreferences"
}  

#
# CODEstartup - open any documents that were previously opened
#
proc code::CODEstartup {} {
    variable CODEdocument
    variable CODEdebug
    variable CODEalt

    if {[PreferenceFile] != {}} {
	# a project file has been opened
        set commands [Preference Build ${CODEalt}startup]
    } else {
	# no project file
        set commands [Preference CODE ${CODEalt}startup]
    }

    # break startup code into lines
    set commands [split $commands "\n"]
    set progress 0.0
    if {[catch {expr 1.0 / [llength $commands].0} increment]} {
	set increment 0.0
    }
    status . "Setting up previous environment"
    foreach command $commands {
	# do each command individually, and ignore errors.
	if {$CODEdebug && [catch {eval $command} msg]} {
            tk_messageBox \
                -parent . \
    	        -icon info \
	        -message "Startup error: $msg" \
	        -type ok
	} elseif {!$CODEdebug} {
	    catch {eval $command}
	}
	set progress [expr $progress + $increment]
	.status configure -progress $progress
    }
    DOCstate $CODEdocument
    .status configure -progress 0.0
    status . ""
}

#
# DOCstate put a docuement in a visable state
#
proc code::DOCstate {w} {
    if {$w != {}} {
	# have a document, raise and return
	if {   [$w cget -state] == "minimized" 
	    || [$w cget -state] == "withdrawn"} {
	    $w configure -state normal
	}
    }
}

#
# OpenFile - the generic file opening proc
#
proc code::OpenFile {{files {}}} {
    if {$files == ""} {
        set initialdir [Preference Editor dirfileopen]
        if {$initialdir == {} || ![file isdirectory $initialdir]} {
  	    set initialdir [pwd]
        }
	set extension {.c .h .ddf .lib .asm .s}
	set aextension {.asm .lib .s}
	set proc [Preference Build processor]
        if {$proc != {}} {
	    lappend extension .s[Preference Build [set proc]E]
	    lappend aextension .s[Preference Build [set proc]E]
	    set executable .e[Preference Build [set proc]E]
	} else {
	    set executable ""
	}

	set file [tk_getOpenFile -title "Open a file" \
	    -initialdir $initialdir \
            -filetypes "
		{{All Files} *}
		{{Project Files} {.code .cod}}
                \"{Assembly Files} [list $aextension]\"
		{{C Files} {.c .h}}
                \"{Source Files} [list $extension]\"
		{{HTML Files} {.html .htm}}
		{{Command Files} {.ld .mk .tcl .mod}}
		{{Control Files} {.mcu .mcd}}
		\"{Executables} [list $executable]\"
            " ]
        if {$file == ""} {
            return ""
        }
        Preference Editor dirfileopen [file dirname $file]
	set files [list $file]
    }
    foreach file $files {
        # determine how to open the file, if given
        if {[catch {file stat $file stat}]} {
            # file is inaccessable for some reason
            set stat(type) file
        }
        set extension [string tolower [file extension $file]]
        set type $stat(type)
        set Type [Preference FileTypes $type$extension]
        if {$Type == {}} {
            set Type [Preference FileTypes $type]
            if {$Type == {}} {
                error "File type botch."
            }
        }
        set toopen [lindex $Type 4]
        if {$toopen != {}} {
	    if {[file pathtype $file] == "relative"} {
	        set file [file join [pwd] $file]
	    }
            eval $toopen \{$file\}
        } else {
	    tk_messageBox \
	        -parent . \
	        -icon error \
	        -message "Don't know how to open $file." \
	        -type ok
	    continue
        }
        # add to the recent files list
	CODEaddrecent $file

    }
}

#
# CODEaddrecent - add a file to the recent file list
#
proc code::CODEaddrecent {file} {
    set file [BuildSubstitute $file]
    if {$file != {} && [file dirname $file] == "."} {
        set file [file join [pwd] $file]
    } 
    set files [Preference General recentfiles]
    set found 0
    set line {}
    foreach entry $files {
        if {[string match $file [lindex $entry 0]]} {
	    # already have this one
	    if {$line == {}} {
	        # use saved line, column
	        set line [lindex $entry 1]
	        set column [lindex $entry 2]
	    }
	    set found 1
	    break
	}
    }
    if {!$found} {
	# add this one
	if {$line == {}} {
	    set line 1
	    set column 0
	}
	set files [concat [list [list $file $line $column]] $files]
	set count [Preference General recentfilecount]
	if {[llength $files] > $count} {
	    set files [lreplace $files $count end]
	}
	Preference General recentfiles $files
    }
}

#
# MainMenuReopen - create the file reopen menu
#
proc code::MainMenuReopen {m} {
    if {![winfo exists $m]} {
        menu $m -tearoff 0 -postcommand "code::MainMenuReopenBuild $m"
    }
    return $m
}

#
# MainMenuReopenBuild - generate the reopen file cascade
#
proc code::MainMenuReopenBuild {m} {
    variable INTROL

    $m delete 0 end
    # build the menu entries
    set list [Preference General recentfiles]
    set needsep 0
    set newlist {}
    if {$list != {}} {
        set project [pwd]/
        set introl $INTROL/
	if {$newlist != {}} {
	    set needsep 1
	}
	# add the recently edited files
        set newlist {}
        set count [Preference General recentfilecount]
	foreach entry $list {
	    if {$count == 0} {
		# have all we want
		continue
	    }
	    set file [BuildSubstitute [lindex $entry 0]]
	    if {[file exists $file]} {
		lappend newlist $entry
	        incr count -1
	    }
        }
	if {$newlist != {}} {
	    if {$needsep} {
	        $m add separator
	    }
	    foreach entry $newlist {
		set file [BuildSubstitute [lindex $entry 0]]
	        set realfile $file
	        if {![regsub ^$project $file {} file]} {
	            regsub ^$introl $file {} file
	        }
                $m add command -label [file nativename $file] \
                    -command "code::OpenFile \{[list $realfile]\}"
	    }
	}
	if {$list != $newlist} {
	    # a file was lost or the count changed
	    Preference General recentfiles $newlist 1
	}
    }
    if {$newlist == {} && !$needsep} {
        $m add command -label "No recent files."
    }
    return $m
}

#
# MainMenuFiles - generate the Files menu
#
proc code::MainMenuFiles {menu} {
    variable menustatus
    variable INTROL

    $menu delete 0 end
    set project [pwd]/
    set introl $INTROL/

    if {[catch {DebuggerDbg files} plist]} {
	set plist {}
    } else {
	# find the names of these files
	set newlist {}
	foreach file $plist {
	    if {![file exists $file]} {
		if {[catch {DebuggerDbg path -find $file} file]} {
		    # could not find
		    continue
		}
	    }
        lappend newlist $file
	}
	set plist $newlist
    }

    set list [concat $plist [Build . projectfiles]]
    if {$list == {}} {
        # no project file open or no files in it
	set name "There are no files in the current project."
        $menu add command -label $name
        return
    }

    set list [MainMenuList $list]
    set count 0
    set lastdir {}
    set lastfile {}
    set allfiles ""
    set break 0
    foreach file $list {
	if {[string equal $lastfile $file]} {
	    # weed out duplicates
	    continue
	}
        if {$count >= 30} {
            set count 0
            set break 1
	}
	if {$file == {}} {
	    # add separator
	    if {$count} {
		# not at bottom of column
	        $menu add separator
		incr count
	    } else {
		# at top of column
	    }
	    set lastdir {}
	    continue
	}
	set lastfile $file
	set file [BuildSubstitute $file]
        set realfile $file
        if {![regsub ^$project $file {} file]} {
            regsub ^$introl $file {} file
        }
	if {[file tail $file] != $file} {
	    # need a cascade
	    set dir [file nativename [file dirname $file]]
	    if {$dir != $lastdir} {
		# a new cascade entry
		set lastdir $dir
		set cm $menu.#$dir
		if {![winfo exists $cm]} {
                    menu $cm -tearoff 0
		} else {
		    $cm delete 0 end
		}
                $menu add cascade -label $dir -menu $cm -columnbreak $break
		set menustatus(.,$dir) "Files in [file nativename [file dirname $realfile]]"
	    } else {
		set break 1; # inhibit count for cascaded files
	    }
	    # add this this to the cascade menu
	    set file [file tail $file]
            $cm add command -label $file \
                -command "code::EDIT [list $realfile]"
	    append allfiles "code::EDIT [list $realfile]; "
	} else {
            set file [file nativename $file]
            $menu add command -label $file \
                -command "code::EDIT [list $realfile]" \
	        -columnbreak $break
            set menustatus(.,$file) "Open [file nativename $realfile]"
	    append allfiles "code::EDIT [list $realfile]; "
	}
	if {$break} {
	    set break 0
	} else {
	    incr count
	}
    }
    $menu add separator
    if {$count >= 30} {
        set break 1
    } else {
        set break 0
    }
    set name "Open All"
    $menu add command -label $name -command $allfiles -columnbreak $break
    set menustatus(.,$name) "Open all project source files"
}
        
#
# MainMenuList - build a file list for a menu
#
proc code::MainMenuList {list} {
    # get a list of files and includes
    set files {}
    set includes {}
    set directories {}
    set inc 0
    foreach file $list {
	if {$file == {}} {
	    set inc 1
	    continue
	}
	if {[file tail $file] != $file} {
	    lappend directories $file
	} elseif {$inc} {
	    lappend includes $file
	} else {
	    lappend files $file
	}
    }
    # sort the lists
    set files [lsort -dictionary $files]
    set includes [lsort -dictionary $includes]
    set directories [lsort -dictionary $directories]

    # return the concatenated lists
    if {$includes != {}} {
	# add a separator
	lappend files {}
	foreach file $includes {
	    lappend files $file
	}
    }

    if {$directories != {}} {
	# add a separator
	lappend files {}
	foreach file $directories {
	    lappend files $file
	}
    }
    return $files
}

#
# MainMenuWindow - generate the windows menu
#
proc code::MainMenuWindow {menu} {
    variable menustatus

    $menu delete 0 end
    set list {
	"Tile Across" "arrangeDocs tile" "Tile subwindows accross main window"
        "Tile Horizontally"  "arrangeDocs horizontally" "Tile subwindows horizontally in the main window"
        "Tile Vertically" "arrangeDocs vertically" "Tile subwindows vertically in the main window"
        "Cascade" "arrangeDocs cascade" "Cascade subwindows in the main window"
	"" "" ""
        "Arrange Icons" "arrangeDocs icons" "Arrange icons in the main window"
        "Minimize All" "arrangeDocs minimize" "Minimize all subwindows in the main window"
	"" "" ""
        "Close All" "code::CODEcloseall" "Close all subwindows in the main window"
        "Add Toolbar" "code::CODEtoolbar top; incr code::CODEtoolcount" "Add a toolbar to the main window"
         "Define Keys" "code::CODEdefinekeys"  "Define keyboard shortcuts for events"
    }

    set count 0
    foreach "name command help" $list {
	set mb mb$count
	incr count
	if {$name == ""} {
	    $menu add separator
	    continue
	}
        $menu add command -label $name -command $command
	set menustatus(.,$name) $help
    }

    # show other windows
    set list [allDocs]
    if {$list == {}} {
	return
    }
    $menu add separator
    foreach doc $list {
        if {$count >= 30} {
            set count 0
            set break 1
	} else {
	    set break 0
	}
	set name [$doc cget -title]
        $menu add command -label $name \
	    -command "code::DOCstate $doc; $doc raise" \
	    -columnbreak $break
        set menustatus(.,$name) "Bring $name to the top"
        incr count
    }
}

#
# CODEhelp - open a manual page appropriate for the current window
#
proc code::CODEhelp {} {
    variable CODEdocument

    if {$CODEdocument == {}} {
	MANUAL
	return
    }
    switch -glob [$CODEdocument cget -type] {
    Folder {
       MANUAL [FindManual Applications Directory contents.html]
    }
    Edit* {
       MANUAL [FindManual Applications Editor contents.html]
    }
    Watch {
       MANUAL [FindManual Applications Debugger Views Watch.html]
    }
    Stdio {
       MANUAL [FindManual Applications Debugger Views Stdio.html]
    }
    Terminal* {
       MANUAL [FindManual Applications Debugger Views Terminal.html]
    }
    Registers {
       MANUAL [FindManual Applications Debugger Views Registers.html]
    }
    Disassembly {
       MANUAL [FindManual Applications Debugger Views Disassembly.html]
    }
    Memory {
       MANUAL [FindManual Applications Debugger Views Memory.html]
    }
    Trace {
       MANUAL [FindManual Applications Debugger Views Trace.html]
    }
    Module* {
       MANUAL [FindManual Applications Debugger Modules.html]
    }
    Files {
       MANUAL [FindManual Applications Project Files.html]
    }
    Output {
       MANUAL [FindManual Applications Project Output.html]
    }
    Tools {
       MANUAL [FindManual Applications Project Tools.html]
    }
    Variant {
       MANUAL [FindManual Applications Project Variant.html]
    }
    Command {
       MANUAL [FindManual Applications Command contents.html]
    }
    default {
	MANUAL
    }
    }
}

#
# MainMenuHelp - display the Help menu
#
proc code::MainMenuHelp {menu} {
    variable menustatus
    variable CODEdocument

    set list {
       About Help
    }
    $menu delete 0 end
    ToolGenerate mainmenu $menu $list
    set list {
	{} {} {}
	"Table of Contents" { contents.html } "Open the manual table of contents"
	"Quick Start" { Applications About.html } "Getting started with CODE"
	"Libraries" { Libraries contents.html } "The CODE runtime library reference"
	"Release Notes" { notes contents.html } "CODE release notes"
	"Reference" { Reference contents.html } "The CODE command line tools reference"
	"Chips" { Reference Chips contents.html } "Microcontrollers supported by CODE"
	"Scripting" { Reference TclTk contents.html } "The CODE Tcl/Tk scripting language reference"
    }
    set needsep 0
    foreach "name where help" $list {
	if {$name == {}} {
	    set needsep 1
	    continue
	}
        set file [eval FindManual $where]
	if {$file == {}} {
	    continue
	}
	if {$needsep} {
	    $menu add separator
	    set needsep 0
	}
	$menu add command -label $name -command "code::MANUAL \{$file\}"
	set menustatus(.,$name) $help
    }
}

#
# status - set the first field in the status bar
#
proc code::status {w msg} {
    if {$w == "."} {
	set w .status
    } else {
	set w $w.status
    }
    if {[catch {set max [winfo width $w]}]} {
	return
    }
    set msg [statusTrim $max $msg]
    $w configure -text $msg
    update idletasks
}

#
# help - set up help on widget entry
#
proc code::help {w widget text {interp {}}} {
    if {$interp != {}} {
	# widget in another interpreter
        interp alias $interp statusenter {} code::statusenter
        interp alias $interp statusleave {} code::statusleave
        interp eval $interp bind $widget <Enter> \[list +statusenter $w $widget \[list $text] $interp]
        interp eval $interp bind $widget <Leave> \[list +statusleave $w $widget]
        interp eval $interp bind $widget <Destroy> \[list +statusleave $w $widget]
    } else {
        bind $widget <Enter> "+code::statusenter $w $widget \"$text\""
        bind $widget <Leave> "+code::statusleave $w $widget"
        bind $widget <Destroy> "+code::statusleave $w $widget"
    }
}

proc code::statusenter {w widget text {interp {}}} {
    variable balloonafter

    if {[Preference General balloonhelp]} {
        after cancel $balloonafter
        status $w {}
	set balloonafter [after 1000 code::balloon $widget \"$text\" $interp]
    } else {
        status $w $text
    }
}

proc code::statusleave {w widget} {
    variable balloonafter

    if {[Preference General balloonhelp]} {
        after cancel $balloonafter
	catch {wm withdraw .balloon}
    } else {
        status $w {}
    }
}

set code::balloonafter {}

#
# balloon - use balloon help
#
proc code::balloon {w arg {interp {}}} {
    if {$interp != {}} {
	set contain [interp eval $interp winfo containing  [winfo pointerxy .]]
    } else {
	set contain [eval winfo containing  [winfo pointerxy .]]
    }
    if {$contain != $w} {
	return
    }
    set top .balloon
    if {![winfo exists $top]} {
        toplevel $top -bd 1 -bg black
        wm overrideredirect $top 1
        pack [message $top.txt -aspect 10000 -bg lightyellow \
            -font fixed -textvariable code::ballon] 
    }
    set code::ballon $arg
    update idletasks
    if {$interp != {}} {
        set wmx [interp eval $interp winfo rootx $w]
	set wmy [interp eval $interp expr \[winfo rooty $w\]+\[winfo height $w\]]
        set screenx [interp eval $interp winfo screenwidth $w]
    } else {
        set wmx [winfo rootx $w]
        set wmy [expr [winfo rooty $w]+[winfo height $w]]
        set screenx [winfo screenwidth $w]
    }
    set width [winfo reqwidth $top.txt]
    if {$wmx + $width > $screenx} {
	# goes off the screen
	incr wmx [expr $screenx - ($wmx + $width)]
    }
    wm geometry $top \
        ${width}x[winfo reqheight $top.txt]+$wmx+$wmy
    wm deiconify $top
    raise $top
    # this code seems to get around a bug in Tk (wm geometry for
    # a hidden window)
    global tcl_platform
    if {$tcl_platform(platform) == "windows"} {
        wm geometry $top \
            [winfo reqwidth $top.txt]x[winfo reqheight $top.txt]+$wmx+$wmy
    }
}

#
# FindManual - find the manual
#
proc code::FindManual {args} {
    variable manual
    variable INTROL

    set file [eval file join \"$code::manual\" $args]
    if {[string match http:* $file]} {
        # fix the first slash
        regsub / $file // file
        # assume http is available
        return $file
    }
    if {[file exists $file]} {
        return $file
    }
    return ""
}

#
# HelpLookupC - lookup a C function name
#
proc code::HelpLookupC {name} {
    variable HelpIndexC

    if {![info exists HelpIndexC]} {
	# fill the index array, only the first time
	foreach dir {ANSI_C Introl_C} {
	     # find the contents
	     set file [FindManual Libraries C $dir contents.html]
	     if {$file == {}} {
		 continue
	     }
	     # get the contents and parse it
	     set fd [open $file]
	     set contents [split [read $fd] "\n"]
	     close $fd
	     foreach line $contents {
		 if {![string match -nocase "*<a href*" $line]} {
		     continue
		}

		# get the name and file for this entry
		regexp {.*[Hh][Rr][Ee][Ff]="([^"]*)".*>(.*)</[Aa]>.*} $line all file function
		if {[string match *contents.html $file]} {
		    # ignore sub-contents files
		    continue
		}
		# create an index
		set HelpIndexC($function) [file join $dir $file]
	     }
	}
    }
    if {[info exists HelpIndexC($name)]} {
	return [FindManual Libraries C $HelpIndexC($name)]
    }
    return {}
}

#
# HelpLookupAsm - lookup an assembly opcode
#
proc code::HelpLookupAsm {name} {

    # assembly opcodes and directives are case insensitive
    set name [string tolower $name]
    # second search: look for name without modifier
    set altname {}
    regexp {(.*)\.[a-zA-Z]} $name all altname

    # look for a directive first
    if {$name != "contents"} {
        set file [FindManual Reference Assembler as10 $name.html]
	if {$file != {}} {
	    return $file
	}
	if {$altname != {}} {
            set file [FindManual Reference Assembler as10 $altname.html]
	    if {$file != {}} {
	        return $file
	    }
	}
    } 

    set proc [Preference Build processor]
    if {$proc == {}} {
        return {}
    }
    # translate if necessary
    switch $proc {
	68000 -
	68010 -
	68020 -
	68030 -
	68332 {
	    set proc 683XX
	}
	6801 -
	6301 {
	    set proc 68HC11
	}
    }

    set file [FindManual Reference Chips $proc $name.html]
    if {$file != {} || $altname == {}} {
        return $file
    }
    return [FindManual Reference Chips $proc $altname.html]
}

#
# CODEmenukey - return a nice key string for a menu accelerator
#
proc code::CODEmenukey {key} {
    regsub -all {\-} $key {+} key
    regsub -all {Control} $key {Ctrl} key
    return $key
}

#
# ToolGenerate - generate tools in a toolbar or menu
#
proc code::ToolGenerate {tooltype args} {
    variable CODEdocument
    variable CODEcurrentfiles
    variable CODEbindkeys
    variable CODEdocument
    variable CODEevents
    variable CODEtoolwidgets
    variable CODEalt
    variable EDITppcontext
    variable EDITerrors
    variable Debugger
    variable menustatus

    set doc $CODEdocument
    if {   ![catch {$doc.contents cget -contents} file] 
	&& [info exists EDITppcontext($file)]} {
	set cpp $EDITppcontext($file)
    } else {
	set cpp {}
    }

    if {[catch {set on $Debugger(notstate)}]} {
	set on 0
    }
    # set initial conditions
    set hasredo 0
    set hasundo 0
    set istext 0
    set hasid 0
    set editable 0
    set hasselection 0
    set hasclipboard 0
    set haserrors 0
    set ismodified 0
    set hasselection 0
    set hasmacro 0
    set hasmacrofile 0
    set hasexpansions 0
    set isinproject 0
    set hasdepends 0
    set notinproject 0
    set canaddtoproject 0
    set options 0
    set isdirectory 0
    set type {}
    set t .
    set nodebugorstopped 0
    set debug 0
    if {[catch {set traceon $Debugger(traceenable)}]} {
	set traceon 0
    }
    set running 0
    set stopped 0
    set nocomplex 0
    set hascontext 0
    set notdisassembly 1
    set isdisplay 0
    set iswatch 0
    set isterminal 0
    set hasselcontext 0
    set foundid 0
    set hasfile 0
    set issource 0
    set isobject 0
    set canmakeobject 0
    set isddf 0
    set onefile 0
    set files {}
    set word word
    set selection selection
    set helpfile {}
    set expansion {}
    set expansions {}
    set macrofile {}
    if {[info commands .breakpoint] == {}} {
	set nocomplex 1
    } else {
	# disable debug commands while editing complex breakpoint
	set on 0
    }
    if {![catch {DebuggerDbg target -id} target]} {
        set debug $on
	set running $Debugger(running)
	set stopped [expr !$running]
	set nodebugorstopped $stopped
    } else {
	set nodebugorstopped 1
    }
    if {$CODEcurrentfiles != {}} {
	set hasfile 1
	set files $CODEcurrentfiles
	set options 1
	set issource 1
	set isobject 1
	set canmakeobject 1
	set isddf 1
	set isinproject 1
        set hasdepends 1
	set notinproject 1
	set canaddtoproject 1
	set objext {}
	foreach file $files {
	    # check each file
	    if {![file exists $file]} {
		set hasfile 0
	    }
	    if {[ConfigureInProject $file]} {
		# is in the project already
		set notinproject 0
	    } else {
		set hasdepends 0
		set options 0
		set isinproject 0
	    }
	    if {[ConfigureCanAdd $file] == {}} {
	        set canaddtoproject 0
	    }
	    if {![ConfigureHasDepends $file]} {
		set hasdepends 0
	    }
	    # check to see if options can be set for this file
            set extension [string tolower [file extension $file]]
	    if {$extension != ".ddf"} {
		set isddf 0
	    }
            set type [Preference FileTypes file$extension]
            set type [lindex $type 3]
	    if {$type != "c" && $type != "asm" && $type != "obj"} {
		set canmakeobject 0
	    }
	    if {$type != "c" && $type != "asm"} {
	        set options 0
		set issource 0
	    } 
	    if {$type != "obj"} {
		set isobject 0
	    } else {
		# make sure object files have the same extension
		if {$objext == {}} {
		    set objext $extension
		} elseif {![string equal $objext $extension]} {
		    set canmakeobject 0
		}
	    }
	}
    }
    if {$doc != {}} {
	if {[lsearch {Memory Disassembly} [$doc cget -type]] != -1} {
	    # this is a debugger memory or disassembly window
	    set isdisplay 1
	}
	if {[string match Watch* [$doc cget -type]]} {
	    # this is a debugger watch
	    set iswatch 1
	}
	if {[string match Terminal* [$doc cget -type]]} {
	    # this is a terminal
	    set isterminal 1
	}
	if {![catch {$doc.contents cget -contents} mycontents]} {
	    # The document is an editor (Editor, Show, etc.)
	    set t $doc.contents
	    set istext 1
	    if {[string equal [$t cget -state] "normal"]} {
		set editable 1
		catch {set ismodified [$t ismodified]}
	    }
	    catch {set hasundo [$t hasundo]}
	    catch {set hasredo [$t hasredo]}
	    if {   ![catch {winfo parent [selection own -displayof $t]} tmp]
	        && $tmp == "$t.text"
	        && ![catch {selection get -displayof $t} selection]
		&& $selection != {}} {
		set hasselection 1
	    } elseif {   ![catch {selection own -displayof $t} tmp]
	        && $tmp == "$t.text"
	        && ![catch {selection get -displayof $t} selection]
		&& $selection != {}} {
		set hasselection 1
	    } else {
		set selection selection
	    } 
	    if {   ![catch {selection get -displayof $t -selection CLIPBOARD} val]
		&& $val != {}} {
		set hasclipboard 1
	    }
			 
	    if {$hasselection} {
		set hasselcontext 1
	        set loc {}
	        # search static context
	        catch {set loc [DebuggerDbg where '$Debugger(source)'.$selection]}
	        if {$loc == {}} {
	            # search global context
	            catch {set loc [DebuggerDbg where $selection]}
	        }
	        set selfile [lindex $loc 0]
	        set selline [lindex $loc 1]
	        set selline [EDITrealline $t $selline]
	        # check for debugger context
	        if {$selfile == {}} {
	            set hasselcontext 0
	        }
                set address '$selfile'@$selline
	        if {[catch {DebuggerDbg where $address}]} {
	            # not a context
	            set hasselcontext 0
	        }
	    }
	    if {$tooltype == "popup"} {
		# get position from mouse event
                set x [lindex $args 3]
                set y [lindex $args 4]
                $t mark set insert [$t index @$x,$y]
	    } 
	    set index [$t index insert]
	    # check for debugger context
	    regexp {([0-9]+)\.([0-9]+)} $index all line column
	    set line [EDITrealline $t $line]
	    set realindex $line.$column
            if {[lsearch [$t tag names insert] "disassembly"] != -1 } {
		    set editable 0
		    # a disassembly line, get address
		    if {[DEBUGaddress $doc insert] != {}} {
			    set hascontext 1
			    set notdisassembly 0
		    }
	    } else {
                set where '[lindex $CODEcurrentfiles 0]'@$line
	        if {![catch {DebuggerDbg expression $where}]} {
	            set hascontext 1
	        }
            }

            set tags [$t tag names $index]
            foreach tag $tags {
                switch $tag {
	            identifier {
                        # get the word we are pointing to
                        set word [eval $t get [$t tag prevrange identifier "$index + 1c"]]
	                set helpfile [HelpLookupC $word]
	                if {![catch {DebuggerDbg where $word} def]} {
		            if {[lindex $def 0] != {} && [lindex $def 1] != 0} {
				set foundid 1
			    }
	                }
                    }
	            opcode {
                        # get the word we are pointing to
                        set word [eval $t get [$t tag prevrange opcode "$index + 1c"]]
	                set helpfile [HelpLookupAsm $word]
                    }
	            macro {
		        # show macro expansion
                        set expansions [cppexpansion -all $cpp $realindex]
			if {$expansions != {}} {
			    set hasexpansions 1
			}
	                if {$cpp != {}} {
		            set macrofile [cppexpansion -define $cpp $realindex]
			    if {$macrofile != {}} {
				set hasmacrofile 1
			    }
                            set expansion [cppexpansion $cpp $realindex]
	                    set word \
			        [eval $t get [$t tag prevrange macro "$index + 1c"]]
			    set hasmacro 1
	                }
	            }
	        }
            }
            if {$helpfile != {}} {
		set hasid 1
	    }
	} elseif {[$doc cget -type] == "Folder"} {
	   set isdirectory 1
	}
    }

    if {[llength $files] == 1} {
	if {$files != {}} {
	    set onefile 1
	} else {
	    set files file
	}
        set filetail [file tail [lindex $files 0]]
        set filenative [file nativename [file tail [lindex $files 0]]]
    } else {
        set filetail "selected files"
	set filenative "selected files"
    }

    if {[info exists EDITerrors]} {
       set haserrors 1
    }
    switch $tooltype {
    check {
	# check an event for validity
	set info $CODEevents([lindex $args 0])
        set predicates [lindex $info 2]
	set state 1
	# determine if entry should be activated
	foreach cond $predicates {
	    if {![set $cond]} {
	        set state 0
		break
	    }  
	}
	return $state
    }
    mainmenu {
	set m [lindex $args 0]
        $m delete 0 end
	set list [lindex $args 1]
	set w [lindex $args 2]
	set menu [lindex $args 3]
        foreach entry $list {
            if {$entry == {}} {
                $m add separator
 	        continue
            }
	    set info $CODEevents($entry)
            set name [lindex $info 0]
            set help [lindex $info 1]
            set predicates [lindex $info 2]
            set noshow [lindex $info 3]
            set buttontype [lindex $info 4]
	    set state normal
	    # determine if entry should be activated
	    foreach cond $predicates {
	        if {![set $cond]} {
		    set state disabled
		    break
	        }  
	    }
	    set sequence {}
	    if {[info exists CODEbindkeys($entry)]} {
                set sequence [CODEmenukey $CODEbindkeys($entry)]
	    }
	    set name [subst -nocommands -nobackslashes $name]
	    set help [subst -nocommands -nobackslashes $help]
	    if {[lindex $buttontype 0] == "check"} {
                set command "event generate $t <<$entry>>"
                $m add checkbutton -label $name  -state $state \
	            -command $command -variable [lindex $buttontype 1] -accelerator $sequence
                set menustatus(.,$name) $help
	    } elseif {[lindex $buttontype 0] == "cascade"} {
	        set sm .menu$entry
		if {![winfo exists $sm]} {
		    menu $sm -tearoff 0 \
		        -postcommand "code::ToolGenerate cascade $sm \{[lindex $buttontype 1]\}"
		}
	        $m add cascade -label $name \
	            -menu $sm -state $state -accelerator $sequence
                set menustatus(.,$name) $help
	    } elseif {$buttontype  != {}} {
	        set sm [eval $buttontype]
	        $m add cascade -label $name \
	            -menu $sm -state $state -accelerator $sequence
                set menustatus(.,$name) $help
	    } else {
                set command "event generate $t <<$entry>>"
                $m add command -label $name  -state $state \
	            -command $command -accelerator $sequence
                set menustatus(.,$name) $help
	    }
        }
    }
    toolbar {
        # do the toolbars
        set count 0
        foreach "side toolbar" [Preference Control ${CODEalt}toolbars] {
	    foreach event $toolbar {
	        if {$event == {}} {
		    # separator
		    continue
	        }
	        set info $CODEevents($event)
                set name [lindex $info 0]
                set help [lindex $info 1]
                set predicates [lindex $info 2]
                set noshow [lindex $info 3]
                set buttontype [lindex $info 4]
                set button [lindex $info 5]
	        set state normal
	        foreach cond $predicates {
	            if {![set $cond]} {
		        set state disabled
		        break
	            }  
	        }
	        set help [subst -nocommands -nobackslashes $help]
	        if {[lindex $buttontype 0] == "check"} {
                    set command "set [lindex $buttontype 1] \[.tbar$count get but$event]; event generate $t <<$event>>"
	        } elseif {[lindex $buttontype 0] == "cascade"} {
	            set sm [ToolGenerate cascade .menu$entry [lindex $buttontype 1]]
		    # set up a position for the cascaded menu
                    set command "event generate $t <<$event>> \
		         -x \[expr \[winfo pointerx $t] - \[winfo rootx $t]] \
		         -y \[expr \[winfo pointery $t] - \[winfo rooty $t]]"
	        } elseif {$buttontype != {}} {
	            set sm [eval $buttontype]
		    # set up a position for the cascaded menu
                    set command "event generate $t <<$event>> \
		         -x \[expr \[winfo pointerx $t] - \[winfo rootx $t]] \
		         -y \[expr \[winfo pointery $t] - \[winfo rooty $t]]"
	        } else {
                    set command "event generate $t <<$event>>"
	        }
                .tbar$count itemconf but$event -state $state \
		    -highlightbackground gray -command $command
                .ttip add $CODEtoolwidgets($event) $help
	    }
	    incr count
        }
        }
    cascade -
    popup {
	set popupname [lindex $args 0]
	if {$tooltype == "popup"} {
            set w [lindex $args 2]
            set X [lindex $args 5]
            set Y [lindex $args 6]
            if {$w == "."} {
	        set base ""
            } else {
	        set base $w
            }
            set m $base.$popupname
            if {![winfo exists $m]} {
                menu $m -tearoff 0
            }
        } else {
	    # a cascade menu
            set m $popupname
	}
        $m delete 0 end

        # generate the menu
	if {$tooltype == "popup"} {
	    set list [Preference Global $CODEalt$popupname]
	} else {
	    set list [lindex $args 1]
	}
        foreach entry $list {
            if {$entry == {}} {
                $m add separator
 	        continue
            }
	    set info $CODEevents($entry)
            set name [lindex $info 0]
            set help [lindex $info 1]
            set predicates [lindex $info 2]
            set noshow [lindex $info 3]
            set buttontype [lindex $info 4]
	    set state normal
	    # determine if entry should be activated
	    foreach cond $predicates {
	        if {![set $cond]} {
		    set state disabled
		    break
	        }  
	    }
	    if {   $tooltype == "popup"
		&& [string equal $state "disabled"] && $noshow != {}} {
	        # don't show in popup if disabled
	        continue
	    }
	    set sequence {}
	    if {$tooltype != "popup" && [info exists CODEbindkeys($entry)]} {
                set sequence [CODEmenukey $CODEbindkeys($entry)]
	    }
	    set name [subst -nocommands -nobackslashes $name]
	    set help [subst -nocommands -nobackslashes $help]
	    if {[lindex $buttontype 0] == "check"} {
                set command "event generate $t <<$entry>>"
                $m add checkbutton -label $name  -state $state \
	            -command $command -variable [lindex $buttontype 1] -accelerator $sequence
                set menustatus(.,$name) $help
	    } elseif {[lindex $buttontype 0] == "cascade"} {
	        set sm .menu$entry
		if {![winfo exists $sm]} {
		    menu $sm -tearoff 0 \
		        -postcommand "code::ToolGenerate cascade $sm \{[lindex $buttontype 1]\}"
		}
	        $m add cascade -label $name \
	            -menu $sm -state $state -accelerator $sequence
                set menustatus(.,$name) $help
	    } elseif {$buttontype  != {}} {
	        set sm [eval $buttontype]
	        $m add cascade -label $name \
	            -menu $sm -state $state -accelerator $sequence
                set menustatus(.,$name) $help
	    } else {
                set command "event generate $t <<$entry>>"
                $m add command -label $name  -state $state \
	            -command $command -accelerator $sequence
                set menustatus(.,$name) $help
	    }
        }

	if {$tooltype == "popup"} {
            tk_popup $m $X $Y
	} else {
	    return $m
	}
    }
    }
}

#
# PROJECTstartup - return the command to open a project window
#
proc code::PROJECTstartup {window doc} {
    return "PROJECT $window"
}

#
# PROJECT - open a project window
#
proc code::PROJECT {window args} {
    variable INTROL

    set w [CODEfind $window {}]
    DOCstate $w
    if {$w != {}} {
	$w raise
	return
    }
    # open a new project window

    set w [eval document .work.p$window -type $window \
	-raiseproc code::DOCraise $args]
    $w configure -startupproc "code::PROJECTstartup $window"
    bind $w <<State>> break

    set f [frame $w.contents]
    $w pack $f -fill both -expand 1 -side right

    # get the ui directory
    set dir [file join $INTROL tcltk code]
    set file [file join $dir $window.ui.tcl]
    source $file
    ${window}_ui $f .
    set image [CODEicon codefile.icon]
    $w configure -image $image -icontext $window -title "Project $window"
    if {$args == {}} {
        update idletasks
        $w configure -width [winfo reqwidth $w] -height [winfo reqheight $w]
    }
    return $w
}

#
# CODEdefinekeys - assign keys to events
#
proc code::CODEdefinekeys {} {
    variable CODEevents
    variable CODEbuttons
    variable CODEbindkeys
    variable CODEkeyrow
    variable CODEavailrow

    set w .
    set box .definekeys
    toplevel $box 
    wm transient $box $w
    wm protocol . WM_DELETE_WINDOW "destroy $box"
    # wm resizable $box 0 0
    set f [frame $box.lists]
    set l [listcontrol $f.list -width 70 -selectmode single]
    set al [listcontrol $f.alist -width 22 -selectmode single]
    grid $f -in $box -row 1 -column 0
    grid $l -in $f -row 0 -column 0 -sticky ew -padx 2 -pady 2
    grid $al -in $f -row 0 -column 1 -sticky ns -padx 2 -pady 2

    set f [frame $box.buttons]
    set changeBtn [button $f.change -text Change -width 6 -state disabled]
    set closeBtn [button $f.close -text Close -width 6 \
        -command "destroy $box"]
    grid $changeBtn -in $f -row 0 -column 0 
    grid $closeBtn -in $f -row 0 -column 1 
    grid $f -in $box -row 2 -column 0 -sticky ew -padx 2 -pady 2

    set CODEkeyrow {}
    set CODEavailrow {}
    $l configure -onselect "set code::CODEkeyrow \[$l selection get]; code::CODEkeyselect $changeBtn"
    $al configure -onselect "set code::CODEavailrow \[$al selection get]; code::CODEkeyselect $changeBtn"
    $changeBtn configure -command "code::CODEchangekey $l $al $changeBtn"
    $l column insert key end -text Key -width 150 -minsize 32
    $l column insert desc end -text Description -width 500 -minsize 32
    $l column insert event end -text Event -width 200 -minsize 32

    set filenative "current file"
    set word "word"
    set selection "selection"
    foreach button [lsort [array names CODEbuttons]] {
	set event $CODEbuttons($button)
        set info $CODEevents($event)
        set help [lindex $info 1]
	set help [subst -nocommands -nobackslashes $help]
	$l insert end [list $CODEbindkeys($event) $help $event]
    }

    $al column insert key end -text "Available Keys" -width 150 -minsize 32
    $al bind <Double-1> "code::CODEchangekey $l $al $changeBtn"

    CODEkeyfill $al
    placewindow $box widget $w
    wm title $box "Define Tool Keys"
}

#
# CODEkeyselect - enable or disable button depending on selection
#
proc code::CODEkeyselect {button} {
    variable CODEkeyrow
    variable CODEavailrow

    if {$CODEkeyrow == {} || $CODEavailrow == {}} {
        $button configure -state disabled
    } else {
        $button configure -state normal
    }
}

#
# CODEchangekey - change a key in the key definition list box
#
proc code::CODEchangekey {list keylist button} {
    variable CODEkeys
    variable CODEkeyrow
    variable CODEavailrow
    variable CODEbindkeys

    set row $CODEkeyrow
    set keyrow $CODEavailrow
    set orig [lindex [$list get -columns key $row] 0]
    set new [lindex [$keylist get -columns key $keyrow] 0]
    if {$orig == $new} {
	return
    }
    if {$orig != {} && $orig != " "} {
        # original key not used now
        set CODEkeys($orig) 0
    }
    if {$keyrow == 1} {
	# clear the key
	$list set -columns key $row [list {}]
	set new {}
    } else {
	$list set -columns key $row $new
        set CODEkeys($new) 1
	set CODEavailrow {}
	CODEkeyselect $button
    }
    CODEkeyfill $keylist

    # change has occured, update the preferences
    set CODEbindkeys([$list get -columns event $row]) $new
    set list ""
    foreach event [array names CODEbindkeys] {
	append list "\t$event\t[list $CODEbindkeys($event)]\n"
    }
    PreferenceSetIfChanged General bindkeys $list
}

#
# CODEkeyfill - fill the available key list
#
proc code::CODEkeyfill {al} {
    variable CODEkeys

    $al delete 0 end
    $al insert 0 { {<No Key>} }
    foreach key [lsort [array names CODEkeys]] {
	if {$CODEkeys($key)} {
	    # in use
	    continue
	}
	$al insert end [list $key]
    }
}

#
# TEXTtab - tab a selection right or left
#
proc code::TEXTtab {t right} {
    set sel [$t tag nextrange sel 1.0 end]
    set start [$t index "[lindex $sel 0] linestart"]
    set end [$t index "[lindex $sel 1] - 1c linestart"]
    set text [$t get $start "$end lineend"]
    set text [split $text \n]
    set result ""
    set newline ""
    foreach line $text {
        if {$right} {
	    # insert a tab on non-empty lines
	    if {$line != ""} {
		append result "$newline\t$line"
	    } else {
		append result "$newline"
	    }
	} else {
	    # remove a tab, if present
	    if {[string index $line 0] == "\t"} {
		append result "$newline[string range $line 1 end]"
	    } else {
		append result "$newline$line"
	    }
	}
	set newline "\n"
    }
    # replace the tabbed text
    $t delete $start "$end lineend"
    $t insert $start $result
    # restore the selection
    $t tag add sel $start "$end lineend"
    $t see insert
}

#
# TEXTstartup - return a command to recreate a text window
#
proc code::TEXTstartup {doc} {
    return "TEXT"
}

#
# TEXT - start a text window
#
proc code::TEXT {args} {
    variable CODEevents
    variable CODEID
    variable TEXTusebindings
    variable CODEbindkeys

    set doc .work.doc$CODEID
    incr CODEID
    
    document $doc
    bind $doc <<State>> "break"
    $doc configure -type Text
    scrolltext $doc.contents -scrollbar auto -wrap none -highlightthickness 0
    DOCraise $doc
    foreach "bind key" $TEXTusebindings {
	bind $doc.contents <<$bind>> "[bind Text <$key>]; break"
    }
    foreach bind [array names CODEevents] {
	set command [lindex $CODEevents($bind) 6]
	if {$command == {}} {
	    # this uses a previous binding
	    continue
	}
	bind $doc.contents <<$bind>> $command
	if {[info exists CODEbindkeys($bind)]} {
	    if {$CODEbindkeys($bind) != {}} {
	        # bind keys to text widget
	        bind $doc.contents.text <$CODEbindkeys($bind)> \
		    "event generate $doc.contents <<$bind>>; break"
	    }
	}
    }
    $doc pack $doc.contents -fill both -expand 1
    # check to see if this file is a source file
    bind $doc.contents <1> "[bind Scrolltext <1>]; $doc raise" 
    bind $doc.contents <3> "$doc raise; code::ToolGenerate popup editorpopup $doc . %x %y %X %Y"
    $doc configure -startupproc "code::TEXTstartup"
    if {$args != {}} {
	# use previous values
        eval $doc configure $args
    } else {
        set x [$doc cget -x]
        set y [$doc cget -y]
        set width [expr {[winfo width .work] - 10 - $x}]
        set height [expr {[winfo height .work] - 10 - $y}]
        $doc config -width $width -height $height
    }
    $doc configure -raiseproc code::DOCraise
    TEXTpreferences $doc
    PreferenceWhenChanged Editor $doc "code::TEXTpreferences $doc"
    PreferenceWhenChanged General $doc "code::TEXTpreferences $doc"
    PreferenceWhenChanged Control $doc "code::TEXTpreferences $doc"
    PreferenceWhenChanged Debugger $doc "code::TEXTpreferences $doc"
    return $doc
}

#
# TEXTpreferences - set the preferences of an text window
#
proc code::TEXTpreferences {doc} {
    variable CODEbindkeys
    variable EDITppcontext

    if {   ![catch {$doc.contents cget -contents} file] 
	&& [info exists EDITppcontext($file)]} {
	set cpp $EDITppcontext($file)
    } else {
	set cpp {}
    }

    set editor $doc.contents
    $editor configure -wrap [Preference Editor modewrap]
    $editor configure -foreground [Preference Editor colornormal]
    $editor configure -font [Preference Editor fonteditor]
    if {![catch {$editor cget -infowidth} width] && $width} {
        $editor infoconfigure -font [Preference Editor fonteditor]
    }
    catch {$editor configure -autoreload [Preference Editor autoreload]}
    if {[string match Edit* [$doc cget -type]]} {
        if {[Preference Editor autoindent]} {
	    bind $editor <Key-Return> "code::EDITindent $editor; break"
        } else {
	    bind $editor <Key-Return> {}
        }
    }
    set tabs [Preference Editor tabs]
    $editor configure -defaulttabstops $tabs

    # syntax element colors
    # C tags
    $editor tag configure operator -foreground [Preference General coloroperator]
    $editor tag configure reserved -foreground [Preference General colorreserved]
    $editor tag configure sel -background [Preference General colorselection]
    $editor tag configure constant -foreground [Preference General colorconstant]
    $editor tag configure string -foreground [Preference General colorstring]
    $editor tag configure identifier -foreground [Preference General coloridentifier]
    $editor tag configure ppdirective -foreground [Preference General colorppdirective]
    $editor tag configure skipped -foreground [Preference General colorskipped]
    $editor tag configure macro -foreground [Preference General colormacro]
    $editor tag configure comment -foreground [Preference General colorcomment]
    $editor tag bind macro <Enter> "code::EDITmacroEnter %W %x %y $cpp"
    $editor tag bind macro <Motion> "code::EDITmacroEnter %W %x %y $cpp"
    $editor tag bind macro <Leave> "code::EDITmacroLeave"

    # assembly tags
    $editor tag configure label -foreground [Preference General coloridentifier]
    $editor tag configure opcode -foreground [Preference General colorreserved]

    $editor tag configure warning -background [Preference Editor colorwarning]
    $editor tag configure error -background [Preference Editor colorerror]
    $editor tag configure curerr -background [Preference Editor colorcurrent]
    $editor config -bg [Preference General colorbackground]
    $editor tag configure cursrc -background [Preference Debugger colorcurrent]
    if {[string match Edit* [$doc cget -type]]} {
        $editor tag configure disassembly -background [Preference Editor colordisassembly]
    }

    foreach event [array names CODEbindkeys] {
	set key $CODEbindkeys($event)
	if {$key == {}} {
	    continue
	}
	bind $editor <$key> [bind $editor <<$event>>]
    }
    EDITposition $doc [$editor index insert]
}

#
# PreferenceMove - get a window's geometry when it moves
#
proc code::PreferenceMove {w which} {
    if {![catch {wm geometry $w} geometry]} {
        Preference Geometry ${which} $geometry
    }
}

#
# CODENotebook - create a notebook containing ui tabs
#
proc code::CODENotebook {name top pref {buttons {}} args} {
    variable INTROL
    variable CODEalt

    # get the ui directory
    set dir [file join $INTROL tcltk code $name]

    # get the tab order
    if {[catch {open [file join $dir Tabs]} f]} {
	return {}
    }
    set list [read $f]
    close $f

    set top [toplevel $top]
    wm title $top $name
    if {$CODEalt == {}} {
	# remember position in the main window only
        wm protocol $top WM_DELETE_WINDOW "code::PreferenceClose $top $pref"
    }
    Preference General show$pref 1
    set n $top.nb

    set geometry [Preference Geometry $pref]
    setGeometry $top $geometry place
    Notebook:create $n -pages $list -pad 5 -status code::status($top)
    if {$args != {}} {
        eval Notebook:config $n $args
    }

    grid $n -in $top -row 0 -column 0 -sticky nsew
    grid rowconfigure $top 0 -weight 1
    grid columnconfigure $top 0 -weight 1
    if {$buttons != {}} {
	# add a button frame
	set b [frame $top.$buttons]
        grid $b -in $top -row 1 -column 0 -sticky ew
    }
    label $top.l -text "" -textvariable code::status($top) \
	-relief sunken -bd 2 \
	-bg [Preference General colorstatusbackground] \
        -anchor w -font [Preference General fontstatus]
    PreferenceWhenChanged General $top "$top.l config \
	-bg \[Preference General colorstatusbackground] \
        -font \[Preference General fontstatus]"
    grid $top.l -in $top -row 2 -column 0 -sticky ew
    grid columnconfigure $top.l 0 -weight 1

    # initialize each tab
    foreach tab $list {
        set w [Notebook:frame $n $tab]
        set file [file join $dir $tab.ui.tcl]
	source $file
	set help [${tab}_ui $w $top]
        code::Notebook:pageconfig $n $tab -status $help
	help $top $w $help
    }

    # this code seems to get around a bug in Tk (wm geometry for
    # a hidden window)
    global tcl_platform
    if {$tcl_platform(platform) == "windows" && $geometry != {}} {
	update idletasks
        setGeometry $top $geometry place
    }

    bind $top <Configure> "code::PreferenceMove $top $pref"
    return $n
}

#
# CODEicon - return the image for an icon or file
#
proc code::CODEicon {file} {
    variable FileIcons

    if {[info exists FileIcons($file)]} {
	return $FileIcons($file)
    }
    if {[catch {file stat $file stat}]} {
        # file is inaccessable for some reason
        set stat(type) file
    }
    set extension [string tolower [file extension $file]]
    set Type [Preference FileTypes $stat(type)$extension]
    if {$Type == {}} {
        set Type [Preference FileTypes $stat(type)]
    }
    if {$Type == {}} {
        error "File type botch."
    }
    return $FileIcons([lindex $Type 0])
}
