package require Metawidget

# name: Combobox
# args: args
# for the first instance defines the image for the arrow button.
# creates all widgets including the popup list and its toplevel window.
# the toplevel is withdrawn. defines bindings for keys and mouse clicks.
# initializes option values by calling the respective option members.
metawidget create Combobox {
  if { ! [ourinfo exists pDown] } {
    our pDown [image create bitmap -data {
      #define _width 8
      #define _height 4
      static char _bits[] = { 0xfe, 0x7c, 0x38, 0x10 };
    }]
  }

  entry     $this.entr      -bg white -selectborder 0 -selectback black -selectfore white
  button    $this.entr.arrb -padx 0 -pady 0 -image [our pDown] -command "$this _showList" -cursor arrow
  toplevel  $this.topl      -border 0
  scrollbar $this.topl.scrv -command "$this.topl.list yview"
  listbox   $this.topl.list -bg white -border 0 -yscroll "$this.topl.scrv set" \
                             -selectborder 0 -selectback {dark blue} -selectfore white

  wm withdraw $this.topl
  pack  $this.entr      -fill x -expand 1
  place $this.entr.arrb -relx 1 -y 0 -relh 1 -anchor ne
  pack  $this.topl.scrv -side right -fill y
  pack  $this.topl.list -side left -fill both -expand 1

  bind $this.entr <Up>   "$this see -"
  bind $this.entr <Down> "$this see +"

  bind $this.topl.list <ButtonRelease-1> "$this _hideList; $this see ?"
  bind $this.topl.list <Key-Return>      "$this _hideList; $this see ?"
  bind $this.topl.list <Key-Escape>      "$this _hideList"

  my iIndex -1                         ;# currently displayed
  my iListH 40                         ;# required height for list in pixels
  my wToplevel [winfo toplevel $this]  ;# toplevel of this object

  my -command {}                       ;# evaluated on value change

  -lines  5
} {} -default entr

# name: _showList
# args: -
# displays the toplevel with listbox and scrollbar at the appropriate
# position. sets a grab on the toplevel. <Escape> releases it.
metawidget proc Combobox _showList {} {
  set iX [winfo rootx  $this]
  set iY [winfo rooty  $this]
  set iW [winfo width  $this]
  set iH [winfo height $this]

  wm overrideredirect $this.topl 1
  wm deiconify $this.topl
  update
  wm geometry  $this.topl ${iW}x[my iListH]+$iX+[expr $iY+$iH]

  $this.topl.list selection clear 0 end
  $this.topl.list selection set [my iIndex]

  raise $this.topl
  focus $this.topl.list
  grab  $this.topl
}

# name: _hideList
# args: -
# releases the grab. lets the listbox toplevel disappear.
metawidget proc Combobox _hideList {} {
  grab release $this.topl
  wm withdraw  $this.topl
  focus        $this.entr
}

# name: -lines
# args: iLines: number of listbox lines
# set the number of lines for the listbox. converts it into pixels by
# multiplying it with the font height. stores pixel height in iListH.
metawidget proc Combobox -lines { iLines } {
  $this.topl.list configure -height $iLines
  my -lines $iLines

  set iFontH [font metrics [$this.topl.list cget -font] -linespace]
  my iListH [expr (1+$iFontH)*[my -lines]+2]
}

# name: -entries
# args: lEntries: list of entries to set for the Combobox
# repopulates the list with the given values in lEntries.
metawidget proc Combobox -entries { args } {
  if { ! [llength $args] } {
    $this.topl.list get 0 end
  } else {
    $this.topl.list delete 0 end
    eval $this.topl.list insert 0 [lindex $args 0]
  }
}

# name: -state
# args: sState: state to change the Combobox to
# applies normal and disabled state simply to the entry widget and the button.
# for restricted state, the two bindings prevent input to the entry field.
# only values from the list can be set with cursor up/down.
metawidget proc Combobox -state { sState } {
  my -state [mkw.complete $sState {normal disabled restricted}]

  switch [my -state] {
    normal - disabled {
      bind $this.entr <Key> {}
      bind $this.entr <KeyRelease> {}
      $this.entr config -state $sState
      $this.entr.arrb config -state $sState
    }
    restricted {
      bind $this.entr <Key> "$this.entr config -state disabled"
      bind $this.entr <KeyRelease> "$this.entr config -state normal"
      $this.entr config -state normal
      $this.entr.arrb config -state normal
      see 0
    }
  }
}

# name: see
# args: iIndex: index of the value list to display
# if index is not given, returns the index of the currently displayed
# value, or -1 if the value is not part of the value list.
# if index is + or -, the next resp. previous value in the value list is
# set. for index ?, the listbox' currently selected element is taken.
# if index is an integer, the corresponding element from the value list
# is displayed. any defined command is evaluated, if the value in
# the entry field has changed.
metawidget proc Combobox see { {iIndex {}} } {
  if { $iIndex == {} } {
    return [lsearch -exact [$this.topl.list get 0 end] [$this.entr get]]
  }

  set iLen [llength [$this.topl.list get 0 end]]
  set iOldIndex [my iIndex]

  if { $iIndex == "?" } {
    my iIndex [$this.topl.list curselection]
  } elseif { $iIndex == "+" && [my iIndex] < [expr $iLen-1] } {
    my iIndex [expr [my iIndex]+1]
  } elseif { $iIndex == "-" && [my iIndex] > 0 } {
    my iIndex [expr [my iIndex]-1]
  } elseif { $iIndex >= 0 && $iIndex < $iLen } {
    my iIndex $iIndex
  }

  if { $iOldIndex != [my iIndex] } {
    $this.entr delete 0 end
    $this.entr insert 0 [$this.topl.list get [my iIndex]]
    eval [my -command]
  }
}

metawidget command Combobox _showList _showList
metawidget command Combobox _hideList _hideList

metawidget command Combobox see       see

metawidget option  Combobox -lines   -lines
metawidget option  Combobox -entries -entries -entries
metawidget option  Combobox -state   -state
metawidget option  Combobox -command

proc test {} {
  pack [frame .upfr -border 0] -fill both
  pack [label .upfr.text -text "Enter values and press Set!"] -anchor w
  pack [entry .upfr.vals] -side left -fill x -expand 1
  pack [button .upfr.setb -pady 0 -text Set! -command {.cbox config -entries [.upfr.vals get]} ] -side right

  .upfr.vals insert end {red green blue yellow cyan magenta}

  pack [combobox .cbox -command {.outp config -text [.cbox get]}] -fill x -pady 10

  pack [label .outp -border 1 -relief sunken] -fill x
  pack [radiobutton .mod1 -text "Normal"     -value 1 -command {.cbox config -state normal}] -fill x -side left
  pack [radiobutton .mod2 -text "Restricted" -value 2 -command {.cbox config -state restricted}] -fill x -side left
  pack [radiobutton .mod3 -text "Disabled"   -value 3 -command {.cbox config -state disabled}] -fill x -side left
}

#test

