package require Metawidget

# hack! get a pointer cursor for all widgets that don't have a cursor defined
bind all <Enter> {catch {if {[%W cget -cursor] == {}} {%W configure -cursor left_ptr}}}

# name: Document
# args: args: option-value pairs
# defines icons for the window title bar. defines a default menu and creates
# all standard window elements (MS Windows look&feel). declares option values.
metawidget create Document {
  # if first object is being created, define some icons
  if { ! [ourinfo exists lAllDocs] } {
    our lAllDocs {}
    our lOnTop {}
    our wFocus {}
    our lMaximized {}; # no windows maximized yet

    # x/y-offset from top left corner
    our iOffset 0

    our pIconMin [image create bitmap -data {
      #define _width 8
      #define _height 8
      static char _bits[] = { 0x00,0x00,0x00,0x00,0x00,0x3e,0x3e,0x00 };
    }]
    our pIconMax [image create bitmap -data {
      #define _width 8
      #define _height 8
      static char _bits[] = { 0x7f,0x7f,0x41,0x41,0x41,0x41,0x7f,0x00 };
    }]
    our pIconRestore [image create bitmap -data {
      #define _width 8
      #define _height 8
      static char _bits[] = { 0x7c,0x7c,0x44,0x5f,0x5f,0x71,0x11,0x1f };
    }]
    our pIconClose [image create bitmap -data {
      #define _width 8
      #define _height 8
      static char _bits[] = { 0x00,0x63,0x36,0x1c,0x1c,0x36,0x63,0x00 };
    }]
    our pIconUntacked [image create bitmap -data {
      #define _width 8
      #define _height 8
      static char _bits[] = { 0x00,0x18,0x1d,0xf7,0xf7,0x1d,0x18,0x00 };
    }]
    our pIconTacked [image create bitmap -data {
      #define _width 8
      #define _height 8
      static char _bits[] = { 0x3c,0x18,0x3c,0x66,0xff,0x18,0x18,0x18 };
    }]
  }

  # define popup menu
  menu $this.menu -tearoff 0
  $this.menu add command -label Restore  -command "$this _restore 1"
  $this.menu add command -label Minimize -command "$this _minimize"
  $this.menu add command -label Maximize -command "$this _maximize"
  $this.menu add separator
  $this.menu add command -label Close -command "destroy $this"

  # create title bar elements
  pack [frame $this.head -border 0] -padx 2 -pady 2 -fill x
  pack [label $this.head.icon -border 0] -side left -padx 1
  pack [label $this.head.tack -border 2 -relief raised -image [our pIconUntacked]] -side left -pady 2 -padx 2
  pack [label $this.head.clob -border 2 -relief raised -image [our pIconClose]] -side right -pady 2 -padx 2
  pack [label $this.head.maxb -border 2 -relief raised -image [our pIconMax]] -side right -pady 2
  pack [label $this.head.minb -border 2 -relief raised -image [our pIconMin]] -side right -pady 2
  pack [label $this.head.text -border 0 -anchor w] -side left -fill x -expand 1
  pack [frame $this.work -border 0] -fill both -expand 1 -padx 2 -pady 2

  # option values, self explaining
  my -title $this
  my -foreground white
  my -background darkblue
  my -font [$this.head.text cget -font]
  my -minsize {100 30}
  my -icontext {}
  my -state normal
  my -image {}
  my -raiseproc {}
  my -iX {}
  my -iY {}
  my -type {}
  my -startupproc {}
  my -ontop 0

  # position and size of the window
  my -x [our iOffset]
  my -y [our iOffset]
  my -width  250
  my -height 150

  # bind menu to icon and close button
  bind $this.head.icon <1>        "$this _showMenu"
  bind $this.head.icon <Double-1> "$this.menu invoke Close"
  bind $this.head.clob <1>        "$this.menu invoke Close"
  bind $this.head.tack <1>        "$this _toggleTack"

  # increase offset for next window
  our iOffset [expr [our iOffset]+16]
  if { [expr [our iOffset]+64] > [winfo height [winfo parent $this]] } {
    our iOffset 0
  }

  # show in normal mode and put it on top
  _restore 1
  raise_

  # add it to the list of all documents
  _register
} {
  if { [my -state] == "minimized" } {
      # search for the next free icon slot from the beginning
      unour iXlast
      unour iYlast
  }
  if { [our wFocus] == $this } {
    our wFocus {}
    if {[my -raiseproc] != {}} {
        eval [my -raiseproc] \{\}
    }
  }

  our lMaximized [mkw.lshrink [our lMaximized] $this]
  if {[our lMaximized] != {}} {
      [lindex [our lMaximized] 0] _maximize
      [lindex [our lMaximized] 0] raise
  }
  our lOnTop [mkw.lshrink [our lOnTop] $this]

  # remove it from all documents
  _unregister
} -border 2 -relief raised

# name: _register
# args: o Document object to register
# adds the current object to a list. required for geometry management.
metawidget proc Document _register {} {
  our lAllDocs [linsert [our lAllDocs] end $this]
}

# name: _unregister
# args: o Document object to unregister
# removes the object from this list.
metawidget proc Document _unregister {} {
  our lAllDocs [mkw.lshrink [our lAllDocs] $this]
}

# name: _showMenu
# args: -
# displays the popup menu right under the document's icon
metawidget proc Document _showMenu {} {
  $this.menu post [winfo rootx $this.head.icon] \
    [expr [winfo height $this.head.icon] + [winfo rooty $this.head.icon]]
  focus $this.menu
  # the following is for Unix only. is a noop on Windows and Mac
  bind $this.menu <FocusOut> "$this.menu unpost"
}

# name: _toggleTack
# args: -
# toggles the stay on top tack
metawidget proc Document _toggleTack {} {
  if {[my -ontop]} {
      $this configure -ontop 0
  } else {
      $this configure -ontop 1
  }
}

# name: _move1
# args: -
# called on a <Button-1> event on the title bar. stores the current mouse
# pointer position. raises the document.
metawidget proc Document _move1 {} {
  raise_

  if { [my -state] != "minimized" } {
      my iXp0 [winfo pointerx $this]
      my iYp0 [winfo pointery $this]
  } else {
      my iXp0i [winfo pointerx $this]
      my iYp0i [winfo pointery $this]
  }
}

# name: _move2
# args: -
# called on a <B1-Motion> event on the title bar. calculates the new
# window position and places it there.
metawidget proc Document _move2 {} {
  # limit mouse to parent window
  set parent [winfo parent $this]
  set rootx [winfo rootx $parent]
  set rooty [winfo rooty $parent]
  set maxx [expr $rootx + [winfo width $parent]]
  set maxy [expr $rooty + [winfo height $parent]]
  if { [my -state] != "minimized" } {
      set iXp [winfo pointerx $this]
      set iYp [winfo pointery $this]

      if {![myinfo exists iXp0]} {
	  # we have B1-Motion without a Button-1 event
	  $this _move1
      }
      if {$iXp < $rootx} {
	  set iXp $rootx
      }
      if {$iXp > $maxx} {
	  set iXp $maxx
      }
      if {$iYp < $rooty} {
	  set iYp $rooty
      }
      if {$iYp > $maxy} {
	  set iYp $maxy
      }
      place $this -x [expr [my -x]-[my iXp0]+$iXp] -y [expr [my -y]-[my iYp0]+$iYp]
  } else {
      set iXpi [winfo pointerx $this]
      set iYpi [winfo pointery $this]

      if {$iXpi < $rootx} {
	  set iXpi $rootx
      }
      if {$iXpi > $maxx} {
	  set iXpi $maxx
      }
      if {$iYpi < $rooty} {
	  set iYpi $rooty
      }
      if {$iYpi > $maxy} {
	  set iYpi $maxy
      }
      set x [expr [my -ix]-[my iXp0i]+$iXpi]
      set y [expr [my -iy]-[my iYp0i]+$iYpi]
      place $this -x $x -y $y
      # adjust the saved icon position
      my -iX $x
      my -iY $y
      unour iXlast
      unour iYlast
  }
}

# name: _move3
# args: -
# called on a <ButtonRelease-1> event on the title bar. stores the actual
# window position in some private variables.
metawidget proc Document _move3 {} {
  if { [my -state] != "minimized" } {
      my -x [winfo x $this]
      my -y [winfo y $this]
  } else {
      my -ix [winfo x $this]
      my -iy [winfo y $this]
  }
}

# name: _resize1
# args: -
# called on a <1> event on one of the borders. stores the current mouse
# position. depending on the mouse position (at an edge or not), the
# resize mode is determined.
metawidget proc Document _resize1 {} {
  raise_

  my iWMin [lindex [my -minsize] 0]
  my iHMin [lindex [my -minsize] 1]

  my iXMax [expr [my -x]+[my -width] -[my iWMin]]
  my iYMax [expr [my -y]+[my -height]-[my iHMin]]

  my iXp0 [winfo pointerx $this]
  my iYp0 [winfo pointery $this]

  # allow max. 16 pixel for edge detection
  set iXEdge [expr [my -width] >>1]
  set iYEdge [expr [my -height]>>1]
  if { $iXEdge > 16 } { set iXEdge 16 }
  if { $iYEdge > 16 } { set iYEdge 16 }

  set iXd [winfo rootx $this]
  set iYd [winfo rooty $this]

  # detect if mouse is on one of the edges (x coordinate)
  if { [my iXp0] < [expr $iXd+$iXEdge] } {
    my iXMove -1           ;# left edge
  } elseif { [my iXp0] > [expr $iXd+[my -width]-$iXEdge] } {
    my iXMove 1            ;# right edge
  } else {
    my iXMove 0            ;# between edge
  }

  # detect if mouse is on one of the edges (y coordinate)
  if { [my iYp0] < [expr $iYd+$iYEdge] } {
    my iYMove -1           ;# upper edge
  } elseif { [my iYp0] > [expr $iYd+[my -height]-$iYEdge] } {
    my iYMove 1            ;# lower edge
  } else {
    my iYMove 0            ;# between edge
  }
}

# name: _resize2
# args: -
# called on a <B1-Motion> event on the borders. calculates the new
# Document sizes and position and places it depending on the resize mode.
metawidget proc Document _resize2 {} {
  set iXp [winfo pointerx $this]
  set iYp [winfo pointery $this]

  if {![myinfo exists iXMove]} {
      # didn't see a Button-1 event
      $this _resize1
  }
  if { [my iXMove] == -1 } {
    set iX [expr [my -x]-[my iXp0]+$iXp]
    if { $iX > [my iXMax] } { set iX [my iXMax] }

    set iW [expr [my -width]+[my iXp0]-$iXp]
    if { $iW < [my iWMin] } { set iW [my iWMin] }

    place $this -x $iX -width $iW
  } elseif { [my iXMove] == 1 } {
    set iW [expr [my -width]-[my iXp0]+$iXp]
    if { $iW < [my iWMin] } { set iW [my iWMin] }

    place $this -width $iW
  }

  if { [my iYMove] == -1 } {
    set iY [expr [my -y]-[my iYp0]+$iYp]
    if { $iY > [my iYMax] } { set iY [my iYMax] }

    set iH [expr [my -height]+[my iYp0]-$iYp]
    if { $iH < [my iHMin] } { set iH [my iHMin] }

    place $this -y $iY -height $iH
  } elseif { [my iYMove] == 1 } {
    set iH [expr [my -height]-[my iYp0]+$iYp]
    if { $iH < [my iHMin] } { set iH [my iHMin] }

    place $this -height $iH
  }
}

# name: _resize3
# args: -
# called on a <ButtonRelease-1> event on the borders. stores the actual
# window size and position in some private variables.
metawidget proc Document _resize3 {} {
  my -x      [winfo x      $this]
  my -y      [winfo y      $this]
  my -width  [winfo width  $this]
  my -height [winfo height $this]
}

# name: _cursor
# args: -
# called on a Motion event on one of the borders. modifies the cursor
metawidget proc Document _cursor {} {

  set iXp [winfo pointerx $this]
  set iYp [winfo pointery $this]

  # allow max. 16 pixel for edge detection
  set iXEdge [expr [my -width] >>1]
  set iYEdge [expr [my -height]>>1]
  if { $iXEdge > 16 } { set iXEdge 16 }
  if { $iYEdge > 16 } { set iYEdge 16 }

  set iXd [winfo rootx $this]
  set iYd [winfo rooty $this]

  # detect if mouse is on one of the edges (x coordinate)
  if { $iXp < [expr $iXd+$iXEdge] } {
    set iXMove -1           ;# left edge
  } elseif { $iXp > [expr $iXd+[my -width]-$iXEdge] } {
    set iXMove 1            ;# right edge
  } else {
    set iXMove 0            ;# between edge
  }

  # detect if mouse is on one of the edges (y coordinate)
  if { $iYp < [expr $iYd+$iYEdge] } {
    set iYMove -1           ;# upper edge
  } elseif { $iYp > [expr $iYd+[my -height]-$iYEdge] } {
    set iYMove 1            ;# lower edge
  } else {
    set iYMove 0            ;# between edge
  }
  if { $iXMove == -1 && $iYMove == -1} {
    # upper left
    $this configure -cursor "top_left_corner"
  } elseif { $iXMove == 1 && $iYMove == -1} {
    # upper right
    $this configure -cursor "top_right_corner"
  } elseif { $iXMove == -1 && $iYMove == 1} {
    # lower left
    $this configure -cursor "bottom_left_corner"
  } elseif { $iXMove == 1 && $iYMove == 1} {
    # lower right
    $this configure -cursor "bottom_right_corner"
  } elseif { $iXMove == -1 } {
    # left
    $this configure -cursor "left_side"
  } elseif { $iXMove == 1 } {
    # right
    $this configure -cursor "right_side"
  } elseif { $iYMove == -1 } {
    # upper
    $this configure -cursor "top_side"
  } elseif { $iYMove == 1 } {
    # lower
    $this configure -cursor "bottom_side"
  } else {
    # not on border
    $this configure -cursor ""
  }
}

# name: _withdraw
# args: -
# simply lets the Document disappear.
metawidget proc Document _withdraw {} {
  our lMaximized [mkw.lshrink [our lMaximized] $this]
  if {[our lMaximized] != {}} {
      [lindex [our lMaximized] 0] _maximize
      [lindex [our lMaximized] 0] raise
  }

  pack forget $this
  place forget $this
  my -state withdrawn
}

# name: _maximize
# args: -
# maximizes the document. exchanges the icon for the maximize button.
# deletes bindings for resize and move.
metawidget proc Document _maximize {} {
  # if it was minimized or withdrawn: set to default
  if { [my -state] != "normal" } {
    _restore -1
  }

  # remove if maximized already
  our lMaximized [mkw.lshrink [our lMaximized] $this]
  if {[our lMaximized] != {}} {
      # another window is maximized, hide it
      pack forget [lindex [our lMaximized] 0]
  }
  # place at the head of the maximized list
  our lMaximized [eval list $this [our lMaximized]]
  pack $this -fill both -expand 1
  update idletasks
  $this.head.maxb configure -image [our pIconRestore]
  $this.menu entryconf Maximize -state disabled
  ::${this}::$this config -border 0

  # delete bindings for resizing
  bind $this <1> {}
  bind $this <B1-Motion> {}
  bind $this <ButtonRelease-1> {}

  # delete cursor binding
  bind $this <Motion> {}

  # delete bindings for moving
  bind $this.head.text <B1-Motion> {}
  bind $this.head.text <ButtonRelease-1> {}

  # change bindings for maximize button and title bar
  bind $this.head.text <Double-1> "$this _restore 2"
  bind $this.head.maxb <1> "$this raise; $this _restore 2"

  my -state maximized
}

# name: _minimize
# args: -
# minimizes the document. exchanges the icon for the minimize button.
# deletes bindings for resize.
metawidget proc Document _minimize {} {
  our lMaximized [mkw.lshrink [our lMaximized] $this]
  if {[our lMaximized] != {}} {
      [lindex [our lMaximized] 0] _maximize
      [lindex [our lMaximized] 0] raise
  }

  # if it was maximized or withdrawn: set to default
  if { [my -state] != "normal" } {
    _restore; # leave other windows alone
  }

  # withdraw the document's working area
  pack forget $this.work
  $this.head.minb configure -image [our pIconRestore]
  $this.menu entryconf Minimize -state disabled
  pack forget $this.head.tack
  our lOnTop [mkw.lshrink [our lOnTop] $this]

  # eventually change title text
  if { [my -icontext] != {} } {
    $this.head.text configure -text [my -icontext]
  }

  if {![ourinfo exists iIconHeight]} {
      # determine the actual height of an icon
      update idletasks; # make sure it has been calculated
      our iIconHeight [expr [winfo reqheight $this.head] + 8]
  }
  place $this -width 128 -height [our iIconHeight]

  # delete bindings for resizing
  bind $this <1> {}
  bind $this <B1-Motion> {}
  bind $this <ButtonRelease-1> {}

  # delete cursor binding
  bind $this <Motion> {}

  # change bindings for minimize button and title bar
  bind $this.head.text <Double-1> "$this _restore 1"
  bind $this.head.minb <1> "$this raise; $this _restore 1"

  my -state minimized
  lower $this
  if {[my -iX] != {}} {
      # was iconified already
      place $this -x [my -iX] -y [my -iY]
      after idle "$this _move3"
  } else {
      after idle "::Document::Arrange icons $this"
  }
}

# name: _restore
# args: iMax - 1 if this window should join maximized windows, if present
#	       2 if this is a max restore (should unmax other windows)
#	       0 if this is a restore to get defaults
#             -1 if this window should not have a border
# restores a Document from any of the other states. assigns the regular
# bindings to all widgets.
metawidget proc Document _restore {{iMax 0}} {
  if {$iMax == 1} {
      if {[our lMaximized] != {}} {
          # one or more other windows is maximized. maximize this
	  _restore -1; # dont want border
          _maximize
          return
      }
   }
   # remove from maximized list if present
   our lMaximized [mkw.lshrink [our lMaximized] $this]
   if {$iMax == 2} {
      # unmaximize all the others
      foreach w [our lMaximized] {
          our lMaximized [mkw.lshrink [our lMaximized] $w]
          $w _restore
      }   
  }
  # show the working area and place Document on last known position
  pack  $this.work -fill both -expand 1 -padx 2 -pady 2
  place $this -x [my -x] -y [my -y] -width [my -width] -height [my -height]
  pack $this.head.tack -side left -after $this.head.icon -pady 2 -padx 2
  $this configure -ontop [my -ontop]

  # restore icons and menu entries for all title bar elements
  $this.head.minb configure -image [our pIconMin]
  $this.head.maxb configure -image [our pIconMax]
  $this.head.text configure -text [my -title]
  $this.menu entryconf Minimize -state normal
  $this.menu entryconf Maximize -state normal
  if {$iMax >= 0} {
      ::${this}::$this config -border 2
  }

  # restore resize bindings
  bind $this <1>               "$this _resize1"
  bind $this <B1-Motion>       "$this _resize2"
  bind $this <ButtonRelease-1> "$this _resize3"

  # set up the cursor binding
  bind $this <Motion>           "$this _cursor"

  # restore move bindings
  bind $this.head.text <1>               "$this _move1"
  bind $this.head.text <B1-Motion>       "$this _move2"
  bind $this.head.text <ButtonRelease-1> "$this _move3"

  # restore maximize and minimize bindings
  bind $this.head.maxb <1>        "$this raise; $this _maximize"
  bind $this.head.minb <1>        "$this _minimize"
  bind $this.head.text <Double-1> "$this _maximize"

  if { [my -state] == "minimized" } {
      # search for the next free icon slot from the beginning
      unour iXlast
      unour iYlast
  }
  my -state normal
}

# name: -x et al.
# args: iX, iY, iWidth iHeight: geometry values
# sets a new size or position of the Document window.
metawidget proc Document -x { iX } {
  my -x $iX
  _restore 1
}
metawidget proc Document -y { iY } {
  my -y $iY
  _restore 1
}
metawidget proc Document -ix { iX } {
  my -ix $iX
}
metawidget proc Document -iy { iY } {
  my -iy $iY
}
metawidget proc Document -iX { iX } {
  my -iX $iX
}
metawidget proc Document -iY { iY } {
  my -iY $iY
}
metawidget proc Document -width { iWidth } {
  my -width $iWidth
  _restore 1
}
metawidget proc Document -height { iHeight } {
  my -height $iHeight
  _restore 1
}

# name: -font
# args: sFont: font to set
# option handler. sets the title font.
metawidget proc Document -font { sFont } {
  my -font $sFont
  $this.head.text configure -font $sFont
}

# name: -image
# args: sImage: image to set
# option handler. sets the image in the title bar.
metawidget proc Document -image { sImage } {
  my -image $sImage
  $this.head.icon configure -image $sImage
}

# name: -title
# args: sTitle: title to set
# option handler. sets title text.
metawidget proc Document -title { sTitle } {
  my -title $sTitle
  if {[$this cget -state] != "minimized"} {
      $this.head.text configure -text $sTitle
  }
}

# name: -icontext
# args: sTitle: title to set
# option handler. sets icon text.
metawidget proc Document -icontext { sTitle } {
  my -icontext $sTitle
  if {[$this cget -state] == "minimized"} {
      $this.head.text configure -text $sTitle
  }
}

# name: -state
# args: sState: new Document state
# option handler. sets the state for the document. the variable -state
# is set in the respective (private) member functions.
metawidget proc Document -state { sState } {
  switch -- [mkw.complete $sState {normal minimized maximized withdrawn}] {
    normal    { _restore 1 } 
    minimized { _minimize }
    maximized { _maximize }
    withdrawn { _withdraw }
  }
}

# name: -raiseproc
# args: sRaiseproc: command to execute when raised
# option handler. sets command to execute when raised.
metawidget proc Document -raiseproc { sRaiseproc } {
  my -raiseproc $sRaiseproc
  raise_ ; # Ugh. Need a reaise to get the initial raiseproc called
}

# name: -type
# args: sType: type of document
# option handler. sets document type
metawidget proc Document -type { sType } {
  my -type $sType
}

# name: -startupproc
# args: sProc: the command executed to generate the editor's startup code
metawidget proc Document -startupproc { sProc } {
  my -startupproc $sProc
}

# name: -ontop
# args: iFlag: non zero if the window should stay on top
metawidget proc Document -ontop { iFlag } {
  our lOnTop [mkw.lshrink [our lOnTop] $this]
  my -ontop $iFlag
  if {$iFlag} {
      # window should stay on top
      $this.head.tack configure -image [our pIconTacked]
      # put on the top of the on-top list
      our lOnTop [mkw.lextend [our lOnTop] $this]
      $this raise
  } else {
      # window should not stay on top, but below on top windows
      if {[our lOnTop] != {}} {
          lower $this [lindex [our lOnTop] 0]
      }
      $this.head.tack configure -image [our pIconUntacked]
  }
}

# name: startup
# args: none
# returns a command to re-create the document
metawidget proc Document startup { } {
  if {![string equal [my -startupproc] {}]} {
      set result [eval [my -startupproc] $this]
  } else {
      set result ""
  }
  return $result
}

# name: menu
# args: args: arguments for the regular menu command.
# simply passes all arguments to the popup menu.
metawidget proc Document menu_ { args } {
  eval $this.menu $args
}

# name: lower
# args: -
# draws the title bar in grey.
metawidget proc Document lower_ {} {
  $this.head configure -bg gray50
  $this.head.icon configure -bg gray50
  $this.head.text configure -bg gray50 -fg gray
}

# name: raise
# args: -
# puts the Document on top and draws the title bar in different colors.
metawidget proc Document raise_ {} {
  catch { [our wFocus] lower }
  our wFocus $this

  if {[our lMaximized] != {} && [lindex [our lMaximized] 0] != $this} {
      _maximize
  }
  $this.head configure -bg [my -background]
  $this.head.icon configure -bg [my -background]
  $this.head.text configure -bg [my -background] -fg [my -foreground]

  if {[my -ontop] || [our lOnTop] == {}} {
      if {[my -ontop]} {
	  # move to the top of the on-top list
	  our lOnTop [mkw.lshrink [our lOnTop] $this]
	  our lOnTop [mkw.lextend [our lOnTop] $this]
      }
      raise $this
  } else {
      # raise below the bottom on top window
      lower $this [lindex [our lOnTop] 0]
  }
  if {[my -raiseproc] != {}} {
      eval [my -raiseproc] $this
  }
}

# name: pack
# args: sWindow: widget to be packed
#       args: option-value pairs
# redirects packing to the work area widget.
metawidget proc Document pack_ { sWindow args } {
  eval pack $sWindow -in $this.work $args
}

# name: grid
# args: sWindow: widget to be gridded
#       args: option-value pairs
# redirects gridding to the work area widget.
metawidget proc Document grid_ { sWindow args } {
  eval grid $sWindow -in $this.work $args
}

# name: place
# args: sWindow: widget to be placed
#       args: option-value pairs
# redirects placing to the work area widget.
metawidget proc Document place_ { sWindow args } {
  eval place $sWindow -in $this.work $args
}

# name: Documents
# args: none
# returns all the documents for this window
metawidget proc Document Documents { } {
  variable lAllDocs
  if {![info exists lAllDocs]} {
      return {}
  }
  return $lAllDocs
}

# name: Arrange
# args: sHow: arrange mode
# static member function. resizes and repositions all Document objects
# in one of several ways.
metawidget proc Document Arrange { sHow { single {}}} {
  variable lAllDocs
  variable lOnTop
  variable iXlast
  variable iYlast
  variable iIconHeight

  if {![info exists lAllDocs]} {
    return				      ;# no documents yet
  }

  set lDocs {}                                ;# verify list contents
  foreach lDoc $lAllDocs {
    if { ! [winfo exists $lDoc] } continue
    lappend lDocs $lDoc
  }

  set iDocs [llength $lDocs]
  if { $iDocs == 0 } return                   ;# leave if there is nothing

  set iW [winfo width  [winfo parent [lindex $lDocs 0]]]
  set iH [winfo height [winfo parent [lindex $lDocs 0]]]

  set sHow [mkw.complete $sHow {horizontally vertically cascade tile icons maximize minimize}]
  switch $sHow {
    horizontally {
      set iStep [expr $iH/$iDocs]
      for { set i 0 } { $i < $iDocs } { incr i } {
        set sDoc [lindex $lDocs $i]
        place $sDoc -x 0 -y [expr $i*$iStep] -width $iW -height $iStep
      }
    }

    vertically {
      set iStep [expr $iW/$iDocs]
      for { set i 0 } { $i < $iDocs } { incr i } {
        set sDoc [lindex $lDocs $i]
        place $sDoc -x [expr $i*$iStep] -y 0 -width $iStep -height $iH
      }
    }

    cascade {
      set iStep 24
      for { set i 0 } { $i < $iDocs } { incr i } {
        set sDoc [lindex $lDocs $i]
        place $sDoc -x [expr $i*$iStep] -y [expr $i*$iStep] -width [expr $iW*2/3] -height [expr $iH*2/3]
	raise $sDoc
      }
      $sDoc raise
    }

    tile {
      for { set i 0 } { $i <= 100 } { incr i } {
        set iHor [expr ($i+3)/2]
        set iVer [expr ($i+2)/2]
        if { [expr $iHor*$iVer] >= $iDocs } break
      }

      set iXStep [expr $iW/$iHor]
      set iYStep [expr $iH/$iVer]
      for { set i 0 } { $i < $iDocs } { incr i } {
        set sDoc [lindex $lDocs $i]
        place $sDoc -x [expr ($i%$iHor)*$iXStep] -y [expr ($i/$iHor)*$iYStep] -width $iXStep -height $iYStep
      }
    }

    icons {
      if {$single != {} && [info exists iXlast]} {
	  # use last starting point for a single icon
	  set iXd $iXlast
	  set iYd $iYlast
      } else {
          set iXd 0
          set iYd 0
      }
      foreach sDoc $lDocs {
        if { [$sDoc cget -state] != "minimized" } continue

        set iWd [winfo width $sDoc]
	if {$iYd == 0} {
	    if {![info exists iIconHeight]} {
		# determine the actual height of an icon
		update idletasks; # make sure it has been calculated
		set iIconHeight [winfo height $sDoc]
	    }
	    # move to the bottom of the screen
	    set iYd [expr $iH - $iIconHeight]
	}
        if { [expr $iXd+$iWd] > $iW } {
          set iXd 0
          incr iYd [expr {-$iIconHeight}]
        }
	if {$single != {}} {
	    # want to place a single icon in an empty spot
	    set found 0
            foreach sDocm $lDocs {
                if { [$sDocm cget -state] != "minimized" } continue
		if { $sDocm == $single } continue

		set info [place info $sDocm]
		set ix [$sDocm cget -iX]
		set iy [$sDocm cget -iY]
		if { $ix == $iXd && $iy == $iYd } {
		    # an icon is here
		    set found 1
		    break
		}
	    }
	    if {$found} {
		# spot was occupied
                incr iXd $iWd
		continue
	    }
	    # place new icon and return
            place $single -x $iXd -y $iYd
	    $single configure -iX $iXd -iY $iYd
	    set iXlast $iXd
	    set iYlast $iYd
            after idle "$single _move3"
	    return
	} else {
            place $sDoc -x $iXd -y $iYd
	    $sDoc configure -iX $iXd -iY $iYd
	    set iXlast $iXd
	    set iYlast $iYd
            after idle "$sDoc _move3"
	}
        incr iXd $iWd
      }
    }

    maximize {
      foreach sDoc $lDocs {
        place $sDoc -x 0 -y 0 -width $iW -height $iH
      }
    }

    minimize {
      foreach sDoc $lDocs {
        $sDoc _minimize
      }
    }
  }

  # get new window geometry and restore default behaviour
  if { $sHow != "minimize" && $sHow != "icons" } {
    foreach sDoc $lDocs {
      after idle "$sDoc _resize3; $sDoc _restore 1"
    }
  }
}

metawidget command Document _showMenu _showMenu
metawidget command Document _toggleTack _toggleTack
metawidget command Document _move1    _move1
metawidget command Document _move2    _move2
metawidget command Document _move3    _move3
metawidget command Document _resize1  _resize1
metawidget command Document _resize2  _resize2
metawidget command Document _resize3  _resize3
metawidget command Document _cursor   _cursor
metawidget command Document _withdraw _withdraw
metawidget command Document _maximize _maximize
metawidget command Document _minimize _minimize
metawidget command Document _restore  _restore

metawidget command Document startup   startup
metawidget command Document menu      menu_
metawidget command Document lower     lower_
metawidget command Document raise     raise_
metawidget command Document pack      pack_
metawidget command Document grid      grid_
metawidget command Document place     place_
metawidget command Document Documents Documents
metawidget command Document Arrange   Arrange

metawidget option  Document -x          -x
metawidget option  Document -y          -y
metawidget option  Document -ix         -ix
metawidget option  Document -iy         -iy
metawidget option  Document -iX         -iX
metawidget option  Document -iY         -iY
metawidget option  Document -width      -width
metawidget option  Document -height     -height
metawidget option  Document -font       -font
metawidget option  Document -image      -image
metawidget option  Document -title      -title
metawidget option  Document -state      -state
metawidget option  Document -foreground
metawidget option  Document -background
metawidget option  Document -minsize
metawidget option  Document -icontext   -icontext
metawidget option  Document -raiseproc  -raiseproc
metawidget option  Document -type       -type
metawidget option  Document -startupproc -startupproc
metawidget option  Document -ontop       -ontop

proc test {} {
  pack [frame .w -back darkgray] -fill both -expand 1
  document .w.d1
  document .w.d2
  document .w.d3
}

#test
