package require Metawidget

# name: Editor
# args: args
# Creates an Editor widget
metawidget create Editor {
  if {![ourinfo exists iCount]} {
      # set up the editor
      our iCount 0
  }
  scrollbar $this.scrv -command "$this yview"
  scrollbar $this.scrh -command "$this xview" -orient horiz
  frame $this.text -bd 2 -relief sunken
  grid columnconf $this.text 1 -weight 1
  grid rowconf    $this.text 0 -weight 1
  text   $this.text.t -bg white \
      -yscroll "$this.scrv set" -xscroll "$this.scrh set" -bd 0
  grid $this.text.t -column 1 -row 0 -sticky nsew
  # the resize box
  frame $this.box
  grid columnconf $this 0 -weight 1
  grid rowconf    $this 0 -weight 1
  grid $this.text -column 0 -row 0 -sticky nsew

  bindtags $this.text.t "$this.text.t $this Editor . all"
  foreach ev [bind Text] {
      set action [bind Text $ev]
      regsub -all {%W} $action {[winfo parent [winfo parent %W]]} action
      bind Editor $ev $action
  }
  # make sure the text widget keeps focus
  bind $this.text.t <FocusOut> "if \{\[focus] == \"$this\"\} \{focus $this.text.t\}"
  bind $this <FocusIn> "focus $this.text.t"
  bind $this.text.t <FocusIn> "$this checktime"

  my -contents {}
  my -line 1
  my -column 0
  my -new 0
  my -openproc {}
  my -saveproc {}
  my -changeproc {}
  my -filewriteproc {}
  my -ignorechanges 0
  my -scrollbar off
  my -positionproc {}
  my -infowidth 0
  our -autoreload 0
} {
} -default text.t

# name: _setscroll
# args: -
# if -scrollbar is auto, displays or hides the two scrollbars.
# uses the fraction values the scrollbars would be set to.
# anything > 0 and < 1 would indicate that the scrollbar should
# be displayed.
metawidget proc Editor _setscroll {} {
  if { [my -scrollbar] != "auto" } return

  my -scrollbar {}
  update idletasks
  set both 1
  mkw.lassign [$this.scrv get] fFrac1 fFrac2
  if { $fFrac1 > 0 || $fFrac2 < 1 } {
    grid $this.scrv -column 2 -row 0 -sticky ns
  } else {
    set both 0
    grid forget $this.scrv
  }

  mkw.lassign [$this.scrh get] fFrac1 fFrac2
  if { $fFrac1 > 0 || $fFrac2 < 1 } {
    grid $this.scrh -column 0 -row 1 -sticky we
  } else {
    set both 0
    grid forget $this.scrh
  }
  if {$both} {
      grid $this.box -column 2 -row 1 -sticky nsew
      raise $this.box
  }
  update idletasks
  my -scrollbar auto
}

# name: _redraw
# args: -
# updates the scrollbars after becoming idle. any
# old job is deleted, so exactly one redraw is done.
metawidget proc Editor _redraw {} {
  catch { cancel [my hJob] }
  my hJob [after idle $this _setscroll]
}

# name: -scrollbar
# args: sScrollbar: on, off or auto
# option member. lets the scrollbars either appear (on) or disappear (off).
# for auto, _setscroll takes care of that. the binding is needed to
# update the scrollbars when the window is resized.
metawidget proc Editor -scrollbar { sScrollbar } {
  my -scrollbar [mkw.complete $sScrollbar {auto on off}]

  switch [my -scrollbar] {
    auto {
      bind $this.text.t <Configure> "$this _setscroll"
      _redraw
    }
    on {
      bind $this.text.t <Configure> {}
      grid $this.scrv -column 2 -row 0 -sticky ns
      grid $this.scrh -column 0 -row 1 -sticky we
      grid $this.box -column 2 -row 1 -sticky nsew
      raise $this.box
    }
    off {
      bind $this.text.t <Configure> {}
      grid forget $this.scrv $this.scrh $this.box
    }
  }
}

# name: -positionproc
# args: sPosProc: proc to execute changed insert position
metawidget proc Editor -positionproc { sPosProc } {
  my -positionproc $sPosProc
}

# name: -infowidth
# args: iWidth: width of the info widget
metawidget proc Editor -infowidth { iWidth } {
  if {$iWidth < 0} {
      set iWidth 0
  }
  if {$iWidth} {
      if {![my -infowidth]} {
          # need to create an info widget
          text $this.text.i -fg blue -bg grey -bd 0 -width $iWidth \
	      -highlightthickness 0 -wrap none -state disabled
          bindtags $this.text.i "$this.text.i . all"
          grid $this.text.i -column 0 -row 0 -sticky nsew
      } else {
          $this.text.i configure -width $iWidth
      }
      $this.text.i configure -cursor left_ptr
  } else {
      # get rid of the info widget
      ::destroy $this.text.i
  }
  my -infowidth $iWidth
}

# name: xview
# args: args passed to 'text xview'
# wrapper around 'text xview'. does a _redraw
metawidget proc Editor xview { args } {
  set result [eval $this.text.t xview $args]
  _redraw
  return $result
}

# name: yview
# args: args passed to 'text yview'
# wrapper around 'text yview'. does a _redraw
metawidget proc Editor yview { args } {
  set result [eval $this.text.t yview $args]
  catch {eval $this.text.i yview $args}
  _redraw
  return $result
}

# name: see
# args: pIndex position in text
# wrapper around 'text see'. positions info
metawidget proc Editor see { pIndex } {
  $this.text.t see $pIndex
  if {[my -infowidth]} {
      set pIndex [$this.text.t index $pIndex]
      $this.text.i see $pIndex
  }
}

# name: focus
# args: none
# set focus to text widget
metawidget proc Editor focus { } {
  ::focus $this.text.t
  if {[my -positionproc] != {}} {
      eval [my -positionproc] [$this index insert]
  }
}

# name: destroy
# args: sContainer - the container widget
# destroy an editor object
metawidget proc Editor destroy { sContainer } {
  if {[close]} {
      ::destroy $sContainer
  }
}

# name: close
# args: none
# close an editor object
metawidget proc Editor close {} {
  set name [my -contents]
  if {[string equal $name {}]} {
      # nothing here
      return 1
  }
  if {[llength [our lFiles($name)]] == 1} {
      if {![$this cget -ignorechanges] && [our iFileModified($name)]} {
          set result [tk_messageBox \
	      -parent $this \
	      -icon question \
	      -message "Save changes to \"$name\"?" \
	      -default yes \
	      -type yesnocancel]
	  if {$result == "cancel"} {
	      return 0
	  }
	  if {$result == "yes" && [save] == {}} {
  	      return 0
	  }
      }
  }
  if {[llength [our lFiles($name)]] == 1} {
      # this is the last edit view

      if {[my -changeproc] != {}} {
	  eval [my -changeproc] closing
      }
      our lFiles($name) {}
      unour lFiles($name)
      unour iFileNoUndo($name)
      unour lFileUndo($name)
      unour lFileRedo($name)
      unour iFileFake($name)
      unour iFileModified($name)
      unour iFileMtime($name)
  } else {
      our lFiles($name) [mkw.lshrink [our lFiles($name)] $this]
  }
  my -contents {}
  return 1
}

# name: open
# args: none
# open an editor object
metawidget proc Editor open {{name {}}} {
  # close any existing contents
  close
  if {[string equal $name {}]} {
      set name [my -contents]
  } else {
      # we were passed a name
      our iFileFake($name) 0
  }
  if {[string equal $name {}]} {
      # fabricate a name
      set name Unknown[our iCount]
      our iCount [expr {[our iCount] + 1}]
      our iFileFake($name) 1
  } 
  my -contents $name
  if {[ourinfo exists lFiles($name)]} {
      # this is another view
      our lFiles($name) [mkw.lextend [our lFiles($name)] $this]
      # copy the contents
      set source [lindex [our lFiles($name)] 0]
      $this.text.t insert insert [$source.text.t get 1.0 end]
      # copy marks
      set marks [$source.text.t dump -mark 1.0 end]
      foreach "type mark index" $marks {
 	  $this.text.t mark set $mark $index
      }
      set tags [$source.text.t tag names]
      foreach tag $tags {
          # copy all tags
	  foreach "begin end" [$source.text.t tag ranges $tag] {
	      $this.text.t tag add $tag $begin $end
	  }
      }
      # in the same state
      $this configure -state [$source cget -state]
      if {[my -infowidth] && [$source cget -infowidth]} {
	  $this.text.i configure -state normal
	  $this.text.i delete 1.0 end
	  $this.text.i insert insert [$source.text.i get 1.0 "end - 1c"]
	  $this.text.i configure -state disabled
	  set images [$source.text.i image names]
	  foreach image $images {
	      # copy all images
	      set index [$this.text.i image create [$source.text.i index $image] -name $image]
	      foreach info [$source.text.i image configure $image] {
		  set option [lindex $info 0]
		  set value [lindex $info 4]
		  $this.text.i image configure $index $option $value
	      }
	  }
	  set tags [$source.text.i tag names]
	  foreach tag $tags {
	      # copy all tags
	      foreach "begin end" [$source.text.i tag ranges $tag] {
	          $this.text.i tag add $tag $begin $end
		  foreach sequence [$source.text.i tag bind $tag] {
	              $this.text.i tag bind $tag $sequence [$source.text.i tag bind $tag $sequence]
		  }
	      }
	  }
      }
  } else {
      # this is a new view
      our lFiles($name) $this
      our iFileNoUndo($name) 0
      our lFileUndo($name) unchanged
      our lFileRedo($name) {}
      our iFileModified($name) 0
      $this.text.t delete 1.0 end
      if {[my -infowidth]} {
	  $this.text.i configure -state normal
	  $this.text.i delete 1.0 end
	  $this.text.i configure -state disabled
      }

      if {![my -new]} {
	  # get the contents
	  if {[my -openproc] != {}} {
	      eval [my -openproc] $this.text.t
	  } else {
      	      # use file open as default
	      if {[catch {::open $name r} fd]} {
              # error opening file
              tk_messageBox \
  	          -parent $this \
	          -icon error \
	          -message "$fd" \
		  -default ok \
	          -type ok
		  set value {}
              } else {
                  set value [read $fd]
                  ::close $fd
	      }
              $this.text.t insert 1.0 $value
          }
      }
      # get file modification time, if it exists
      if {[catch {file mtime $name} mtime]} {
          our iFileMtime($name) {}
      } else {
          our iFileMtime($name) $mtime
      }
  }
  $this.text.t mark set insert [my -line].[my -column]
  $this.text.t see end
  $this.text.t see insert
}

# name: reload
# args: none
# reload an editor object
metawidget proc Editor reload {{name {}}} {
    set name [my -contents]
    if {$name == {}} {
	return
    }
    if {[our iFileFake($name)]} {
        tk_messageBox \
	    -parent $this \
	    -icon error \
	    -message "No file currently open" \
	    -default ok \
	    -type ok
	return
    }
    if {![$this cget -ignorechanges] && [our iFileModified($name)]} {
        set result [tk_messageBox \
    	    -parent $this \
	    -icon question \
	    -message "Save changes to \"$name\"?" \
	    -default yes \
	    -type yesnocancel]
        if {$result == "cancel"} {
  	    return ""
        }
        if {$result == "yes"} {
            if {[save] == {}} {
		    return 
	    }
	}
    }
    # delete old contents in all buffers
    set insert [$this index insert]
    $this delete 1.0 end
    # get new contents
    if {[my -openproc] != {}} {
        eval [my -openproc] $this.text.t
    } else {
        # use file open as default
        if {[catch {::open $name r} fd]} {
            # error opening file
            tk_messageBox \
                -parent $this \
                -icon error \
                -message "$fd" \
	        -default ok \
                -type ok
	    set value {}
        } else {
            set value [read $fd]
            ::close $fd
        }
        $this insert 1.0 $value
    }
    $this mark set insert $insert
    $this see insert
    # get file modification time, if it exists
    if {[catch {file mtime $name} mtime]} {
        our iFileMtime($name) {}
    } else {
        our iFileMtime($name) $mtime
    }
    our iFileModified($name) 0
    # reset the undo list
    our lFileUndo($name) unchanged
    our lFileRedo($name) {}
    if {![string equal [my -changeproc] {}]} {
        eval [my -changeproc] modified 0
    }
}

# name: save
# args: none
# save an editor object
metawidget proc Editor save {} {
  set name [my -contents]
  if {[string equal $name {}]} {
      # nothing here
      return [saveas]
  }
  if {[$this cget -ignorechanges] || ![our iFileModified($name)]} {
      # not modified
      return $name
  }
  if {[our iFileFake($name)]} {
      return [saveas]
  }
  if {[my -saveproc] != {}} {
      set value [eval [my -saveproc] \{$name\} $this.text.t]
  } else {
      # use file save as default
      if {[catch {::open $name w} fd]} {
          # error opening file
	  tk_messageBox \
	      -parent $this \
	      -icon error \
	      -message "$fd" \
	       -default ok \
	      -type ok
	    return {}
      } else {
	  puts -nonewline $fd [$this.text.t get 1.0 end-1c]
	  ::close $fd
	  set value $name
	}
  }
  # update file modification time, if it exists
  if {[catch {file mtime $name} mtime]} {
      our iFileMtime($name) {}
  } else {
      our iFileMtime($name) $mtime
  }
  # notify about a file write
  if {![string equal [my -filewriteproc] {}]} {
      eval [my -filewriteproc]
  }
  our iFileModified($name) 0
  # move the undo list unchanged marker
  our lFileUndo($name) [mkw.lshrink [our lFileUndo($name)] unchanged]
  our lFileRedo($name) [mkw.lshrink [our lFileRedo($name)] unchanged]
  our lFileUndo($name) [mkw.lextend [our lFileUndo($name)] unchanged]
  if {![string equal [my -changeproc] {}]} {
      eval [my -changeproc] modified 0
  }
  return $value
}

# name: saveas
# args: none
# save an editor object with a name
metawidget proc Editor saveas {} {
  set name [my -contents]
  set file [tk_getSaveFile -initialdir [pwd] -initialfile $name]
  if {$file == ""} {
      return {}
  }
  # set the new name
  if {[catch {$this configure -contents $file} msg]} {
      # can't rename
      tk_messageBox \
          -parent $this \
	  -icon error \
	  -message "$msg" \
	  -default ok \
	  -type ok
      return 0
  }
  return [save]
}

# name: savecopyas
# args: none
# save an editor object with a name
metawidget proc Editor savecopyas {} {
  set name [my -contents]
  set file [tk_getSaveFile -initialdir [pwd] -initialfile $name]
  if {$file == ""} {
      return {}
  }
  # save the file
  if {[catch {::open $file w} fd]} {
      # error opening file
      tk_messageBox \
          -parent $this \
	  -icon error \
	  -message "$fd" \
	  -default ok \
	  -type ok
      return {}
  } else {
      puts -nonewline $fd [$this get 1.0 end-1c]
      ::close $fd
      # notify about a file write
      if {![string equal [my -filewriteproc] {}]} {
          eval [my -filewriteproc]
      }
  }
  return $file
}

# name: checktime
# args: none
# check the timestamp for an editor's contents
metawidget proc Editor checktime {} {
    set name [my -contents]
    if {$name == {}} {
	return
    }
    if {[catch {file mtime $name} mtime]} {
        set mtime {}
    }
    if {$mtime == {} && [our iFileMtime($name)] != {}} {
	set result [tk_messageBox \
	    -icon info \
	    -parent $this \
	    -message "\"$name\" has been deleted or renamed on the disk." \
	    -default ok \
	    -type ok]
	our iFileMtime($name) {}
	our iFileModified($name) 1
        if {![string equal [my -changeproc] {}]} {
            eval [my -changeproc] modified 1
        }
	return
    }
    if {([our iFileMtime($name)] != {} && [our iFileMtime($name)] < $mtime)} {
        if {![our -autoreload] || (![$this cget -ignorechanges] && [our iFileModified($name)])} {
	    # file is modified or autoreload isn't set
	    # ask if the file should be reloaded
	    set result [tk_messageBox \
	        -parent $this \
	        -icon question \
	        -message "\"$name\" has changed on the disk. Reload?" \
	        -default yes \
	        -type yesno]
	    if {$result == "no"} {
		# don't reload, but remember new time
	        our iFileMtime($name) $mtime
		return
	    }
	}
	our iFileModified($name) 0
        # reset the undo list
        our lFileUndo($name) unchanged
        our lFileRedo($name) {}
        if {![string equal [my -changeproc] {}]} {
            eval [my -changeproc] modified 0
        }
	reload
    }
}

# name: undo
# args: none
# undo the last change
metawidget proc Editor undo {} {
  variable iFileNoUndo

  set name [my -contents]
  if {$name == {}} {
      return
  }
  if {$iFileNoUndo($name)} {
      # no undo on this editor
      return
  }

  # get the undo command
  if {   [our lFileUndo($name)] == {}
      || [our lFileUndo($name)] == "unchanged"} {
      # nothing to undo
      return
  }
  set last [lindex [our lFileUndo($name)] end]
  if {$last == "unchanged"} {
      # this command causes the buffer to revert to unchanged
      set unchanged 1
      # remove the unchanged marker
      RemoveUndo $name 0
      # get the undo command
      set last [lindex [our lFileUndo($name)] end]
  } else {
      set unchanged 0
  }
  # create the redo command
  if {[lindex $last 0] == "insert"} {
      set delete 1
      set redocommand "delete [lindex $last 1]"
  } else {
      set delete 0
      set end [lindex $last 2]
      set redocommand "insert [lindex $last 1] [list [$this get [lindex $last 1] [lindex $last 2]]]"
    }
    # disable undo while we update buffers
    our iFileNoUndo($name) 1
    # update buffers
    eval $this $last
    our iFileNoUndo($name) 0
    if {$delete} {
	# the redo command is a delete
	set len [string length [lindex $last 2]]
        $this mark set insert "[lindex $last 1] + ${len}c"
	# finish the delete command
	append redocommand " [$this index insert]"
    } else {
        $this mark set insert [lindex $last 1]
    }
    # remove the last undo command
    RemoveUndo $name 0
    # add the redo command to the redo list
    AddUndo $name $redocommand 1
    if {$unchanged} {
	# put the unchanged marker on the redo list
        AddUndo $name unchanged 1
    }
}

# name: hasundo
# args: none
# is there undo info?
metawidget proc Editor hasundo {} {
  set name [my -contents]
  if {$name == {}} {
      return 0
  }
  return [expr {[llength [our lFileUndo($name)]] != 1 \
	     || [our lFileUndo($name)] != "unchanged"}]
}

# name: redo
# args: none
# redo the last undo
metawidget proc Editor redo {} {
  variable iFileNoUndo

  set name [my -contents]
  if {$name == {}} {
      return
  }
  if {$iFileNoUndo($name)} {
      # no undo on this editor
      return
  }

  # get the redo command
  if {[our lFileRedo($name)] == ""} {
      # nothing to redo yet
      return
  }
  set last [lindex [our lFileRedo($name)] end]
  if {$last == "unchanged"} {
      # this command causes the buffer to revert to unchanged
      set unchanged 1
      # remove the unchanged marker
      RemoveUndo $name 1
      # get the redo command
      set last [lindex [our lFileRedo($name)] end]
  } else {
      set unchanged 0
  }
  if {[lindex $last 0] == "insert"} {
      set delete 1
      set redocommand "delete [lindex $last 1]"
  } else {
      set delete 0
      set end [lindex $last 2]
      set redocommand "insert [lindex $last 1] [list [$this get [lindex $last 1] [lindex $last 2]]]"
    }
    # disable undo temporarily
    our iFileNoUndo($name) 1
    eval $this $last
    our iFileNoUndo($name) 0
    if {$delete} {
	set len [string length [lindex $last 2]]
        $this mark set insert "[lindex $last 1] + ${len}c"
	# finish the delete command
	append redocommand " [$this index insert]"
    } else {
        $this mark set insert [lindex $last 1]
    }
    # remove this command from the redo list
    RemoveUndo $name 1
    # add the opposite command to the undo list
    AddUndo $name $redocommand
    if {$unchanged} {
        AddUndo $name unchanged
    }
}

# name: hasredo
# args: none
# is there redo info?
metawidget proc Editor hasredo {} {
  set name [my -contents]
  if {$name == {}} {
      return 0
  }
  return [expr {[our lFileRedo($name)] != {}}]
}

# name: RemoveUndo
# args: sName - file name, sCommand - command for redo, iRedo - set if redo
# add a change to the redo or undo list
metawidget proc Editor RemoveUndo { sName iRedo } {
    variable lFiles
    variable iFileModified

    if {$iRedo} {
        set list ::Editor::lFileRedo($sName)
    } else {
        set list ::Editor::lFileUndo($sName)
    }
    upvar #0 $list undo
    # remove from list
    set undo [lreplace $undo end end]
    if {!$iFileModified($sName)} {
	# the file must have been saved before the redo
	set iFileModified($sName) 1
	foreach wid $lFiles($sName) {
            if {![string equal [my -changeproc] {}]} {
                eval [my -changeproc] modified 1
	    }
	}
    } elseif {0 && $undo == {}} {
        # file is now unmodified
        set iFileModified($sName) 0
	foreach wid $lFiles($sName) {
            if {![string equal [my -changeproc] {}]} {
                eval [my -changeproc] modified 0
	    }
	}
    }
}

# name: AddUndo
# args: sName - file name, sCommand - command for redo, iRedo - set if redo
# add a change to the redo or undo list
metawidget proc Editor AddUndo { sName sCommand {iRedo 0}} {
    variable lFiles
    variable iFileModified
    variable iFileNoUndo

    if {$iFileNoUndo($sName)} {
	# undo not enabled
        return
    }

    # get the appropriate list
    if {$iRedo} {
        set list ::Editor::lFileRedo($sName)
    } else {
        set list ::Editor::lFileUndo($sName)
    }
    upvar #0 $list undo
    set last [lindex $undo end]
    set found 0
    if {$last == {}} {
	# nothing on the undo list
	set found 1
	set undo [list $sCommand]
    } elseif {$sCommand != "unchanged" && $last != "unchanged"} {
	# check to see if we can merge undos
        set lastcmd [lindex $last 0]
        set thiscmd [lindex $sCommand 0]
        scan [lindex $last 1] "%d.%d" lastline lastcolumn
        scan [lindex $sCommand 1] "%d.%d" line column
        set samestart [expr $lastline == $line && $lastcolumn == $column]
        if {$thiscmd == "insert" && $lastcmd == "delete" && $samestart} {
	    # check for a string deleted immediately after entering
            scan [lindex $last 2] "%d.%d" line column
	    if {$column - $lastcolumn == [string length [lindex $sCommand 2]]} {
	        # the same
	        RemoveUndo $sName $iRedo
		set found 1
	    }
        }
	if {!$found} {
            # merge single character changes on the same line
            if {$lastline == $line} {
                if {$lastcmd == "insert" && $thiscmd == "insert"} {
                    # merge if single character, in before last
	            if {   [string length [lindex $sCommand 2]] == 1 \
		        && $column + 1 == $lastcolumn} {
		        # remove the last one
	                RemoveUndo $sName $iRedo
		        set sCommand "insert $line.$column [list [lindex $sCommand 2][lindex $last 2]]"
	            }
                } elseif {$lastcmd == "delete" && $thiscmd == "delete"} {
                    # merge if single character after last
                    scan [lindex $last 2] "%d.%d" lastlineend lastcolumnend
                    scan [lindex $sCommand 2] "%d.%d" line endcolumn
	            if {   $lastlineend == $line \
		        && $column + 1 == $endcolumn \
		        && $lastcolumnend == $column} {
		        # remove the last one
	                RemoveUndo $sName $iRedo
		        set sCommand "delete $lastline.$lastcolumn $line.$endcolumn"
	            }
                }
            }
        }
    }

    if {!$found} {
        lappend undo $sCommand
    }
    # get the last command on the undo list
    set last [lindex $::Editor::lFileUndo($sName) end]
    # see if the modified state changed
    if {$last == "unchanged"} {
	# the last command on the undo list is the unchanged marker
        if {$iFileModified($sName)} {
	    set iFileModified($sName) 0
	    foreach wid $lFiles($sName) {
                if {![string equal [my -changeproc] {}]} {
                    eval [my -changeproc] modified 0
	        }
	    }
        }
    }
}

# name: insert
# args: args: passed to 'text insert'. redraws the scrollbars.
metawidget proc Editor insert { args } {
  if {[$this cget -state] == "disabled"} {
        return
  }

  set start [$this.text.t index [lindex $args 0]]
  set args [lrange $args 1 end]
  set name [my -contents]
  if {$name != {}} {
      set mod [our iFileModified($name)]
      our iFileModified($name) 1
      if {   !$mod
          && ![string equal [my -changeproc] {}]
          && ![eval [my -changeproc] modified 1]} {
          # ignore insert
          return
      }
      # find the end of the insertion
      set undo "delete $start"
      eval $this.text.t insert $start $args
      set end [$this.text.t index insert]
      AddUndo $name "$undo $end"
      # copy to other text buffers
      foreach t [our lFiles($name)] {
	  if {$t == $this} {
	      # already done
	      continue
	  }
          eval $t.text.t insert $start $args
      }
      if {![string equal [my -changeproc] {}]} {
          eval [$this cget -changeproc] insert $start $end 
      }
  } else {
      eval $this.text.t insert $start $args
      set end [$this index insert]
      if {![string equal [$this cget -changeproc] {}]} {
          eval [$this cget -changeproc] insert $start $end 
      }
  }
  if {[my -positionproc] != {}} {
      eval [my -positionproc] [$this index insert]
  }
  _redraw
}

# name: quietinsert
# args: args: passed to 'text quietinsert'. like insert but no changes recorded
metawidget proc Editor quietinsert { args } {
  if {[$this cget -state] == "disabled"} {
        return
  }

  set start [$this.text.t index [lindex $args 0]]
  set args [lrange $args 1 end]
  set name [my -contents]
  if {$name != {}} {
      # find the end of the insertion
      set undo "delete $start"
      eval $this.text.t insert $start $args
      set end [$this.text.t index insert]
      # copy to other text buffers
      foreach t [our lFiles($name)] {
	  if {$t == $this} {
	      # already done
	      continue
	  }
          eval $t.text.t insert $start $args
      }
  } else {
      eval $this.text.t insert $start $args
  }
  _redraw
}

# name: tag
# args: args: passed to 'text tag'.
# control tags in the text widget
metawidget proc Editor tag { args } {
  set name [my -contents]
  if {$name != {}} {
      # copy to all info buffers
      foreach t [our lFiles($name)] {
          if {$t == $this} {
	      set result [eval $t.text.t tag $args]
	  } else {
	      eval $t.text.t tag $args
	  }
      }
  } else {
      set result [eval $this.text.t tag $args]
  }
  return $result
}

# name: infotag
# args: args: passed to 'text infotag'.
# control tags in the info widget
metawidget proc Editor infotag { args } {
  set name [my -contents]
  if {$name != {}} {
      # copy to all info buffers
      foreach t [our lFiles($name)] {
	  $t.text.i configure -state normal
          if {$t == $this} {
	      set result [eval $t.text.i tag $args]
	  } else {
	      eval $t.text.i tag $args
	  }
	  $t.text.i configure -state disabled
      }
  } else {
      $this.text.i configure -state normal
      set result [eval $this.text.i tag $args]
      $this.text.i configure -state disabled
  }
  return $result
}

# name: infoimage
# args: args: passed to 'text infoimage'.
# control images in the info widget
metawidget proc Editor infoimage { args } {
  set name [my -contents]
  if {$name != {}} {
      # copy to other info buffers
      foreach t [our lFiles($name)] {
	  $t.text.i configure -state normal
          eval $t.text.i image $args
	  $t.text.i configure -state disabled
      }
  } else {
      $this.text.i configure -state normal
      eval $this.text.i image $args
      $this.text.i configure -state disabled
  }
}

# name: infoinsert
# args: args: passed to 'text infoinsert'.
# insert into the info widget
metawidget proc Editor infoinsert { args } {
  set name [my -contents]
  if {$name != {}} {
      # copy to other info buffers
      foreach t [our lFiles($name)] {
          set yview [lindex [$t.text.t yview] 0]
	  $t.text.i configure -state normal
          eval $t.text.i insert $args
	  $t.text.i configure -state disabled
	  $t.text.i yview moveto $yview
      }
  } else {
      set yview [lindex [$this.text.t yview] 0]
      $this.text.i configure -state normal
      eval $this.text.i insert $args
      $this.text.i configure -state disabled
      $this.text.i yview moveto $yview
  }
}

# name: infoconfigure
# args: args: passed to 'text infoconfigure'.
# configure the info widget
metawidget proc Editor infoconfigure { args } {
  set name [my -contents]
  if {$name != {}} {
      # copy to other info buffers
      foreach t [our lFiles($name)] {
          eval $t.text.i configure $args
      }
  } else {
      eval $this.text.i configure $args
  }
}

# name: infoindex
# args: pIndex: passed to 'text infoindex'.
# find an index in the info widget
metawidget proc Editor infoindex { pIndex } {
  return [$this.text.i index $pIndex]
}

# name: delete
# args: args: passed to 'text delete'. redraws the scrollbars
metawidget proc Editor delete { args } {
  if {[$this cget -state] == "disabled"} {
        return
  }

  # get the actual start and end end points
  set start [$this.text.t index [lindex $args 0]]
  set end [lindex $args 1]
  if {$end != {}} {
      set end [$this.text.t index $end]
  } else {
      set end [$this.text.t index "$start + 1c"]
  }

  set name [my -contents]
  if {$name != {}} {
      set mod [our iFileModified($name)]
      our iFileModified($name) 1
      if {   !$mod
          && ![string equal [my -changeproc] {}]
          && ![eval [my -changeproc] modified 1]} {
          # ignore delete
          return
      }

      # copy to other text buffers
      AddUndo $name "insert $start [list [$this.text.t get $start $end]]"
      foreach t [our lFiles($name)] {
          eval $t.text.t delete $start $end
      }
      if {![string equal [$this cget -changeproc] {}]} {
          eval [$this cget -changeproc] delete $start $end
      }
  } else {
      eval $this.text.t delete $start $end
      if {![string equal [my -changeproc] {}]} {
          eval [my -changeproc] delete $start $end
      }
  }
  if {[my -positionproc] != {}} {
      eval [my -positionproc] [$this index insert]
  }
  _redraw
}

# name: quietdelete
# args: args: passed to 'text quietdelete'. like delete but no changes recorded
metawidget proc Editor quietdelete { args } {
  if {[$this cget -state] == "disabled"} {
        return
  }

  # get the actual start and end end points
  set start [$this.text.t index [lindex $args 0]]
  set end [lindex $args 1]
  if {$end != {}} {
      set end [$this.text.t index $end]
  } else {
      set end [$this.text.t index "$start + 1c"]
  }

  set name [my -contents]
  if {$name != {}} {
      # copy to other text buffers
      foreach t [our lFiles($name)] {
          eval $t.text.t delete $start $end
      }
  } else {
      eval $this.text.t delete $start $end
  }
  _redraw
}

# name: infodelete
# args: args: passed to 'text infodelete'.
# delete from the info widget
metawidget proc Editor infodelete { args } {
  # get the actual start and end end points
  set start [$this.text.t index [lindex $args 0]]
  set end [lindex $args 1]
  if {$end != {}} {
      set end [$this.text.i index $end]
  } else {
      set end [$this.text.i index "$start + 1c"]
  }

  set name [my -contents]
  if {$name != {}} {
      # copy to other text buffers
      foreach t [our lFiles($name)] {
          set yview [lindex [$t.text.t yview] 0]
	  $t.text.i configure -state normal
          eval $t.text.i delete $start $end
	  $t.text.i configure -state disabled
	  $t.text.i yview moveto $yview
      }
  } else {
      set yview [lindex [$this.text.t yview] 0]
      $this.text.i configure -state normal
      eval $this.text.i delete $start $end
      $this.text.i configure -state disabled
      $this.text.i yview moveto $yview
  }
}

# name: infotext
# args: none
# returns the info widget if it exists
metawidget proc Editor infotext { } {
    if {[my -infowidth]} {
	return $this.text.i
    } else {
	return {}
    }
}

# name: mark
# args: args passed to 'text mark'
# wrapper around 'text mark'. checks for insert, set and unset
# and sets or unsets in all linked buffers
metawidget proc Editor mark { args } {
  if {![string equal [lindex $args 0] "set"] &&
      ![string equal [lindex $args 0] "unset"]} {
      return [eval $this.text.t mark $args]
  }

  set name [my -contents]
  if {$name != {}} {
      # copy to other text buffers
      if {[string equal [lindex $args 0] "set"]} {
	  set mark [lindex $args 1]
	  set index [$this index [lindex $args 2]]
	  if {$mark != "insert"} {
              foreach t [our lFiles($name)] {
                  if {$t == $this} {
		      set result [eval $t.text.t mark set $mark $index]
		  } else {
		      eval $t.text.t mark set $mark $index
		  }
              }
	  } else {
                  set result [eval $this.text.t mark set $mark $index]
          }
      } else {
          foreach t [our lFiles($name)] {
              eval $t.text.t mark $args
          }
      }
  } else {
      set result [eval $this.text.t mark $args]
  }
  if {[my -positionproc] != {} && \
    [string equal [lindex $args 0] "set"] && \
    [string equal [lindex $args 1] "insert"]} {
      eval [my -positionproc] [$this.text.t index insert]
  }
  return $result 
}

# name: isfirst
# args: none
# returns 1 if this editor is the first in the shared list
metawidget proc Editor isfirst {} {
  set name [my -contents]
  if {[string equal $name {}]} {
      return 1
  }
  return [string equal [lindex [our lFiles($name)] 0] $this]
}

# name: ismodified
# args: none
# returns 1 if the contents of this editor have been modified
metawidget proc Editor ismodified {} {
  set name [my -contents]
  if {[string equal $name {}] || [my -ignorechanges]} {
      return 0
  }
  return [our iFileModified($name)]
}

# name: modified
# args: iFlag
# sets the modified flag for the editor's contents
metawidget proc Editor modified { iFlag } {
  set name [my -contents]
  if {[string equal $name {}]} {
      return
  }
  our iFileModified($name) $iFlag
  if {![string equal [my -changeproc] {}]} {
      eval [my -changeproc] modified $iFlag
  }
}

# name: noundo
# args: iFlag
# sets the noundo flag for the editor's contents
metawidget proc Editor noundo { iFlag } {
  set name [my -contents]
  if {[string equal $name {}]} {
      return
  }
  our iFileNoUndo($name) $iFlag
}

# name: -contents
# args: sContents: an edit buffer's contents
# set the contents name
metawidget proc Editor -contents { sContents } {
  set name [my -contents]
  if {[string equal $name $sContents]} {
      # not changed
      return
  }
  if {![string equal $name {}]} {
      # changing the name
      if {[ourinfo exists lFiles($name)]} {
	if {![ourinfo exists lFiles($sContents)]} {
	    our lFiles($sContents) [our lFiles($name)]
	    our iFileNoUndo($sContents) [our iFileNoUndo($name)]
	    our lFileUndo($sContents) [our lFileUndo($name)]
	    our lFileRedo($sContents) [our lFileRedo($name)]
	    our iFileFake($sContents) [our iFileFake($name)]
	    our iFileModified($sContents) [our iFileModified($name)]
	    our iFileMtime($sContents) [our iFileMtime($name)]
	} else {
	  # a pre-existing editor exists, can't do it
	  error "\"$sContents\" already exists in an open editor"
	}
	unour lFiles($name)
        unour iFileNoUndo($name)
        unour lFileUndo($name)
        unour lFileRedo($name)
	unour iFileFake($name)
	unour iFileModified($name)
	unour iFileMtime($name)
      }
  }
  my -contents $sContents
  if {$sContents != {}} {
      our iFileFake($sContents) 0
      if {![string equal [my -changeproc] {}]} {
            eval [my -changeproc] renamed
      }
  }
  return
}

# name: -line
# args: iLine: the initial line number
# set the initial line number
metawidget proc Editor -line { iLine } {
  my -line $iLine
}

# name: -column
# args: iColumn: the initial column number
# set the initial line number
metawidget proc Editor -column { iColumn } {
  my -column $iColumn
}

# name: -new
# args: iFlag: set if not a file (yet)
metawidget proc Editor -new { iFlag } {
  my -new $iFlag
}

# name: -openproc
# args: sProc: the command to get the initial contents
metawidget proc Editor -openproc { sProc } {
  my -openproc $sProc
}

# name: -saveproc
# args: sProc: the command to save the editor's contents
metawidget proc Editor -saveproc { sProc } {
  my -saveproc $sProc
}

# name: -changeproc
# args: sProc: the command executed when the editor's contents changes
metawidget proc Editor -changeproc { sProc } {
  my -changeproc $sProc
}

# name: -filewriteproc
# args: sProc: the command executed when a file is written
metawidget proc Editor -filewriteproc { sProc } {
  my -filewriteproc $sProc
}

# name: -ignorechanges
# args: iFlag: set if changes should be ignored in buffer
metawidget proc Editor -ignorechanges { iFlag } {
  my -ignorechanges $iFlag
}

# name: -autoreload
# args: iFlag: set if auto reload should be done for modified files
metawidget proc Editor -autoreload { iFlag } {
  our -autoreload $iFlag
}

metawidget command Editor _setscroll   _setscroll
metawidget command Editor delete       delete
metawidget command Editor quietdelete  quietdelete
metawidget command Editor infodelete   infodelete
metawidget command Editor infotext     infotext
metawidget command Editor insert       insert
metawidget command Editor quietinsert  quietinsert
metawidget command Editor tag          tag
metawidget command Editor infotag      infotag
metawidget command Editor infoimage    infoimage
metawidget command Editor infoinsert   infoinsert
metawidget command Editor infoconfigure infoconfigure
metawidget command Editor infoindex    infoindex
metawidget command Editor mark         mark
metawidget command Editor xview        xview
metawidget command Editor yview        yview
metawidget command Editor see          see
metawidget command Editor focus        focus
metawidget command Editor destroy      destroy
metawidget command Editor open         open
metawidget command Editor reload       reload
metawidget command Editor close        close
metawidget command Editor save         save
metawidget command Editor saveas       saveas
metawidget command Editor savecopyas   savecopyas
metawidget command Editor checktime    checktime
metawidget command Editor undo         undo
metawidget command Editor hasundo      hasundo
metawidget command Editor redo         redo
metawidget command Editor hasredo      hasredo
metawidget command Editor mark         mark
metawidget command Editor isfirst      isfirst
metawidget command Editor ismodified   ismodified
metawidget command Editor modified     modified
metawidget command Editor noundo       noundo

metawidget option  Editor -scrollbar    -scrollbar
metawidget option  Editor -positionproc -positionproc
metawidget option  Editor -infowidth    -infowidth
metawidget option  Editor -contents      -contents
metawidget option  Editor -line          -line
metawidget option  Editor -column        -column
metawidget option  Editor -new           -new
metawidget option  Editor -ignorechanges -ignorechanges
metawidget option  Editor -autoreload    -autoreload
metawidget option  Editor -openproc      -openproc
metawidget option  Editor -saveproc      -saveproc
metawidget option  Editor -changeproc    -changeproc
metawidget option  Editor -filewriteproc -filewriteproc

# test
