package require Metawidget

# name: Scrollcanvas
# args: args
# creates the widgets. the widget option -scrollbar is defined
# by calling its respective member functions.
metawidget create Scrollcanvas {
  scrollbar $this.scrv -command "$this yview"
  scrollbar $this.scrh -command "$this xview" -orient horiz
  canvas   $this.canvas -bg white \
      -yscrollcommand "$this.scrv set" -xscrollcommand "$this.scrh set"
  # the resize box
  frame $this.box
  grid columnconf $this 0 -weight 1
  grid rowconf    $this 0 -weight 1
  grid $this.canvas -column 0 -row 0 -sticky nsew

  -scrollbar off
  -contents {}
} {} -default canvas

# name: _setscroll
# args: -
# if -scrollbar is auto, displays or hides the two scrollbars.
# uses the fraction values the scrollbars would be set to.
# anything > 0 and < 1 would indicate that the scrollbar should
# be displayed.
metawidget proc Scrollcanvas _setscroll {} {
  if { [my -scrollbar] != "auto" } return

  my -scrollbar {}
  update idletasks
  set both 1
  $this.canvas config -scrollregion [$this.canvas bbox all]
  update idletasks
  mkw.lassign [$this.scrv get] fFrac1 fFrac2
  if { $fFrac1 > 0 || $fFrac2 < 1 } {
    grid $this.scrv -column 1 -row 0 -sticky ns
    raise $this.scrv
  } else {
    set both 0
    grid forget $this.scrv
  }

  mkw.lassign [$this.scrh get] fFrac1 fFrac2
  if { $fFrac1 > 0 || $fFrac2 < 1 } {
    grid $this.scrh -column 0 -row 1 -sticky we
    raise $this.scrh
  } else {
    set both 0
    grid forget $this.scrh
  }
  if {$both} {
      grid $this.box -column 1 -row 1 -sticky nsew
      raise $this.box
  }
  update idletasks
  my -scrollbar auto
}

# name: _redraw
# args: -
# updates the scrollbars after becoming idle. any
# old job is deleted, so exactly one redraw is done.
metawidget proc Scrollcanvas _redraw {} {
  catch { cancel [my hJob] }
  my hJob [after idle $this _setscroll]
}

# name: -scrollbar
# args: sScrollbar: on, off or auto
# option member. lets the scrollbars either appear (on) or disappear (off).
# for auto, _setscroll takes care of that. the binding is needed to
# update the scrollbars when the window is resized.
metawidget proc Scrollcanvas -scrollbar { sScrollbar } {
  my -scrollbar [mkw.complete $sScrollbar {auto on off}]

  switch [my -scrollbar] {
    auto {
      bind $this.canvas <Configure> "$this _setscroll"
      _redraw
    }
    on {
      bind $this.canvas <Configure> {}
      grid $this.scrv -column 1 -row 0 -sticky ns
      grid $this.scrh -column 0 -row 1 -sticky we
      grid $this.box -column 1 -row 1 -sticky nsew
      raise $this.box
    }
    off {
      bind $this.canvas <Configure> {}
      grid forget $this.scrv $this.scrh $this.box
    }
  }
}

# name: -contents
# args: sContents: arbitrary contents
metawidget proc Scrollcanvas -contents { sContents } {
  my -contents $sContents
}

# name: xview
# args: args passed to 'canvas xview'
# wrapper around 'canvas xview'. does a _redraw
metawidget proc Scrollcanvas xview { args } {
  set result [eval $this.canvas xview $args]
  _redraw
  return $result
}

# name: yview
# args: args passed to 'canvas yview'
# wrapper around 'canvas yview'. does a _redraw
metawidget proc Scrollcanvas yview { args } {
  set result [eval $this.canvas yview $args]
  _redraw
  return $result
}

metawidget command Scrollcanvas _setscroll _setscroll
metawidget command Scrollcanvas xview      xview
metawidget command Scrollcanvas yview      yview

metawidget option  Scrollcanvas -scrollbar -scrollbar
metawidget option  Scrollcanvas -contents  -contents

proc test {} {
  global iLine

  set iLine 0

  pack [radiobutton .rad1 -value 0 -text "Scrollbars always on"    -command {.sbox config -scrollbar on}] -anchor w
  pack [radiobutton .rad2 -value 1 -text "Scrollbars always off"   -command {.sbox config -scrollbar off}] -anchor w
  pack [radiobutton .rad3 -value 2 -text "Scrollbars in auto mode" -command {.sbox config -scrollbar auto}] -anchor w

  pack [button .addl -text "Add a line" -command {
    incr iLine
    .sbox insert end "It is [clock format [clock seconds] -format %H:%M:%S]. This is line $iLine.\n"
    .sbox see end
  }] -fill x

  pack [button .reml -text "Remove first line" -command {.sbox delete 1.0 2.0}] -fill x
  pack [scrolltext .sbox -wrap none] -fill both -expand 1
  .sbox config -insertproc "echo insert" -deleteproc "echo delete" -positionproc "echo position"
  proc echo { args } {
      puts $args
      return 1
  }
}

#test
