package require Metawidget

# name: Scrolltext
# args: args
# creates the widgets. the widget option -scrollbar is defined
# by calling its respective member functions.
metawidget create Scrolltext {
  scrollbar $this.scrv -command "$this yview"
  scrollbar $this.scrh -command "$this xview" -orient horiz
  text $this.text -bg white \
      -yscroll "$this.scrv set" -xscroll "$this.scrh set"
  # the resize box
  frame $this.box
  grid columnconf $this.text 1 -weight 1
  grid rowconf    $this.text 0 -weight 1
  grid $this.text -column 1 -row 0 -sticky nsew
  grid columnconf $this 0 -weight 1
  grid rowconf    $this 0 -weight 1
  grid $this.text -column 0 -row 0 -sticky nsew

  bindtags $this.text "$this.text $this Scrolltext . all"
  foreach ev [bind Text] {
      set action [bind Text $ev]
      regsub -all {%W} $action {[winfo parent %W]} action
      bind Scrolltext $ev $action
  }
  # make sure the text widget keeps focus
  bind $this.text <FocusOut> "if \{\[focus] == \"$this\"\} \{focus $this.text\}"
  bind $this <FocusIn> "focus $this.text"

  -contents {}
  -scrollbar off
  -insertproc {}
  -deleteproc {}
  -positionproc {}
} {} -default text

# name: _setscroll
# args: -
# if -scrollbar is auto, displays or hides the two scrollbars.
# uses the fraction values the scrollbars would be set to.
# anything > 0 and < 1 would indicate that the scrollbar should
# be displayed.
metawidget proc Scrolltext _setscroll {} {
  if { [my -scrollbar] != "auto" } return

  my -scrollbar {}
  update idletasks
  set both 1
  mkw.lassign [$this.scrv get] fFrac1 fFrac2
  if { $fFrac1 > 0 || $fFrac2 < 1 } {
    grid $this.scrv -column 1 -row 0 -sticky ns
  } else {
    set both 0
    grid forget $this.scrv
  }

  mkw.lassign [$this.scrh get] fFrac1 fFrac2
  if { $fFrac1 > 0 || $fFrac2 < 1 } {
    grid $this.scrh -column 0 -row 1 -sticky we
  } else {
    set both 0
    grid forget $this.scrh
  }
  if {$both} {
      grid $this.box -column 1 -row 1 -sticky nsew
      raise $this.box
  }
  update idletasks
  my -scrollbar auto
}

# name: _redraw
# args: -
# updates the scrollbars after becoming idle. any
# old job is deleted, so exactly one redraw is done.
metawidget proc Scrolltext _redraw {} {
  catch { cancel [my hJob] }
  my hJob [after idle $this _setscroll]
}

# name: -contents
# args: sContents: an scrolltext buffer's contents
# set the contents name
metawidget proc Scrolltext -contents { sContents } {
  my -contents $sContents
}

# name: -scrollbar
# args: sScrollbar: on, off or auto
# option member. lets the scrollbars either appear (on) or disappear (off).
# for auto, _setscroll takes care of that. the binding is needed to
# update the scrollbars when the window is resized.
metawidget proc Scrolltext -scrollbar { sScrollbar } {
  my -scrollbar [mkw.complete $sScrollbar {auto on off}]

  switch [my -scrollbar] {
    auto {
      bind $this.text <Configure> "$this _setscroll"
      _redraw
    }
    on {
      bind $this.text <Configure> {}
      grid $this.scrv -column 1 -row 0 -sticky ns
      grid $this.scrh -column 0 -row 1 -sticky we
      grid $this.box -column 1 -row 1 -sticky nsew
      raise $this.box
    }
    off {
      bind $this.text <Configure> {}
      grid forget $this.scrv $this.scrh $this.box
    }
  }
}

# name: -insertproc
# args: sInsertProc: proc to execute on insert
metawidget proc Scrolltext -insertproc { sInsertProc } {
  my -insertproc $sInsertProc
}

# name: -deleteproc
# args: sDeleteProc: proc to execute on delete
metawidget proc Scrolltext -deleteproc { sDeleteProc } {
  my -deleteproc $sDeleteProc
}

# name: -positionproc
# args: sPosProc: proc to execute changed insert position
metawidget proc Scrolltext -positionproc { sPosProc } {
  my -positionproc $sPosProc
}

# name: insert
# args: args passed to 'text insert'
# wrapper around 'text insert'. redraws the scrollbars.
metawidget proc Scrolltext insert { args } {
  set start [$this.text index [lindex $args 0]]
  if {![ourinfo exists ininsertproc] && [my -insertproc] != {}} {
      our ininsertproc 1
      if {[eval [my -insertproc] $start [lrange $args 1 end]]} {
          eval $this.text insert $args
      }
      unour ininsertproc
  } else {
      eval $this.text insert $args
  }
  if {[my -positionproc] != {}} {
      eval [my -positionproc] [$this index insert]
  }
  _redraw
}

# name: delete
# args: args passed to 'text delete'
# wrapper around 'text delete'. redraws the scrollbars.
metawidget proc Scrolltext delete { args } {
  set start [$this.text index [lindex $args 0]]
  set end [lindex $args 1]
  if {$end != {}} {
      set end [$this.text index $end]
  } else {
      set end [$this.text index "$start + 1c"]
  }
  if {![ourinfo exists indeleteproc] && [my -deleteproc] != {}} {
      our indeleteproc 1
      if {[eval [my -deleteproc] $start $end]} {
          eval $this.text delete $args
      }
      unour indeleteproc
  } else {
      eval $this.text delete $args
  }
  if {[my -positionproc] != {}} {
      eval [my -positionproc] [$this index insert]
  }
  _redraw
}

# name: mark
# args: args passed to 'text mark'
# wrapper around 'text mark'. checks for set insert
metawidget proc Scrolltext mark { args } {
  set result [eval $this.text mark $args]
  if {[my -positionproc] != {} && \
    [string equal [lindex $args 0] "set"] && \
    [string equal [lindex $args 1] "insert"]} {
      eval [my -positionproc] [$this.text index insert]
  }
  return $result
}

# name: xview
# args: args passed to 'text xview'
# wrapper around 'text xview'. does a _redraw
metawidget proc Scrolltext xview { args } {
  set result [eval $this.text xview $args]
  _redraw
  return $result
}

# name: yview
# args: args passed to 'text yview'
# wrapper around 'text yview'. does a _redraw
metawidget proc Scrolltext yview { args } {
  set result [eval $this.text yview $args]
  _redraw
  return $result
}

# name: focus
# args: none
# set focus to text widget
metawidget proc Scrolltext focus { } {
  ::focus $this.text
  if {[my -positionproc] != {}} {
      eval [my -positionproc] [$this index insert]
  }
}

metawidget command Scrolltext _setscroll _setscroll
metawidget command Scrolltext delete     delete
metawidget command Scrolltext insert     insert
metawidget command Scrolltext mark       mark
metawidget command Scrolltext xview      xview
metawidget command Scrolltext yview      yview
metawidget command Scrolltext focus      focus

metawidget option  Scrolltext -contents     -contents
metawidget option  Scrolltext -scrollbar    -scrollbar
metawidget option  Scrolltext -insertproc   -insertproc
metawidget option  Scrolltext -deleteproc   -deleteproc
metawidget option  Scrolltext -positionproc -positionproc

proc test {} {
  global iLine

  set iLine 0

  pack [radiobutton .rad1 -value 0 -text "Scrollbars always on"    -command {.sbox config -scrollbar on}] -anchor w
  pack [radiobutton .rad2 -value 1 -text "Scrollbars always off"   -command {.sbox config -scrollbar off}] -anchor w
  pack [radiobutton .rad3 -value 2 -text "Scrollbars in auto mode" -command {.sbox config -scrollbar auto}] -anchor w

  pack [button .addl -text "Add a line" -command {
    incr iLine
    .sbox insert end "It is [clock format [clock seconds] -format %H:%M:%S]. This is line $iLine.\n"
    .sbox see end
  }] -fill x

  pack [button .reml -text "Remove first line" -command {.sbox delete 1.0 2.0}] -fill x
  pack [scrolltext .sbox -wrap none] -fill both -expand 1
  .sbox config -insertproc "echo insert" -deleteproc "echo delete" -positionproc "echo position"
  proc echo { args } {
      puts $args
      return 1
  }
}

# test
